/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.ext.asyncsql.impl.pool;

import com.github.jasync.sql.db.Connection;
import com.github.jasync.sql.db.ConnectionPoolConfiguration;
import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.json.JsonObject;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import io.vertx.ext.asyncsql.impl.ConversionUtils;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

public abstract class AsyncConnectionPool {
    public static final int DEFAULT_MAX_POOL_SIZE = 10;
    public static final int DEFAULT_MAX_CONNECTION_RETRIES = 0;
    public static final int DEFAULT_CONNECTION_RETRY_DELAY = 5000;
    public static final int DEFAULT_CONNECTION_RELEASE_DELAY = 0;
    private static final Logger logger = LoggerFactory.getLogger(AsyncConnectionPool.class);
    private final int maxPoolSize;
    private final int maxConnectionRetries;
    private final int connectionRetryDelay;
    private final int connectionReleaseDelay;
    protected final ConnectionPoolConfiguration connectionConfig;
    protected final Vertx vertx;
    private int poolSize = 0;
    private final Deque<Connection> availableConnections = new ArrayDeque<Connection>();
    private final Deque<Handler<AsyncResult<Connection>>> waiters = new ArrayDeque<Handler<AsyncResult<Connection>>>();
    private final Map<Connection, Long> timers = new HashMap<Connection, Long>();

    public AsyncConnectionPool(Vertx vertx, JsonObject globalConfig, ConnectionPoolConfiguration connectionConfig) {
        this.vertx = vertx;
        this.maxPoolSize = globalConfig.getInteger("maxPoolSize", Integer.valueOf(10));
        this.maxConnectionRetries = globalConfig.getInteger("maxConnectionRetries", Integer.valueOf(0));
        this.connectionRetryDelay = globalConfig.getInteger("connectionRetryDelay", Integer.valueOf(5000));
        this.connectionReleaseDelay = globalConfig.getInteger("connectionReleaseDelay", Integer.valueOf(0));
        this.connectionConfig = connectionConfig;
    }

    protected abstract Connection create();

    public synchronized int getPoolSize() {
        return this.poolSize;
    }

    private synchronized void createConnection(final Handler<AsyncResult<Connection>> handler) {
        ++this.poolSize;
        this.createAndConnect(new Handler<AsyncResult<Connection>>(){
            int retries = 0;

            public void handle(AsyncResult<Connection> connectionResult) {
                if (connectionResult.succeeded()) {
                    handler.handle(connectionResult);
                } else if (AsyncConnectionPool.this.maxConnectionRetries < 0 || this.retries < AsyncConnectionPool.this.maxConnectionRetries) {
                    ++this.retries;
                    logger.debug((Object)("Error creating connection. Waiting " + AsyncConnectionPool.this.connectionRetryDelay + " ms for retry " + this.retries + (AsyncConnectionPool.this.maxConnectionRetries >= 0 ? " of " + AsyncConnectionPool.this.maxConnectionRetries : "")));
                    AsyncConnectionPool.this.vertx.setTimer((long)AsyncConnectionPool.this.connectionRetryDelay, timerId -> AsyncConnectionPool.this.createAndConnect((Handler<AsyncResult<Connection>>)this));
                } else {
                    AsyncConnectionPool.this.poolSize = AsyncConnectionPool.this.poolSize - 1;
                    AsyncConnectionPool.this.notifyWaitersAboutAvailableConnection();
                    handler.handle(connectionResult);
                }
            }
        });
    }

    private synchronized void createAndConnect(Handler<AsyncResult<Connection>> handler) {
        try {
            this.create().connect().whenCompleteAsync((connection, error) -> {
                try {
                    if (error != null) {
                        logger.info((Object)"failed to create connection", error);
                        handler.handle((Object)Future.failedFuture((Throwable)error));
                    } else {
                        handler.handle((Object)Future.succeededFuture((Object)connection));
                    }
                }
                catch (Throwable exception) {
                    Handler exceptionHandler = this.vertx.getOrCreateContext().exceptionHandler();
                    if (exceptionHandler != null) {
                        exceptionHandler.handle((Object)exception);
                    }
                    throw exception;
                }
            }, ConversionUtils.vertxToExecutor(this.vertx));
        }
        catch (Throwable e) {
            logger.info((Object)"creating a connection went wrong", e);
            handler.handle((Object)Future.failedFuture((Throwable)e));
        }
    }

    private synchronized void waitForAvailableConnection(Handler<AsyncResult<Connection>> handler) {
        this.waiters.add(handler);
    }

    private synchronized void createOrWaitForAvailableConnection(Handler<AsyncResult<Connection>> handler) {
        if (this.poolSize < this.maxPoolSize) {
            this.createConnection(handler);
        } else {
            this.waitForAvailableConnection(handler);
        }
    }

    public synchronized void take(Handler<AsyncResult<Connection>> handler) {
        Connection connection = this.availableConnections.poll();
        if (connection == null) {
            this.createOrWaitForAvailableConnection(handler);
        } else {
            Long timerId = this.timers.remove(connection);
            if (timerId != null) {
                this.vertx.cancelTimer(timerId.longValue());
            }
            if (connection.isConnected()) {
                if (this.connectionConfig != null && this.connectionConfig.getConnectionTestTimeout() > 0L) {
                    AtomicBoolean testCompleted = new AtomicBoolean(false);
                    long timer = this.vertx.setTimer(this.connectionConfig.getConnectionTestTimeout(), ignored -> {
                        if (testCompleted.compareAndSet(false, true)) {
                            logger.info((Object)"connection test timeout");
                            connection.disconnect();
                            AsyncConnectionPool asyncConnectionPool = this;
                            synchronized (asyncConnectionPool) {
                                --this.poolSize;
                            }
                            this.take(handler);
                        }
                    });
                    connection.sendQuery("SELECT 1 AS alive").whenCompleteAsync((ignored, error) -> {
                        if (error != null) {
                            logger.info((Object)"connection test failed", error);
                            connection.disconnect();
                            AsyncConnectionPool asyncConnectionPool = this;
                            synchronized (asyncConnectionPool) {
                                --this.poolSize;
                            }
                            this.take(handler);
                        } else if (testCompleted.compareAndSet(false, true)) {
                            if (this.connectionConfig.getConnectionTestTimeout() > 0L) {
                                this.vertx.cancelTimer(timer);
                            }
                            handler.handle((Object)Future.succeededFuture((Object)connection));
                        }
                    }, ConversionUtils.vertxToExecutor(this.vertx));
                } else {
                    handler.handle((Object)Future.succeededFuture((Object)connection));
                }
            } else {
                --this.poolSize;
                this.take(handler);
            }
        }
    }

    private synchronized void notifyWaitersAboutAvailableConnection() {
        Handler<AsyncResult<Connection>> handler = this.waiters.poll();
        if (handler != null) {
            this.take(handler);
        }
    }

    public synchronized void expire(Connection connection) {
        connection.disconnect();
        this.availableConnections.remove(connection);
        --this.poolSize;
    }

    public synchronized void giveBack(Connection connection) {
        if (connection.isConnected()) {
            this.availableConnections.add(connection);
            if (this.connectionReleaseDelay > 0) {
                Long timerId = this.vertx.setTimer((long)this.connectionReleaseDelay, res -> this.expire(connection));
                this.timers.put(connection, timerId);
            }
        } else {
            --this.poolSize;
        }
        this.notifyWaitersAboutAvailableConnection();
    }

    public synchronized void close() {
        for (long id : this.timers.values()) {
            this.vertx.cancelTimer(id);
        }
        this.timers.clear();
        this.availableConnections.forEach(Connection::disconnect);
    }

    public synchronized void close(Handler<AsyncResult<Void>> handler) {
        this.close();
        if (handler != null) {
            handler.handle((Object)Future.succeededFuture());
        }
    }
}

