/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.redis.impl;

import io.vertx.core.AsyncResult;
import io.vertx.core.Context;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.impl.VertxInternal;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import io.vertx.core.net.NetClient;
import io.vertx.core.net.NetClientOptions;
import io.vertx.core.net.NetSocket;
import io.vertx.core.streams.WriteStream;
import io.vertx.redis.RedisOptions;
import io.vertx.redis.impl.Command;
import io.vertx.redis.impl.RedisCommand;
import io.vertx.redis.impl.RedisMasterResolver;
import io.vertx.redis.impl.RedisSubscriptions;
import io.vertx.redis.impl.Reply;
import io.vertx.redis.impl.ReplyParser;
import io.vertx.redis.impl.ResponseTransform;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.atomic.AtomicReference;

class RedisConnection {
    private static final Logger log = LoggerFactory.getLogger(RedisConnection.class);
    private final Vertx vertx;
    private final Context context;
    private final Queue<Command<?>> pending = new LinkedList();
    private final Queue<Command<?>> waiting = new LinkedList();
    private final ReplyParser replyParser;
    private final RedisSubscriptions subscriptions;
    private final RedisOptions config;
    private final AtomicReference<State> state = new AtomicReference<State>(State.DISCONNECTED);
    private volatile boolean reconnect = true;
    private volatile NetSocket netSocket;

    public RedisConnection(Vertx vertx, RedisOptions config, RedisSubscriptions subscriptions) {
        Context ctx = Vertx.currentContext();
        if (ctx == null) {
            ctx = vertx.getOrCreateContext();
        } else if (!ctx.isEventLoopContext()) {
            VertxInternal vi = (VertxInternal)vertx;
            ctx = vi.createEventLoopContext(null, null, new JsonObject(), Thread.currentThread().getContextClassLoader());
        }
        this.vertx = vertx;
        this.context = ctx;
        this.config = config;
        this.subscriptions = subscriptions;
        this.replyParser = subscriptions != null ? new ReplyParser((Handler<Reply>)((Handler)reply -> {
            Reply[] data;
            if (reply.is('*') && (data = (Reply[])reply.data()) != null) {
                if (data.length == 3) {
                    if (data[0].is('$') && "message".equals(data[0].asType(String.class))) {
                        String channel = data[1].asType(String.class);
                        subscriptions.handleChannel(channel, data);
                        return;
                    }
                } else if (data.length == 4 && data[0].is('$') && "pmessage".equals(data[0].asType(String.class))) {
                    String pattern = data[1].asType(String.class);
                    subscriptions.handlePattern(pattern, data);
                    return;
                }
            }
            this.handleReply((Reply)reply);
        })) : new ReplyParser((Handler<Reply>)((Handler)this::handleReply));
    }

    private boolean useSentinel() {
        this.reconnect = true;
        return this.config.getSentinels() != null && this.config.getSentinels().size() > 0 && this.config.getMasterName() != null;
    }

    private void connect(String host, int port, boolean checkMaster) {
        this.replyParser.reset();
        NetClient client = this.vertx.createNetClient((NetClientOptions)this.config);
        client.connect(port, host, asyncResult -> {
            if (asyncResult.failed()) {
                if (this.state.compareAndSet(State.CONNECTING, State.ERROR)) {
                    this.clearQueue(this.waiting, asyncResult.cause());
                    this.clearQueue(this.pending, asyncResult.cause());
                    this.state.set(State.DISCONNECTED);
                    if (this.reconnect) {
                        this.vertx.setTimer(this.config.getReconnectInterval(), v0 -> this.connect());
                    }
                }
            } else {
                this.netSocket = ((NetSocket)asyncResult.result()).handler((Handler)this.replyParser).closeHandler(v2 -> {
                    this.state.set(State.ERROR);
                    this.clearQueue(this.waiting, "Connection closed");
                    this.clearQueue(this.pending, "Connection closed");
                    this.state.set(State.DISCONNECTED);
                    client.close();
                    if (this.reconnect) {
                        this.vertx.setTimer(this.config.getReconnectInterval(), v0 -> this.connect());
                    }
                }).exceptionHandler(e -> this.netSocket.close());
                this.clearQueue(this.waiting, "Connection lost");
                this.doAuth();
                if (checkMaster) {
                    this.doCheckMaster();
                }
            }
        });
    }

    private void connect() {
        if (this.state.compareAndSet(State.DISCONNECTED, State.CONNECTING)) {
            this.runOnContext((Handler<Void>)((Handler)v -> {
                if (this.useSentinel()) {
                    RedisMasterResolver resolver = new RedisMasterResolver(this.context.owner(), this.config);
                    resolver.getMasterAddressByName((Handler<AsyncResult<JsonObject>>)((Handler)jsonObjectAsyncResult -> {
                        if (jsonObjectAsyncResult.succeeded()) {
                            JsonObject masterAddress = (JsonObject)jsonObjectAsyncResult.result();
                            this.connect(masterAddress.getString("host"), masterAddress.getInteger("port"), true);
                        } else {
                            this.clearQueue(this.waiting, jsonObjectAsyncResult.cause());
                            this.clearQueue(this.pending, jsonObjectAsyncResult.cause());
                            this.state.set(State.DISCONNECTED);
                        }
                        resolver.close();
                    }));
                } else {
                    this.connect(this.config.getHost(), this.config.getPort(), false);
                }
            }));
        }
    }

    void disconnect(Handler<AsyncResult<Void>> closeHandler) {
        this.reconnect = false;
        switch (this.state.get()) {
            case CONNECTING: 
            case CONNECTED: {
                Command<Void> cmd = new Command<Void>(this.context, RedisCommand.QUIT, null, Charset.defaultCharset(), ResponseTransform.NONE, Void.class);
                cmd.handler((Handler<AsyncResult<Void>>)((Handler)v -> {
                    if (this.state.compareAndSet(State.CONNECTED, State.ERROR)) {
                        this.clearQueue(this.waiting, "Connection closed");
                        this.clearQueue(this.pending, "Connection closed");
                        this.netSocket.close();
                        closeHandler.handle((Object)Future.succeededFuture());
                    }
                }));
                this.send(cmd);
                break;
            }
            case ERROR: 
            case DISCONNECTED: {
                closeHandler.handle((Object)Future.succeededFuture());
            }
        }
    }

    void send(Command<?> command) {
        if (this.state.get() == State.DISCONNECTED) {
            this.connect();
        }
        this.runOnContext((Handler<Void>)((Handler)v -> {
            switch (this.state.get()) {
                case CONNECTED: {
                    this.write(command);
                    break;
                }
                case CONNECTING: 
                case ERROR: 
                case DISCONNECTED: {
                    this.pending.add(command);
                }
            }
        }));
    }

    private void write(Command<?> command) {
        for (int i = 0; i < command.getExpectedReplies(); ++i) {
            this.waiting.add(command);
        }
        command.writeTo((WriteStream<Buffer>)this.netSocket);
    }

    private void doAuth() {
        if (this.config.getAuth() != null) {
            ArrayList<String> args = new ArrayList<String>();
            args.add(this.config.getAuth());
            Command<String> authCmd = new Command<String>(this.context, RedisCommand.AUTH, args, Charset.forName(this.config.getEncoding()), ResponseTransform.NONE, String.class).handler((Handler<AsyncResult<String>>)((Handler)auth -> {
                if (auth.failed()) {
                    this.clearQueue(this.pending, auth.cause());
                    this.netSocket.close();
                } else {
                    this.doSelect();
                }
            }));
            this.write(authCmd);
        } else {
            this.doSelect();
        }
    }

    private void doCheckMaster() {
        Command<JsonObject> infoCommand = new Command<JsonObject>(this.context, RedisCommand.INFO, Collections.emptyList(), Charset.forName(this.config.getEncoding()), ResponseTransform.INFO, JsonObject.class).handler((Handler<AsyncResult<JsonObject>>)((Handler)info -> {
            if (info.failed()) {
                this.clearQueue(this.pending, info.cause());
                this.netSocket.close();
            } else if (!"master".equals(((JsonObject)info.result()).getJsonObject("replication").getString("role"))) {
                this.clearQueue(this.pending, info.cause());
                this.netSocket.close();
                log.error((Object)"Forced disconnect of non-master");
            } else {
                log.info((Object)"Verification of master role succeeded");
            }
        }));
        this.write(infoCommand);
    }

    private void doSelect() {
        if (this.config.getSelect() != null) {
            ArrayList<Integer> args = new ArrayList<Integer>();
            args.add(this.config.getSelect());
            Command<String> selectCmd = new Command<String>(this.context, RedisCommand.SELECT, args, Charset.forName(this.config.getEncoding()), ResponseTransform.NONE, String.class).handler((Handler<AsyncResult<String>>)((Handler)select -> {
                if (select.failed()) {
                    this.clearQueue(this.pending, select.cause());
                    this.netSocket.close();
                } else {
                    this.restoreState();
                }
            }));
            this.write(selectCmd);
        } else {
            this.restoreState();
        }
    }

    private void restoreState() {
        if (this.state.compareAndSet(State.CONNECTING, State.CONNECTED)) {
            Command<?> command;
            while ((command = this.pending.poll()) != null) {
                this.write(command);
            }
            if (this.subscriptions != null) {
                ArrayList<String> args;
                for (String channel : this.subscriptions.channelNames()) {
                    args = new ArrayList<String>();
                    args.add(channel);
                    this.write(new Command<JsonArray>(this.context, RedisCommand.SUBSCRIBE, args, Charset.forName(this.config.getEncoding()), ResponseTransform.NONE, JsonArray.class));
                }
                for (String pattern : this.subscriptions.patternNames()) {
                    args = new ArrayList();
                    args.add(pattern);
                    this.write(new Command<JsonArray>(this.context, RedisCommand.PSUBSCRIBE, args, Charset.forName(this.config.getEncoding()), ResponseTransform.NONE, JsonArray.class));
                }
            }
        }
    }

    private void handleReply(Reply reply) {
        Command<?> cmd = this.waiting.poll();
        if (cmd != null) {
            switch (reply.type()) {
                case 45: {
                    cmd.handle((AsyncResult<?>)Future.failedFuture((String)reply.asType(String.class)));
                    return;
                }
                case 43: {
                    switch (cmd.responseTransform()) {
                        case ARRAY: {
                            cmd.handle((AsyncResult<?>)Future.succeededFuture((Object)new JsonArray().add(reply.asType(String.class))));
                            break;
                        }
                        default: {
                            cmd.handle((AsyncResult<?>)Future.succeededFuture(reply.asType(cmd.returnType())));
                        }
                    }
                    return;
                }
                case 36: {
                    switch (cmd.responseTransform()) {
                        case ARRAY: {
                            cmd.handle((AsyncResult<?>)Future.succeededFuture((Object)new JsonArray().add(reply.asType(String.class, cmd.encoding()))));
                            break;
                        }
                        case INFO: {
                            String info = reply.asType(String.class, cmd.encoding());
                            if (info == null) {
                                cmd.handle((AsyncResult<?>)Future.succeededFuture(null));
                                break;
                            }
                            String[] lines = info.split("\\r?\\n");
                            JsonObject value = new JsonObject();
                            JsonObject section = null;
                            for (String line : lines) {
                                if (line.length() == 0) {
                                    section = null;
                                    continue;
                                }
                                if (line.charAt(0) == '#') {
                                    section = new JsonObject();
                                    value.put(line.substring(2).toLowerCase(), section);
                                    continue;
                                }
                                int split = line.indexOf(58);
                                if (section == null) {
                                    value.put(line.substring(0, split), line.substring(split + 1));
                                    continue;
                                }
                                section.put(line.substring(0, split), line.substring(split + 1));
                            }
                            cmd.handle((AsyncResult<?>)Future.succeededFuture((Object)value));
                            break;
                        }
                        default: {
                            cmd.handle((AsyncResult<?>)Future.succeededFuture(reply.asType(cmd.returnType(), cmd.encoding())));
                        }
                    }
                    return;
                }
                case 42: {
                    switch (cmd.responseTransform()) {
                        case HASH: {
                            cmd.handle((AsyncResult<?>)Future.succeededFuture((Object)reply.asType(JsonObject.class, cmd.encoding())));
                            break;
                        }
                        default: {
                            cmd.handle((AsyncResult<?>)Future.succeededFuture((Object)reply.asType(JsonArray.class, cmd.encoding())));
                        }
                    }
                    return;
                }
                case 58: {
                    switch (cmd.responseTransform()) {
                        case ARRAY: {
                            cmd.handle((AsyncResult<?>)Future.succeededFuture((Object)new JsonArray().add(reply.asType(Long.class))));
                            break;
                        }
                        default: {
                            cmd.handle((AsyncResult<?>)Future.succeededFuture(reply.asType(cmd.returnType())));
                        }
                    }
                    return;
                }
            }
            cmd.handle((AsyncResult<?>)Future.failedFuture((String)"Unknown message type"));
        } else {
            log.error((Object)("No handler waiting for message: " + reply.asType(String.class)));
        }
    }

    private void runOnContext(Handler<Void> handler) {
        if (Vertx.currentContext() == this.context && Context.isOnEventLoopThread()) {
            handler.handle(null);
        } else {
            this.context.runOnContext(handler);
        }
    }

    private void clearQueue(Queue<Command<?>> q, String message) {
        Command<?> cmd;
        while ((cmd = q.poll()) != null) {
            cmd.handle((AsyncResult<?>)Future.failedFuture((String)message));
        }
    }

    private void clearQueue(Queue<Command<?>> q, Throwable cause) {
        Command<?> cmd;
        while ((cmd = q.poll()) != null) {
            cmd.handle((AsyncResult<?>)Future.failedFuture((Throwable)cause));
        }
    }

    private static enum State {
        DISCONNECTED,
        CONNECTING,
        CONNECTED,
        ERROR;

    }
}

