/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.rxjava.redis;

import java.util.Map;
import rx.Observable;
import rx.Single;
import io.vertx.redis.op.BitOperation;
import io.vertx.redis.op.ResetOptions;
import io.vertx.redis.op.ObjectCmd;
import io.vertx.redis.op.KillFilter;
import io.vertx.redis.op.ClientReplyOptions;
import java.util.Map;
import io.vertx.redis.op.GeoUnit;
import io.vertx.core.json.JsonObject;
import io.vertx.core.AsyncResult;
import io.vertx.redis.op.RangeOptions;
import io.vertx.redis.op.GeoMember;
import io.vertx.redis.op.GeoRadiusOptions;
import io.vertx.redis.op.InsertOptions;
import io.vertx.redis.op.AggregateOptions;
import io.vertx.redis.op.SetOptions;
import io.vertx.redis.op.SortOptions;
import io.vertx.redis.op.MigrateOptions;
import io.vertx.redis.op.ScanOptions;
import io.vertx.redis.op.BitFieldOptions;
import io.vertx.redis.op.ScriptDebugOptions;
import io.vertx.rxjava.core.Vertx;
import io.vertx.redis.op.FailoverOptions;
import io.vertx.redis.op.SlotCmd;
import io.vertx.redis.op.BitFieldOverflowOptions;
import io.vertx.redis.op.RangeLimitOptions;
import io.vertx.redis.op.LimitOptions;
import io.vertx.core.json.JsonArray;
import java.util.List;
import io.vertx.rxjava.core.buffer.Buffer;
import io.vertx.redis.RedisOptions;
import io.vertx.core.Handler;


@io.vertx.lang.rxjava.RxGen(io.vertx.redis.RedisClient.class)
public class RedisClient {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    RedisClient that = (RedisClient) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.rxjava.TypeArg<RedisClient> __TYPE_ARG = new io.vertx.lang.rxjava.TypeArg<>(
    obj -> new RedisClient((io.vertx.redis.RedisClient) obj),
    RedisClient::getDelegate
  );

  private final io.vertx.redis.RedisClient delegate;
  
  public RedisClient(io.vertx.redis.RedisClient delegate) {
    this.delegate = delegate;
  }

  public io.vertx.redis.RedisClient getDelegate() {
    return delegate;
  }

  public static RedisClient create(Vertx vertx) { 
    RedisClient ret = RedisClient.newInstance(io.vertx.redis.RedisClient.create(vertx.getDelegate()));
    return ret;
  }

  public static RedisClient create(Vertx vertx, RedisOptions config) { 
    RedisClient ret = RedisClient.newInstance(io.vertx.redis.RedisClient.create(vertx.getDelegate(), config));
    return ret;
  }

  /**
   * Close the client - when it is fully closed the handler will be called.
   * @param handler 
   */
  public void close(Handler<AsyncResult<Void>> handler) { 
    delegate.close(handler);
  }

  /**
   * Close the client - when it is fully closed the handler will be called.
   * @return 
   * @deprecated use {@link #rxClose} instead
   */
  @Deprecated()
  public Observable<Void> closeObservable() { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    close(handler.toHandler());
    return handler;
  }

  /**
   * Close the client - when it is fully closed the handler will be called.
   * @return 
   */
  public Single<Void> rxClose() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      close(fut);
    }));
  }

  /**
   * Append a value to a key
   * @param key Key string
   * @param value Value to append
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient append(String key, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.append(key, value, handler);
    return this;
  }

  /**
   * Append a value to a key
   * @param key Key string
   * @param value Value to append
   * @return 
   * @deprecated use {@link #rxAppend} instead
   */
  @Deprecated()
  public Observable<Long> appendObservable(String key, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    append(key, value, handler.toHandler());
    return handler;
  }

  /**
   * Append a value to a key
   * @param key Key string
   * @param value Value to append
   * @return 
   */
  public Single<Long> rxAppend(String key, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      append(key, value, fut);
    }));
  }

  /**
   * Authenticate to the server
   * @param password Password for authentication
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient auth(String password, Handler<AsyncResult<String>> handler) { 
    delegate.auth(password, handler);
    return this;
  }

  /**
   * Authenticate to the server
   * @param password Password for authentication
   * @return 
   * @deprecated use {@link #rxAuth} instead
   */
  @Deprecated()
  public Observable<String> authObservable(String password) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    auth(password, handler.toHandler());
    return handler;
  }

  /**
   * Authenticate to the server
   * @param password Password for authentication
   * @return 
   */
  public Single<String> rxAuth(String password) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      auth(password, fut);
    }));
  }

  /**
   * Asynchronously rewrite the append-only file
   * @param handler 
   * @return 
   */
  public RedisClient bgrewriteaof(Handler<AsyncResult<String>> handler) { 
    delegate.bgrewriteaof(handler);
    return this;
  }

  /**
   * Asynchronously rewrite the append-only file
   * @return 
   * @deprecated use {@link #rxBgrewriteaof} instead
   */
  @Deprecated()
  public Observable<String> bgrewriteaofObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    bgrewriteaof(handler.toHandler());
    return handler;
  }

  /**
   * Asynchronously rewrite the append-only file
   * @return 
   */
  public Single<String> rxBgrewriteaof() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      bgrewriteaof(fut);
    }));
  }

  /**
   * Asynchronously save the dataset to disk
   * @param handler 
   * @return 
   */
  public RedisClient bgsave(Handler<AsyncResult<String>> handler) { 
    delegate.bgsave(handler);
    return this;
  }

  /**
   * Asynchronously save the dataset to disk
   * @return 
   * @deprecated use {@link #rxBgsave} instead
   */
  @Deprecated()
  public Observable<String> bgsaveObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    bgsave(handler.toHandler());
    return handler;
  }

  /**
   * Asynchronously save the dataset to disk
   * @return 
   */
  public Single<String> rxBgsave() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      bgsave(fut);
    }));
  }

  /**
   * Count set bits in a string
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient bitcount(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.bitcount(key, handler);
    return this;
  }

  /**
   * Count set bits in a string
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxBitcount} instead
   */
  @Deprecated()
  public Observable<Long> bitcountObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    bitcount(key, handler.toHandler());
    return handler;
  }

  /**
   * Count set bits in a string
   * @param key Key string
   * @return 
   */
  public Single<Long> rxBitcount(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      bitcount(key, fut);
    }));
  }

  /**
   * Count set bits in a string
   * @param key Key string
   * @param start Start index
   * @param end End index
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient bitcountRange(String key, long start, long end, Handler<AsyncResult<Long>> handler) { 
    delegate.bitcountRange(key, start, end, handler);
    return this;
  }

  /**
   * Count set bits in a string
   * @param key Key string
   * @param start Start index
   * @param end End index
   * @return 
   * @deprecated use {@link #rxBitcountRange} instead
   */
  @Deprecated()
  public Observable<Long> bitcountRangeObservable(String key, long start, long end) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    bitcountRange(key, start, end, handler.toHandler());
    return handler;
  }

  /**
   * Count set bits in a string
   * @param key Key string
   * @param start Start index
   * @param end End index
   * @return 
   */
  public Single<Long> rxBitcountRange(String key, long start, long end) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      bitcountRange(key, start, end, fut);
    }));
  }

  /**
   * Perform bitwise operations between strings
   * @param operation Bitwise operation to perform
   * @param destkey Destination key where result is stored
   * @param keys List of keys on which to perform the operation
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient bitop(BitOperation operation, String destkey, List<String> keys, Handler<AsyncResult<Long>> handler) { 
    delegate.bitop(operation, destkey, keys, handler);
    return this;
  }

  /**
   * Perform bitwise operations between strings
   * @param operation Bitwise operation to perform
   * @param destkey Destination key where result is stored
   * @param keys List of keys on which to perform the operation
   * @return 
   * @deprecated use {@link #rxBitop} instead
   */
  @Deprecated()
  public Observable<Long> bitopObservable(BitOperation operation, String destkey, List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    bitop(operation, destkey, keys, handler.toHandler());
    return handler;
  }

  /**
   * Perform bitwise operations between strings
   * @param operation Bitwise operation to perform
   * @param destkey Destination key where result is stored
   * @param keys List of keys on which to perform the operation
   * @return 
   */
  public Single<Long> rxBitop(BitOperation operation, String destkey, List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      bitop(operation, destkey, keys, fut);
    }));
  }

  /**
   * Find first bit set or clear in a string
   * @param key Key string
   * @param bit What bit value to look for - must be 1, or 0
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient bitpos(String key, int bit, Handler<AsyncResult<Long>> handler) { 
    delegate.bitpos(key, bit, handler);
    return this;
  }

  /**
   * Find first bit set or clear in a string
   * @param key Key string
   * @param bit What bit value to look for - must be 1, or 0
   * @return 
   * @deprecated use {@link #rxBitpos} instead
   */
  @Deprecated()
  public Observable<Long> bitposObservable(String key, int bit) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    bitpos(key, bit, handler.toHandler());
    return handler;
  }

  /**
   * Find first bit set or clear in a string
   * @param key Key string
   * @param bit What bit value to look for - must be 1, or 0
   * @return 
   */
  public Single<Long> rxBitpos(String key, int bit) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      bitpos(key, bit, fut);
    }));
  }

  /**
   * Find first bit set or clear in a string
   *
   * See also bitposRange() method, which takes start, and stop offset.
   * @param key Key string
   * @param bit What bit value to look for - must be 1, or 0
   * @param start Start offset
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient bitposFrom(String key, int bit, int start, Handler<AsyncResult<Long>> handler) { 
    delegate.bitposFrom(key, bit, start, handler);
    return this;
  }

  /**
   * Find first bit set or clear in a string
   *
   * See also bitposRange() method, which takes start, and stop offset.
   * @param key Key string
   * @param bit What bit value to look for - must be 1, or 0
   * @param start Start offset
   * @return 
   * @deprecated use {@link #rxBitposFrom} instead
   */
  @Deprecated()
  public Observable<Long> bitposFromObservable(String key, int bit, int start) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    bitposFrom(key, bit, start, handler.toHandler());
    return handler;
  }

  /**
   * Find first bit set or clear in a string
   *
   * See also bitposRange() method, which takes start, and stop offset.
   * @param key Key string
   * @param bit What bit value to look for - must be 1, or 0
   * @param start Start offset
   * @return 
   */
  public Single<Long> rxBitposFrom(String key, int bit, int start) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      bitposFrom(key, bit, start, fut);
    }));
  }

  /**
   * Find first bit set or clear in a string
   *
   * Note: when both start, and stop offsets are specified,
   * behaviour is slightly different than if only start is specified
   * @param key Key string
   * @param bit What bit value to look for - must be 1, or 0
   * @param start Start offset
   * @param stop End offset - inclusive
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient bitposRange(String key, int bit, int start, int stop, Handler<AsyncResult<Long>> handler) { 
    delegate.bitposRange(key, bit, start, stop, handler);
    return this;
  }

  /**
   * Find first bit set or clear in a string
   *
   * Note: when both start, and stop offsets are specified,
   * behaviour is slightly different than if only start is specified
   * @param key Key string
   * @param bit What bit value to look for - must be 1, or 0
   * @param start Start offset
   * @param stop End offset - inclusive
   * @return 
   * @deprecated use {@link #rxBitposRange} instead
   */
  @Deprecated()
  public Observable<Long> bitposRangeObservable(String key, int bit, int start, int stop) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    bitposRange(key, bit, start, stop, handler.toHandler());
    return handler;
  }

  /**
   * Find first bit set or clear in a string
   *
   * Note: when both start, and stop offsets are specified,
   * behaviour is slightly different than if only start is specified
   * @param key Key string
   * @param bit What bit value to look for - must be 1, or 0
   * @param start Start offset
   * @param stop End offset - inclusive
   * @return 
   */
  public Single<Long> rxBitposRange(String key, int bit, int start, int stop) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      bitposRange(key, bit, start, stop, fut);
    }));
  }

  /**
   * Remove and get the first element in a list, or block until one is available
   * @param key Key string identifying a list to watch
   * @param seconds Timeout in seconds
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient blpop(String key, int seconds, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.blpop(key, seconds, handler);
    return this;
  }

  /**
   * Remove and get the first element in a list, or block until one is available
   * @param key Key string identifying a list to watch
   * @param seconds Timeout in seconds
   * @return 
   * @deprecated use {@link #rxBlpop} instead
   */
  @Deprecated()
  public Observable<JsonArray> blpopObservable(String key, int seconds) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    blpop(key, seconds, handler.toHandler());
    return handler;
  }

  /**
   * Remove and get the first element in a list, or block until one is available
   * @param key Key string identifying a list to watch
   * @param seconds Timeout in seconds
   * @return 
   */
  public Single<JsonArray> rxBlpop(String key, int seconds) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      blpop(key, seconds, fut);
    }));
  }

  /**
   * Remove and get the first element in any of the lists, or block until one is available
   * @param keys List of key strings identifying lists to watch
   * @param seconds Timeout in seconds
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient blpopMany(List<String> keys, int seconds, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.blpopMany(keys, seconds, handler);
    return this;
  }

  /**
   * Remove and get the first element in any of the lists, or block until one is available
   * @param keys List of key strings identifying lists to watch
   * @param seconds Timeout in seconds
   * @return 
   * @deprecated use {@link #rxBlpopMany} instead
   */
  @Deprecated()
  public Observable<JsonArray> blpopManyObservable(List<String> keys, int seconds) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    blpopMany(keys, seconds, handler.toHandler());
    return handler;
  }

  /**
   * Remove and get the first element in any of the lists, or block until one is available
   * @param keys List of key strings identifying lists to watch
   * @param seconds Timeout in seconds
   * @return 
   */
  public Single<JsonArray> rxBlpopMany(List<String> keys, int seconds) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      blpopMany(keys, seconds, fut);
    }));
  }

  /**
   * Remove and get the last element in a list, or block until one is available
   * @param key Key string identifying a list to watch
   * @param seconds Timeout in seconds
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient brpop(String key, int seconds, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.brpop(key, seconds, handler);
    return this;
  }

  /**
   * Remove and get the last element in a list, or block until one is available
   * @param key Key string identifying a list to watch
   * @param seconds Timeout in seconds
   * @return 
   * @deprecated use {@link #rxBrpop} instead
   */
  @Deprecated()
  public Observable<JsonArray> brpopObservable(String key, int seconds) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    brpop(key, seconds, handler.toHandler());
    return handler;
  }

  /**
   * Remove and get the last element in a list, or block until one is available
   * @param key Key string identifying a list to watch
   * @param seconds Timeout in seconds
   * @return 
   */
  public Single<JsonArray> rxBrpop(String key, int seconds) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      brpop(key, seconds, fut);
    }));
  }

  /**
   * Remove and get the last element in any of the lists, or block until one is available
   * @param keys List of key strings identifying lists to watch
   * @param seconds Timeout in seconds
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient brpopMany(List<String> keys, int seconds, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.brpopMany(keys, seconds, handler);
    return this;
  }

  /**
   * Remove and get the last element in any of the lists, or block until one is available
   * @param keys List of key strings identifying lists to watch
   * @param seconds Timeout in seconds
   * @return 
   * @deprecated use {@link #rxBrpopMany} instead
   */
  @Deprecated()
  public Observable<JsonArray> brpopManyObservable(List<String> keys, int seconds) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    brpopMany(keys, seconds, handler.toHandler());
    return handler;
  }

  /**
   * Remove and get the last element in any of the lists, or block until one is available
   * @param keys List of key strings identifying lists to watch
   * @param seconds Timeout in seconds
   * @return 
   */
  public Single<JsonArray> rxBrpopMany(List<String> keys, int seconds) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      brpopMany(keys, seconds, fut);
    }));
  }

  /**
   * Pop a value from a list, push it to another list and return it; or block until one is available
   * @param key Key string identifying the source list
   * @param destkey Key string identifying the destination list
   * @param seconds Timeout in seconds
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient brpoplpush(String key, String destkey, int seconds, Handler<AsyncResult<String>> handler) { 
    delegate.brpoplpush(key, destkey, seconds, handler);
    return this;
  }

  /**
   * Pop a value from a list, push it to another list and return it; or block until one is available
   * @param key Key string identifying the source list
   * @param destkey Key string identifying the destination list
   * @param seconds Timeout in seconds
   * @return 
   * @deprecated use {@link #rxBrpoplpush} instead
   */
  @Deprecated()
  public Observable<String> brpoplpushObservable(String key, String destkey, int seconds) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    brpoplpush(key, destkey, seconds, handler.toHandler());
    return handler;
  }

  /**
   * Pop a value from a list, push it to another list and return it; or block until one is available
   * @param key Key string identifying the source list
   * @param destkey Key string identifying the destination list
   * @param seconds Timeout in seconds
   * @return 
   */
  public Single<String> rxBrpoplpush(String key, String destkey, int seconds) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      brpoplpush(key, destkey, seconds, fut);
    }));
  }

  /**
   * Kill the connection of a client
   * @param filter Filter options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clientKill(KillFilter filter, Handler<AsyncResult<Long>> handler) { 
    delegate.clientKill(filter, handler);
    return this;
  }

  /**
   * Kill the connection of a client
   * @param filter Filter options
   * @return 
   * @deprecated use {@link #rxClientKill} instead
   */
  @Deprecated()
  public Observable<Long> clientKillObservable(KillFilter filter) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clientKill(filter, handler.toHandler());
    return handler;
  }

  /**
   * Kill the connection of a client
   * @param filter Filter options
   * @return 
   */
  public Single<Long> rxClientKill(KillFilter filter) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clientKill(filter, fut);
    }));
  }

  /**
   * Get the list of client connections
   * @param handler 
   * @return 
   */
  public RedisClient clientList(Handler<AsyncResult<String>> handler) { 
    delegate.clientList(handler);
    return this;
  }

  /**
   * Get the list of client connections
   * @return 
   * @deprecated use {@link #rxClientList} instead
   */
  @Deprecated()
  public Observable<String> clientListObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clientList(handler.toHandler());
    return handler;
  }

  /**
   * Get the list of client connections
   * @return 
   */
  public Single<String> rxClientList() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clientList(fut);
    }));
  }

  /**
   * Get the current connection name
   * @param handler 
   * @return 
   */
  public RedisClient clientGetname(Handler<AsyncResult<String>> handler) { 
    delegate.clientGetname(handler);
    return this;
  }

  /**
   * Get the current connection name
   * @return 
   * @deprecated use {@link #rxClientGetname} instead
   */
  @Deprecated()
  public Observable<String> clientGetnameObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clientGetname(handler.toHandler());
    return handler;
  }

  /**
   * Get the current connection name
   * @return 
   */
  public Single<String> rxClientGetname() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clientGetname(fut);
    }));
  }

  /**
   * Stop processing commands from clients for some time
   * @param millis Pause time in milliseconds
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clientPause(long millis, Handler<AsyncResult<String>> handler) { 
    delegate.clientPause(millis, handler);
    return this;
  }

  /**
   * Stop processing commands from clients for some time
   * @param millis Pause time in milliseconds
   * @return 
   * @deprecated use {@link #rxClientPause} instead
   */
  @Deprecated()
  public Observable<String> clientPauseObservable(long millis) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clientPause(millis, handler.toHandler());
    return handler;
  }

  /**
   * Stop processing commands from clients for some time
   * @param millis Pause time in milliseconds
   * @return 
   */
  public Single<String> rxClientPause(long millis) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clientPause(millis, fut);
    }));
  }

  /**
   * Set the current connection name
   * @param name New name for current connection
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clientSetname(String name, Handler<AsyncResult<String>> handler) { 
    delegate.clientSetname(name, handler);
    return this;
  }

  /**
   * Set the current connection name
   * @param name New name for current connection
   * @return 
   * @deprecated use {@link #rxClientSetname} instead
   */
  @Deprecated()
  public Observable<String> clientSetnameObservable(String name) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clientSetname(name, handler.toHandler());
    return handler;
  }

  /**
   * Set the current connection name
   * @param name New name for current connection
   * @return 
   */
  public Single<String> rxClientSetname(String name) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clientSetname(name, fut);
    }));
  }

  /**
   * Assign new hash slots to receiving node.
   * @param slots 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterAddslots(List<Long> slots, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterAddslots(slots, handler);
    return this;
  }

  /**
   * Assign new hash slots to receiving node.
   * @param slots 
   * @return 
   * @deprecated use {@link #rxClusterAddslots} instead
   */
  @Deprecated()
  public Observable<Void> clusterAddslotsObservable(List<Long> slots) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterAddslots(slots, handler.toHandler());
    return handler;
  }

  /**
   * Assign new hash slots to receiving node.
   * @param slots 
   * @return 
   */
  public Single<Void> rxClusterAddslots(List<Long> slots) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterAddslots(slots, fut);
    }));
  }

  /**
   * Return the number of failure reports active for a given node.
   * @param nodeId 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterCountFailureReports(String nodeId, Handler<AsyncResult<Long>> handler) { 
    delegate.clusterCountFailureReports(nodeId, handler);
    return this;
  }

  /**
   * Return the number of failure reports active for a given node.
   * @param nodeId 
   * @return 
   * @deprecated use {@link #rxClusterCountFailureReports} instead
   */
  @Deprecated()
  public Observable<Long> clusterCountFailureReportsObservable(String nodeId) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterCountFailureReports(nodeId, handler.toHandler());
    return handler;
  }

  /**
   * Return the number of failure reports active for a given node.
   * @param nodeId 
   * @return 
   */
  public Single<Long> rxClusterCountFailureReports(String nodeId) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterCountFailureReports(nodeId, fut);
    }));
  }

  /**
   * Return the number of local keys in the specified hash slot.
   * @param slot 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterCountkeysinslot(long slot, Handler<AsyncResult<Long>> handler) { 
    delegate.clusterCountkeysinslot(slot, handler);
    return this;
  }

  /**
   * Return the number of local keys in the specified hash slot.
   * @param slot 
   * @return 
   * @deprecated use {@link #rxClusterCountkeysinslot} instead
   */
  @Deprecated()
  public Observable<Long> clusterCountkeysinslotObservable(long slot) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterCountkeysinslot(slot, handler.toHandler());
    return handler;
  }

  /**
   * Return the number of local keys in the specified hash slot.
   * @param slot 
   * @return 
   */
  public Single<Long> rxClusterCountkeysinslot(long slot) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterCountkeysinslot(slot, fut);
    }));
  }

  /**
   * Set hash slots as unbound in receiving node.
   * @param slot 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterDelslots(long slot, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterDelslots(slot, handler);
    return this;
  }

  /**
   * Set hash slots as unbound in receiving node.
   * @param slot 
   * @return 
   * @deprecated use {@link #rxClusterDelslots} instead
   */
  @Deprecated()
  public Observable<Void> clusterDelslotsObservable(long slot) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterDelslots(slot, handler.toHandler());
    return handler;
  }

  /**
   * Set hash slots as unbound in receiving node.
   * @param slot 
   * @return 
   */
  public Single<Void> rxClusterDelslots(long slot) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterDelslots(slot, fut);
    }));
  }

  /**
   * Set hash slots as unbound in receiving node.
   * @param slots 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterDelslotsMany(List<Long> slots, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterDelslotsMany(slots, handler);
    return this;
  }

  /**
   * Set hash slots as unbound in receiving node.
   * @param slots 
   * @return 
   * @deprecated use {@link #rxClusterDelslotsMany} instead
   */
  @Deprecated()
  public Observable<Void> clusterDelslotsManyObservable(List<Long> slots) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterDelslotsMany(slots, handler.toHandler());
    return handler;
  }

  /**
   * Set hash slots as unbound in receiving node.
   * @param slots 
   * @return 
   */
  public Single<Void> rxClusterDelslotsMany(List<Long> slots) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterDelslotsMany(slots, fut);
    }));
  }

  /**
   * Forces a slave to perform a manual failover of its master.
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterFailover(Handler<AsyncResult<Void>> handler) { 
    delegate.clusterFailover(handler);
    return this;
  }

  /**
   * Forces a slave to perform a manual failover of its master.
   * @return 
   * @deprecated use {@link #rxClusterFailover} instead
   */
  @Deprecated()
  public Observable<Void> clusterFailoverObservable() { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterFailover(handler.toHandler());
    return handler;
  }

  /**
   * Forces a slave to perform a manual failover of its master.
   * @return 
   */
  public Single<Void> rxClusterFailover() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterFailover(fut);
    }));
  }

  /**
   * Forces a slave to perform a manual failover of its master.
   * @param options 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterFailOverWithOptions(FailoverOptions options, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterFailOverWithOptions(options, handler);
    return this;
  }

  /**
   * Forces a slave to perform a manual failover of its master.
   * @param options 
   * @return 
   * @deprecated use {@link #rxClusterFailOverWithOptions} instead
   */
  @Deprecated()
  public Observable<Void> clusterFailOverWithOptionsObservable(FailoverOptions options) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterFailOverWithOptions(options, handler.toHandler());
    return handler;
  }

  /**
   * Forces a slave to perform a manual failover of its master.
   * @param options 
   * @return 
   */
  public Single<Void> rxClusterFailOverWithOptions(FailoverOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterFailOverWithOptions(options, fut);
    }));
  }

  /**
   * Remove a node from the nodes table.
   * @param nodeId 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterForget(String nodeId, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterForget(nodeId, handler);
    return this;
  }

  /**
   * Remove a node from the nodes table.
   * @param nodeId 
   * @return 
   * @deprecated use {@link #rxClusterForget} instead
   */
  @Deprecated()
  public Observable<Void> clusterForgetObservable(String nodeId) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterForget(nodeId, handler.toHandler());
    return handler;
  }

  /**
   * Remove a node from the nodes table.
   * @param nodeId 
   * @return 
   */
  public Single<Void> rxClusterForget(String nodeId) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterForget(nodeId, fut);
    }));
  }

  /**
   * Return local key names in the specified hash slot.
   * @param slot 
   * @param count 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterGetkeysinslot(long slot, long count, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.clusterGetkeysinslot(slot, count, handler);
    return this;
  }

  /**
   * Return local key names in the specified hash slot.
   * @param slot 
   * @param count 
   * @return 
   * @deprecated use {@link #rxClusterGetkeysinslot} instead
   */
  @Deprecated()
  public Observable<JsonArray> clusterGetkeysinslotObservable(long slot, long count) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterGetkeysinslot(slot, count, handler.toHandler());
    return handler;
  }

  /**
   * Return local key names in the specified hash slot.
   * @param slot 
   * @param count 
   * @return 
   */
  public Single<JsonArray> rxClusterGetkeysinslot(long slot, long count) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterGetkeysinslot(slot, count, fut);
    }));
  }

  /**
   * Provides info about Redis Cluster node state.
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterInfo(Handler<AsyncResult<JsonArray>> handler) { 
    delegate.clusterInfo(handler);
    return this;
  }

  /**
   * Provides info about Redis Cluster node state.
   * @return 
   * @deprecated use {@link #rxClusterInfo} instead
   */
  @Deprecated()
  public Observable<JsonArray> clusterInfoObservable() { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterInfo(handler.toHandler());
    return handler;
  }

  /**
   * Provides info about Redis Cluster node state.
   * @return 
   */
  public Single<JsonArray> rxClusterInfo() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterInfo(fut);
    }));
  }

  /**
   * Returns the hash slot of the specified key.
   * @param key 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterKeyslot(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.clusterKeyslot(key, handler);
    return this;
  }

  /**
   * Returns the hash slot of the specified key.
   * @param key 
   * @return 
   * @deprecated use {@link #rxClusterKeyslot} instead
   */
  @Deprecated()
  public Observable<Long> clusterKeyslotObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterKeyslot(key, handler.toHandler());
    return handler;
  }

  /**
   * Returns the hash slot of the specified key.
   * @param key 
   * @return 
   */
  public Single<Long> rxClusterKeyslot(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterKeyslot(key, fut);
    }));
  }

  /**
   * Force a node cluster to handshake with another node.
   * @param ip 
   * @param port 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterMeet(String ip, long port, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterMeet(ip, port, handler);
    return this;
  }

  /**
   * Force a node cluster to handshake with another node.
   * @param ip 
   * @param port 
   * @return 
   * @deprecated use {@link #rxClusterMeet} instead
   */
  @Deprecated()
  public Observable<Void> clusterMeetObservable(String ip, long port) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterMeet(ip, port, handler.toHandler());
    return handler;
  }

  /**
   * Force a node cluster to handshake with another node.
   * @param ip 
   * @param port 
   * @return 
   */
  public Single<Void> rxClusterMeet(String ip, long port) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterMeet(ip, port, fut);
    }));
  }

  /**
   * Get Cluster config for the node.
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterNodes(Handler<AsyncResult<JsonArray>> handler) { 
    delegate.clusterNodes(handler);
    return this;
  }

  /**
   * Get Cluster config for the node.
   * @return 
   * @deprecated use {@link #rxClusterNodes} instead
   */
  @Deprecated()
  public Observable<JsonArray> clusterNodesObservable() { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterNodes(handler.toHandler());
    return handler;
  }

  /**
   * Get Cluster config for the node.
   * @return 
   */
  public Single<JsonArray> rxClusterNodes() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterNodes(fut);
    }));
  }

  /**
   * Reconfigure a node as a slave of the specified master node.
   * @param nodeId 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterReplicate(String nodeId, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterReplicate(nodeId, handler);
    return this;
  }

  /**
   * Reconfigure a node as a slave of the specified master node.
   * @param nodeId 
   * @return 
   * @deprecated use {@link #rxClusterReplicate} instead
   */
  @Deprecated()
  public Observable<Void> clusterReplicateObservable(String nodeId) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterReplicate(nodeId, handler.toHandler());
    return handler;
  }

  /**
   * Reconfigure a node as a slave of the specified master node.
   * @param nodeId 
   * @return 
   */
  public Single<Void> rxClusterReplicate(String nodeId) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterReplicate(nodeId, fut);
    }));
  }

  /**
   * Reset a Redis Cluster node.
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterReset(Handler<AsyncResult<Void>> handler) { 
    delegate.clusterReset(handler);
    return this;
  }

  /**
   * Reset a Redis Cluster node.
   * @return 
   * @deprecated use {@link #rxClusterReset} instead
   */
  @Deprecated()
  public Observable<Void> clusterResetObservable() { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterReset(handler.toHandler());
    return handler;
  }

  /**
   * Reset a Redis Cluster node.
   * @return 
   */
  public Single<Void> rxClusterReset() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterReset(fut);
    }));
  }

  /**
   * Reset a Redis Cluster node.
   * @param options 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterResetWithOptions(ResetOptions options, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterResetWithOptions(options, handler);
    return this;
  }

  /**
   * Reset a Redis Cluster node.
   * @param options 
   * @return 
   * @deprecated use {@link #rxClusterResetWithOptions} instead
   */
  @Deprecated()
  public Observable<Void> clusterResetWithOptionsObservable(ResetOptions options) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterResetWithOptions(options, handler.toHandler());
    return handler;
  }

  /**
   * Reset a Redis Cluster node.
   * @param options 
   * @return 
   */
  public Single<Void> rxClusterResetWithOptions(ResetOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterResetWithOptions(options, fut);
    }));
  }

  /**
   * Forces the node to save cluster state on disk.
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterSaveconfig(Handler<AsyncResult<Void>> handler) { 
    delegate.clusterSaveconfig(handler);
    return this;
  }

  /**
   * Forces the node to save cluster state on disk.
   * @return 
   * @deprecated use {@link #rxClusterSaveconfig} instead
   */
  @Deprecated()
  public Observable<Void> clusterSaveconfigObservable() { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterSaveconfig(handler.toHandler());
    return handler;
  }

  /**
   * Forces the node to save cluster state on disk.
   * @return 
   */
  public Single<Void> rxClusterSaveconfig() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterSaveconfig(fut);
    }));
  }

  /**
   * Set the configuration epoch in a new node.
   * @param epoch 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterSetConfigEpoch(long epoch, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterSetConfigEpoch(epoch, handler);
    return this;
  }

  /**
   * Set the configuration epoch in a new node.
   * @param epoch 
   * @return 
   * @deprecated use {@link #rxClusterSetConfigEpoch} instead
   */
  @Deprecated()
  public Observable<Void> clusterSetConfigEpochObservable(long epoch) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterSetConfigEpoch(epoch, handler.toHandler());
    return handler;
  }

  /**
   * Set the configuration epoch in a new node.
   * @param epoch 
   * @return 
   */
  public Single<Void> rxClusterSetConfigEpoch(long epoch) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterSetConfigEpoch(epoch, fut);
    }));
  }

  /**
   * Bind an hash slot to a specific node.
   * @param slot 
   * @param subcommand 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterSetslot(long slot, SlotCmd subcommand, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterSetslot(slot, subcommand, handler);
    return this;
  }

  /**
   * Bind an hash slot to a specific node.
   * @param slot 
   * @param subcommand 
   * @return 
   * @deprecated use {@link #rxClusterSetslot} instead
   */
  @Deprecated()
  public Observable<Void> clusterSetslotObservable(long slot, SlotCmd subcommand) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterSetslot(slot, subcommand, handler.toHandler());
    return handler;
  }

  /**
   * Bind an hash slot to a specific node.
   * @param slot 
   * @param subcommand 
   * @return 
   */
  public Single<Void> rxClusterSetslot(long slot, SlotCmd subcommand) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterSetslot(slot, subcommand, fut);
    }));
  }

  /**
   * Bind an hash slot to a specific node.
   * @param slot 
   * @param subcommand 
   * @param nodeId 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterSetslotWithNode(long slot, SlotCmd subcommand, String nodeId, Handler<AsyncResult<Void>> handler) { 
    delegate.clusterSetslotWithNode(slot, subcommand, nodeId, handler);
    return this;
  }

  /**
   * Bind an hash slot to a specific node.
   * @param slot 
   * @param subcommand 
   * @param nodeId 
   * @return 
   * @deprecated use {@link #rxClusterSetslotWithNode} instead
   */
  @Deprecated()
  public Observable<Void> clusterSetslotWithNodeObservable(long slot, SlotCmd subcommand, String nodeId) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterSetslotWithNode(slot, subcommand, nodeId, handler.toHandler());
    return handler;
  }

  /**
   * Bind an hash slot to a specific node.
   * @param slot 
   * @param subcommand 
   * @param nodeId 
   * @return 
   */
  public Single<Void> rxClusterSetslotWithNode(long slot, SlotCmd subcommand, String nodeId) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterSetslotWithNode(slot, subcommand, nodeId, fut);
    }));
  }

  /**
   * List slave nodes of the specified master node.
   * @param nodeId 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient clusterSlaves(String nodeId, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.clusterSlaves(nodeId, handler);
    return this;
  }

  /**
   * List slave nodes of the specified master node.
   * @param nodeId 
   * @return 
   * @deprecated use {@link #rxClusterSlaves} instead
   */
  @Deprecated()
  public Observable<JsonArray> clusterSlavesObservable(String nodeId) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterSlaves(nodeId, handler.toHandler());
    return handler;
  }

  /**
   * List slave nodes of the specified master node.
   * @param nodeId 
   * @return 
   */
  public Single<JsonArray> rxClusterSlaves(String nodeId) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterSlaves(nodeId, fut);
    }));
  }

  /**
   * Get array of Cluster slot to node mappings
   * @param handler 
   * @return 
   */
  public RedisClient clusterSlots(Handler<AsyncResult<JsonArray>> handler) { 
    delegate.clusterSlots(handler);
    return this;
  }

  /**
   * Get array of Cluster slot to node mappings
   * @return 
   * @deprecated use {@link #rxClusterSlots} instead
   */
  @Deprecated()
  public Observable<JsonArray> clusterSlotsObservable() { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clusterSlots(handler.toHandler());
    return handler;
  }

  /**
   * Get array of Cluster slot to node mappings
   * @return 
   */
  public Single<JsonArray> rxClusterSlots() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clusterSlots(fut);
    }));
  }

  /**
   * Get array of Redis command details
   * @param handler 
   * @return 
   */
  public RedisClient command(Handler<AsyncResult<JsonArray>> handler) { 
    delegate.command(handler);
    return this;
  }

  /**
   * Get array of Redis command details
   * @return 
   * @deprecated use {@link #rxCommand} instead
   */
  @Deprecated()
  public Observable<JsonArray> commandObservable() { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    command(handler.toHandler());
    return handler;
  }

  /**
   * Get array of Redis command details
   * @return 
   */
  public Single<JsonArray> rxCommand() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      command(fut);
    }));
  }

  /**
   * Get total number of Redis commands
   * @param handler 
   * @return 
   */
  public RedisClient commandCount(Handler<AsyncResult<Long>> handler) { 
    delegate.commandCount(handler);
    return this;
  }

  /**
   * Get total number of Redis commands
   * @return 
   * @deprecated use {@link #rxCommandCount} instead
   */
  @Deprecated()
  public Observable<Long> commandCountObservable() { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    commandCount(handler.toHandler());
    return handler;
  }

  /**
   * Get total number of Redis commands
   * @return 
   */
  public Single<Long> rxCommandCount() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      commandCount(fut);
    }));
  }

  /**
   * Extract keys given a full Redis command
   * @param handler 
   * @return 
   */
  public RedisClient commandGetkeys(Handler<AsyncResult<JsonArray>> handler) { 
    delegate.commandGetkeys(handler);
    return this;
  }

  /**
   * Extract keys given a full Redis command
   * @return 
   * @deprecated use {@link #rxCommandGetkeys} instead
   */
  @Deprecated()
  public Observable<JsonArray> commandGetkeysObservable() { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    commandGetkeys(handler.toHandler());
    return handler;
  }

  /**
   * Extract keys given a full Redis command
   * @return 
   */
  public Single<JsonArray> rxCommandGetkeys() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      commandGetkeys(fut);
    }));
  }

  /**
   * Get array of specific Redis command details
   * @param commands List of commands to get info for
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient commandInfo(List<String> commands, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.commandInfo(commands, handler);
    return this;
  }

  /**
   * Get array of specific Redis command details
   * @param commands List of commands to get info for
   * @return 
   * @deprecated use {@link #rxCommandInfo} instead
   */
  @Deprecated()
  public Observable<JsonArray> commandInfoObservable(List<String> commands) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    commandInfo(commands, handler.toHandler());
    return handler;
  }

  /**
   * Get array of specific Redis command details
   * @param commands List of commands to get info for
   * @return 
   */
  public Single<JsonArray> rxCommandInfo(List<String> commands) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      commandInfo(commands, fut);
    }));
  }

  /**
   * Get the value of a configuration parameter
   * @param parameter Configuration parameter
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient configGet(String parameter, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.configGet(parameter, handler);
    return this;
  }

  /**
   * Get the value of a configuration parameter
   * @param parameter Configuration parameter
   * @return 
   * @deprecated use {@link #rxConfigGet} instead
   */
  @Deprecated()
  public Observable<JsonArray> configGetObservable(String parameter) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    configGet(parameter, handler.toHandler());
    return handler;
  }

  /**
   * Get the value of a configuration parameter
   * @param parameter Configuration parameter
   * @return 
   */
  public Single<JsonArray> rxConfigGet(String parameter) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      configGet(parameter, fut);
    }));
  }

  /**
   * Rewrite the configuration file with the in memory configuration
   * @param handler 
   * @return 
   */
  public RedisClient configRewrite(Handler<AsyncResult<String>> handler) { 
    delegate.configRewrite(handler);
    return this;
  }

  /**
   * Rewrite the configuration file with the in memory configuration
   * @return 
   * @deprecated use {@link #rxConfigRewrite} instead
   */
  @Deprecated()
  public Observable<String> configRewriteObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    configRewrite(handler.toHandler());
    return handler;
  }

  /**
   * Rewrite the configuration file with the in memory configuration
   * @return 
   */
  public Single<String> rxConfigRewrite() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      configRewrite(fut);
    }));
  }

  /**
   * Set a configuration parameter to the given value
   * @param parameter Configuration parameter
   * @param value New value
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient configSet(String parameter, String value, Handler<AsyncResult<String>> handler) { 
    delegate.configSet(parameter, value, handler);
    return this;
  }

  /**
   * Set a configuration parameter to the given value
   * @param parameter Configuration parameter
   * @param value New value
   * @return 
   * @deprecated use {@link #rxConfigSet} instead
   */
  @Deprecated()
  public Observable<String> configSetObservable(String parameter, String value) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    configSet(parameter, value, handler.toHandler());
    return handler;
  }

  /**
   * Set a configuration parameter to the given value
   * @param parameter Configuration parameter
   * @param value New value
   * @return 
   */
  public Single<String> rxConfigSet(String parameter, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      configSet(parameter, value, fut);
    }));
  }

  /**
   * Reset the stats returned by INFO
   * @param handler 
   * @return 
   */
  public RedisClient configResetstat(Handler<AsyncResult<String>> handler) { 
    delegate.configResetstat(handler);
    return this;
  }

  /**
   * Reset the stats returned by INFO
   * @return 
   * @deprecated use {@link #rxConfigResetstat} instead
   */
  @Deprecated()
  public Observable<String> configResetstatObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    configResetstat(handler.toHandler());
    return handler;
  }

  /**
   * Reset the stats returned by INFO
   * @return 
   */
  public Single<String> rxConfigResetstat() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      configResetstat(fut);
    }));
  }

  /**
   * Return the number of keys in the selected database
   * @param handler 
   * @return 
   */
  public RedisClient dbsize(Handler<AsyncResult<Long>> handler) { 
    delegate.dbsize(handler);
    return this;
  }

  /**
   * Return the number of keys in the selected database
   * @return 
   * @deprecated use {@link #rxDbsize} instead
   */
  @Deprecated()
  public Observable<Long> dbsizeObservable() { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    dbsize(handler.toHandler());
    return handler;
  }

  /**
   * Return the number of keys in the selected database
   * @return 
   */
  public Single<Long> rxDbsize() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      dbsize(fut);
    }));
  }

  /**
   * Get debugging information about a key
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient debugObject(String key, Handler<AsyncResult<String>> handler) { 
    delegate.debugObject(key, handler);
    return this;
  }

  /**
   * Get debugging information about a key
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxDebugObject} instead
   */
  @Deprecated()
  public Observable<String> debugObjectObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    debugObject(key, handler.toHandler());
    return handler;
  }

  /**
   * Get debugging information about a key
   * @param key Key string
   * @return 
   */
  public Single<String> rxDebugObject(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      debugObject(key, fut);
    }));
  }

  /**
   * Make the server crash
   * @param handler 
   * @return 
   */
  public RedisClient debugSegfault(Handler<AsyncResult<String>> handler) { 
    delegate.debugSegfault(handler);
    return this;
  }

  /**
   * Make the server crash
   * @return 
   * @deprecated use {@link #rxDebugSegfault} instead
   */
  @Deprecated()
  public Observable<String> debugSegfaultObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    debugSegfault(handler.toHandler());
    return handler;
  }

  /**
   * Make the server crash
   * @return 
   */
  public Single<String> rxDebugSegfault() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      debugSegfault(fut);
    }));
  }

  /**
   * Decrement the integer value of a key by one
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient decr(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.decr(key, handler);
    return this;
  }

  /**
   * Decrement the integer value of a key by one
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxDecr} instead
   */
  @Deprecated()
  public Observable<Long> decrObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    decr(key, handler.toHandler());
    return handler;
  }

  /**
   * Decrement the integer value of a key by one
   * @param key Key string
   * @return 
   */
  public Single<Long> rxDecr(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      decr(key, fut);
    }));
  }

  /**
   * Decrement the integer value of a key by the given number
   * @param key Key string
   * @param decrement Value by which to decrement
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient decrby(String key, long decrement, Handler<AsyncResult<Long>> handler) { 
    delegate.decrby(key, decrement, handler);
    return this;
  }

  /**
   * Decrement the integer value of a key by the given number
   * @param key Key string
   * @param decrement Value by which to decrement
   * @return 
   * @deprecated use {@link #rxDecrby} instead
   */
  @Deprecated()
  public Observable<Long> decrbyObservable(String key, long decrement) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    decrby(key, decrement, handler.toHandler());
    return handler;
  }

  /**
   * Decrement the integer value of a key by the given number
   * @param key Key string
   * @param decrement Value by which to decrement
   * @return 
   */
  public Single<Long> rxDecrby(String key, long decrement) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      decrby(key, decrement, fut);
    }));
  }

  /**
   * Delete a key
   * @param key Keys to delete
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient del(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.del(key, handler);
    return this;
  }

  /**
   * Delete a key
   * @param key Keys to delete
   * @return 
   * @deprecated use {@link #rxDel} instead
   */
  @Deprecated()
  public Observable<Long> delObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    del(key, handler.toHandler());
    return handler;
  }

  /**
   * Delete a key
   * @param key Keys to delete
   * @return 
   */
  public Single<Long> rxDel(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      del(key, fut);
    }));
  }

  /**
   * Delete many keys
   * @param keys List of keys to delete
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient delMany(List<String> keys, Handler<AsyncResult<Long>> handler) { 
    delegate.delMany(keys, handler);
    return this;
  }

  /**
   * Delete many keys
   * @param keys List of keys to delete
   * @return 
   * @deprecated use {@link #rxDelMany} instead
   */
  @Deprecated()
  public Observable<Long> delManyObservable(List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    delMany(keys, handler.toHandler());
    return handler;
  }

  /**
   * Delete many keys
   * @param keys List of keys to delete
   * @return 
   */
  public Single<Long> rxDelMany(List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      delMany(keys, fut);
    }));
  }

  /**
   * Return a serialized version of the value stored at the specified key.
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient dump(String key, Handler<AsyncResult<String>> handler) { 
    delegate.dump(key, handler);
    return this;
  }

  /**
   * Return a serialized version of the value stored at the specified key.
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxDump} instead
   */
  @Deprecated()
  public Observable<String> dumpObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    dump(key, handler.toHandler());
    return handler;
  }

  /**
   * Return a serialized version of the value stored at the specified key.
   * @param key Key string
   * @return 
   */
  public Single<String> rxDump(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      dump(key, fut);
    }));
  }

  /**
   * Echo the given string
   * @param message String to echo
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient echo(String message, Handler<AsyncResult<String>> handler) { 
    delegate.echo(message, handler);
    return this;
  }

  /**
   * Echo the given string
   * @param message String to echo
   * @return 
   * @deprecated use {@link #rxEcho} instead
   */
  @Deprecated()
  public Observable<String> echoObservable(String message) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    echo(message, handler.toHandler());
    return handler;
  }

  /**
   * Echo the given string
   * @param message String to echo
   * @return 
   */
  public Single<String> rxEcho(String message) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      echo(message, fut);
    }));
  }

  /**
   * Execute a Lua script server side. Due to the dynamic nature of this command any response type could be returned
   * for This reason and to ensure type safety the reply is always guaranteed to be a JsonArray.
   *
   * When a reply if for example a String the handler will be called with a JsonArray with a single element containing
   * the String.
   * @param script Lua script to evaluate
   * @param keys List of keys
   * @param args List of argument values
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient eval(String script, List<String> keys, List<String> args, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.eval(script, keys, args, handler);
    return this;
  }

  /**
   * Execute a Lua script server side. Due to the dynamic nature of this command any response type could be returned
   * for This reason and to ensure type safety the reply is always guaranteed to be a JsonArray.
   *
   * When a reply if for example a String the handler will be called with a JsonArray with a single element containing
   * the String.
   * @param script Lua script to evaluate
   * @param keys List of keys
   * @param args List of argument values
   * @return 
   * @deprecated use {@link #rxEval} instead
   */
  @Deprecated()
  public Observable<JsonArray> evalObservable(String script, List<String> keys, List<String> args) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    eval(script, keys, args, handler.toHandler());
    return handler;
  }

  /**
   * Execute a Lua script server side. Due to the dynamic nature of this command any response type could be returned
   * for This reason and to ensure type safety the reply is always guaranteed to be a JsonArray.
   *
   * When a reply if for example a String the handler will be called with a JsonArray with a single element containing
   * the String.
   * @param script Lua script to evaluate
   * @param keys List of keys
   * @param args List of argument values
   * @return 
   */
  public Single<JsonArray> rxEval(String script, List<String> keys, List<String> args) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      eval(script, keys, args, fut);
    }));
  }

  /**
   * Execute a Lua script server side. Due to the dynamic nature of this command any response type could be returned
   * for This reason and to ensure type safety the reply is always guaranteed to be a JsonArray.
   *
   * When a reply if for example a String the handler will be called with a JsonArray with a single element containing
   * the String.
   * @param sha1 SHA1 digest of the script cached on the server
   * @param keys List of keys
   * @param values List of values
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient evalsha(String sha1, List<String> keys, List<String> values, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.evalsha(sha1, keys, values, handler);
    return this;
  }

  /**
   * Execute a Lua script server side. Due to the dynamic nature of this command any response type could be returned
   * for This reason and to ensure type safety the reply is always guaranteed to be a JsonArray.
   *
   * When a reply if for example a String the handler will be called with a JsonArray with a single element containing
   * the String.
   * @param sha1 SHA1 digest of the script cached on the server
   * @param keys List of keys
   * @param values List of values
   * @return 
   * @deprecated use {@link #rxEvalsha} instead
   */
  @Deprecated()
  public Observable<JsonArray> evalshaObservable(String sha1, List<String> keys, List<String> values) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    evalsha(sha1, keys, values, handler.toHandler());
    return handler;
  }

  /**
   * Execute a Lua script server side. Due to the dynamic nature of this command any response type could be returned
   * for This reason and to ensure type safety the reply is always guaranteed to be a JsonArray.
   *
   * When a reply if for example a String the handler will be called with a JsonArray with a single element containing
   * the String.
   * @param sha1 SHA1 digest of the script cached on the server
   * @param keys List of keys
   * @param values List of values
   * @return 
   */
  public Single<JsonArray> rxEvalsha(String sha1, List<String> keys, List<String> values) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      evalsha(sha1, keys, values, fut);
    }));
  }

  /**
   * Execute a Lua script server side. This method is a high level wrapper around EVAL and EVALSHA
   * using the latter if possible, falling back to EVAL if the script is not cached by the server yet.
   * According to Redis documentation, executed scripts are guaranteed to be in the script cache of a
   * given execution of a Redis instance forever, which means typically the overhead incurred by
   * optimistically sending EVALSHA is minimal, while improving performance and saving bandwidth
   * compared to using EVAL every time.
   * @param script Lua script and its SHA1 digest
   * @param keys List of keys
   * @param args List of argument values
   * @param handler Handler for the result of this call. group: scripting
   * @return 
   */
  public RedisClient evalScript(Script script, List<String> keys, List<String> args, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.evalScript(script.getDelegate(), keys, args, handler);
    return this;
  }

  /**
   * Execute a Lua script server side. This method is a high level wrapper around EVAL and EVALSHA
   * using the latter if possible, falling back to EVAL if the script is not cached by the server yet.
   * According to Redis documentation, executed scripts are guaranteed to be in the script cache of a
   * given execution of a Redis instance forever, which means typically the overhead incurred by
   * optimistically sending EVALSHA is minimal, while improving performance and saving bandwidth
   * compared to using EVAL every time.
   * @param script Lua script and its SHA1 digest
   * @param keys List of keys
   * @param args List of argument values
   * @return 
   * @deprecated use {@link #rxEvalScript} instead
   */
  @Deprecated()
  public Observable<JsonArray> evalScriptObservable(Script script, List<String> keys, List<String> args) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    evalScript(script, keys, args, handler.toHandler());
    return handler;
  }

  /**
   * Execute a Lua script server side. This method is a high level wrapper around EVAL and EVALSHA
   * using the latter if possible, falling back to EVAL if the script is not cached by the server yet.
   * According to Redis documentation, executed scripts are guaranteed to be in the script cache of a
   * given execution of a Redis instance forever, which means typically the overhead incurred by
   * optimistically sending EVALSHA is minimal, while improving performance and saving bandwidth
   * compared to using EVAL every time.
   * @param script Lua script and its SHA1 digest
   * @param keys List of keys
   * @param args List of argument values
   * @return 
   */
  public Single<JsonArray> rxEvalScript(Script script, List<String> keys, List<String> args) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      evalScript(script, keys, args, fut);
    }));
  }

  /**
   * Determine if a key exists
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient exists(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.exists(key, handler);
    return this;
  }

  /**
   * Determine if a key exists
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxExists} instead
   */
  @Deprecated()
  public Observable<Long> existsObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    exists(key, handler.toHandler());
    return handler;
  }

  /**
   * Determine if a key exists
   * @param key Key string
   * @return 
   */
  public Single<Long> rxExists(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      exists(key, fut);
    }));
  }

  /**
   * Determine if one or many keys exist
   * @param keys List of key strings
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient existsMany(List<String> keys, Handler<AsyncResult<Long>> handler) { 
    delegate.existsMany(keys, handler);
    return this;
  }

  /**
   * Determine if one or many keys exist
   * @param keys List of key strings
   * @return 
   * @deprecated use {@link #rxExistsMany} instead
   */
  @Deprecated()
  public Observable<Long> existsManyObservable(List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    existsMany(keys, handler.toHandler());
    return handler;
  }

  /**
   * Determine if one or many keys exist
   * @param keys List of key strings
   * @return 
   */
  public Single<Long> rxExistsMany(List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      existsMany(keys, fut);
    }));
  }

  /**
   * Set a key's time to live in seconds
   * @param key Key string
   * @param seconds Time to live in seconds
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient expire(String key, long seconds, Handler<AsyncResult<Long>> handler) { 
    delegate.expire(key, seconds, handler);
    return this;
  }

  /**
   * Set a key's time to live in seconds
   * @param key Key string
   * @param seconds Time to live in seconds
   * @return 
   * @deprecated use {@link #rxExpire} instead
   */
  @Deprecated()
  public Observable<Long> expireObservable(String key, long seconds) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    expire(key, seconds, handler.toHandler());
    return handler;
  }

  /**
   * Set a key's time to live in seconds
   * @param key Key string
   * @param seconds Time to live in seconds
   * @return 
   */
  public Single<Long> rxExpire(String key, long seconds) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      expire(key, seconds, fut);
    }));
  }

  /**
   * Set the expiration for a key as a UNIX timestamp
   * @param key Key string
   * @param seconds Expiry time as Unix timestamp in seconds
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient expireat(String key, long seconds, Handler<AsyncResult<Long>> handler) { 
    delegate.expireat(key, seconds, handler);
    return this;
  }

  /**
   * Set the expiration for a key as a UNIX timestamp
   * @param key Key string
   * @param seconds Expiry time as Unix timestamp in seconds
   * @return 
   * @deprecated use {@link #rxExpireat} instead
   */
  @Deprecated()
  public Observable<Long> expireatObservable(String key, long seconds) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    expireat(key, seconds, handler.toHandler());
    return handler;
  }

  /**
   * Set the expiration for a key as a UNIX timestamp
   * @param key Key string
   * @param seconds Expiry time as Unix timestamp in seconds
   * @return 
   */
  public Single<Long> rxExpireat(String key, long seconds) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      expireat(key, seconds, fut);
    }));
  }

  /**
   * Remove all keys from all databases
   * @param handler 
   * @return 
   */
  public RedisClient flushall(Handler<AsyncResult<String>> handler) { 
    delegate.flushall(handler);
    return this;
  }

  /**
   * Remove all keys from all databases
   * @return 
   * @deprecated use {@link #rxFlushall} instead
   */
  @Deprecated()
  public Observable<String> flushallObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    flushall(handler.toHandler());
    return handler;
  }

  /**
   * Remove all keys from all databases
   * @return 
   */
  public Single<String> rxFlushall() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      flushall(fut);
    }));
  }

  /**
   * Remove all keys from the current database
   * @param handler 
   * @return 
   */
  public RedisClient flushdb(Handler<AsyncResult<String>> handler) { 
    delegate.flushdb(handler);
    return this;
  }

  /**
   * Remove all keys from the current database
   * @return 
   * @deprecated use {@link #rxFlushdb} instead
   */
  @Deprecated()
  public Observable<String> flushdbObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    flushdb(handler.toHandler());
    return handler;
  }

  /**
   * Remove all keys from the current database
   * @return 
   */
  public Single<String> rxFlushdb() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      flushdb(fut);
    }));
  }

  /**
   * Get the value of a key
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient get(String key, Handler<AsyncResult<String>> handler) { 
    delegate.get(key, handler);
    return this;
  }

  /**
   * Get the value of a key
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxGet} instead
   */
  @Deprecated()
  public Observable<String> getObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    get(key, handler.toHandler());
    return handler;
  }

  /**
   * Get the value of a key
   * @param key Key string
   * @return 
   */
  public Single<String> rxGet(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      get(key, fut);
    }));
  }

  /**
   * Get the value of a key - without decoding as utf-8
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient getBinary(String key, Handler<AsyncResult<Buffer>> handler) { 
    delegate.getBinary(key, new Handler<AsyncResult<io.vertx.core.buffer.Buffer>>() {
      public void handle(AsyncResult<io.vertx.core.buffer.Buffer> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(Buffer.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Get the value of a key - without decoding as utf-8
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxGetBinary} instead
   */
  @Deprecated()
  public Observable<Buffer> getBinaryObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Buffer> handler = io.vertx.rx.java.RxHelper.observableFuture();
    getBinary(key, handler.toHandler());
    return handler;
  }

  /**
   * Get the value of a key - without decoding as utf-8
   * @param key Key string
   * @return 
   */
  public Single<Buffer> rxGetBinary(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      getBinary(key, fut);
    }));
  }

  /**
   * Returns the bit value at offset in the string value stored at key
   * @param key Key string
   * @param offset Offset in bits
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient getbit(String key, long offset, Handler<AsyncResult<Long>> handler) { 
    delegate.getbit(key, offset, handler);
    return this;
  }

  /**
   * Returns the bit value at offset in the string value stored at key
   * @param key Key string
   * @param offset Offset in bits
   * @return 
   * @deprecated use {@link #rxGetbit} instead
   */
  @Deprecated()
  public Observable<Long> getbitObservable(String key, long offset) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    getbit(key, offset, handler.toHandler());
    return handler;
  }

  /**
   * Returns the bit value at offset in the string value stored at key
   * @param key Key string
   * @param offset Offset in bits
   * @return 
   */
  public Single<Long> rxGetbit(String key, long offset) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      getbit(key, offset, fut);
    }));
  }

  /**
   * Get a substring of the string stored at a key
   * @param key Key string
   * @param start Start offset
   * @param end End offset - inclusive
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient getrange(String key, long start, long end, Handler<AsyncResult<String>> handler) { 
    delegate.getrange(key, start, end, handler);
    return this;
  }

  /**
   * Get a substring of the string stored at a key
   * @param key Key string
   * @param start Start offset
   * @param end End offset - inclusive
   * @return 
   * @deprecated use {@link #rxGetrange} instead
   */
  @Deprecated()
  public Observable<String> getrangeObservable(String key, long start, long end) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    getrange(key, start, end, handler.toHandler());
    return handler;
  }

  /**
   * Get a substring of the string stored at a key
   * @param key Key string
   * @param start Start offset
   * @param end End offset - inclusive
   * @return 
   */
  public Single<String> rxGetrange(String key, long start, long end) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      getrange(key, start, end, fut);
    }));
  }

  /**
   * Set the string value of a key and return its old value
   * @param key Key of which value to set
   * @param value New value for the key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient getset(String key, String value, Handler<AsyncResult<String>> handler) { 
    delegate.getset(key, value, handler);
    return this;
  }

  /**
   * Set the string value of a key and return its old value
   * @param key Key of which value to set
   * @param value New value for the key
   * @return 
   * @deprecated use {@link #rxGetset} instead
   */
  @Deprecated()
  public Observable<String> getsetObservable(String key, String value) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    getset(key, value, handler.toHandler());
    return handler;
  }

  /**
   * Set the string value of a key and return its old value
   * @param key Key of which value to set
   * @param value New value for the key
   * @return 
   */
  public Single<String> rxGetset(String key, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      getset(key, value, fut);
    }));
  }

  /**
   * Delete one or more hash fields
   * @param key Key string
   * @param field Field name
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hdel(String key, String field, Handler<AsyncResult<Long>> handler) { 
    delegate.hdel(key, field, handler);
    return this;
  }

  /**
   * Delete one or more hash fields
   * @param key Key string
   * @param field Field name
   * @return 
   * @deprecated use {@link #rxHdel} instead
   */
  @Deprecated()
  public Observable<Long> hdelObservable(String key, String field) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hdel(key, field, handler.toHandler());
    return handler;
  }

  /**
   * Delete one or more hash fields
   * @param key Key string
   * @param field Field name
   * @return 
   */
  public Single<Long> rxHdel(String key, String field) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hdel(key, field, fut);
    }));
  }

  /**
   * Delete one or more hash fields
   * @param key Key string
   * @param fields Field names
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hdelMany(String key, List<String> fields, Handler<AsyncResult<Long>> handler) { 
    delegate.hdelMany(key, fields, handler);
    return this;
  }

  /**
   * Delete one or more hash fields
   * @param key Key string
   * @param fields Field names
   * @return 
   * @deprecated use {@link #rxHdelMany} instead
   */
  @Deprecated()
  public Observable<Long> hdelManyObservable(String key, List<String> fields) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hdelMany(key, fields, handler.toHandler());
    return handler;
  }

  /**
   * Delete one or more hash fields
   * @param key Key string
   * @param fields Field names
   * @return 
   */
  public Single<Long> rxHdelMany(String key, List<String> fields) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hdelMany(key, fields, fut);
    }));
  }

  /**
   * Determine if a hash field exists
   * @param key Key string
   * @param field Field name
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hexists(String key, String field, Handler<AsyncResult<Long>> handler) { 
    delegate.hexists(key, field, handler);
    return this;
  }

  /**
   * Determine if a hash field exists
   * @param key Key string
   * @param field Field name
   * @return 
   * @deprecated use {@link #rxHexists} instead
   */
  @Deprecated()
  public Observable<Long> hexistsObservable(String key, String field) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hexists(key, field, handler.toHandler());
    return handler;
  }

  /**
   * Determine if a hash field exists
   * @param key Key string
   * @param field Field name
   * @return 
   */
  public Single<Long> rxHexists(String key, String field) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hexists(key, field, fut);
    }));
  }

  /**
   * Get the value of a hash field
   * @param key Key string
   * @param field Field name
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hget(String key, String field, Handler<AsyncResult<String>> handler) { 
    delegate.hget(key, field, handler);
    return this;
  }

  /**
   * Get the value of a hash field
   * @param key Key string
   * @param field Field name
   * @return 
   * @deprecated use {@link #rxHget} instead
   */
  @Deprecated()
  public Observable<String> hgetObservable(String key, String field) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hget(key, field, handler.toHandler());
    return handler;
  }

  /**
   * Get the value of a hash field
   * @param key Key string
   * @param field Field name
   * @return 
   */
  public Single<String> rxHget(String key, String field) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hget(key, field, fut);
    }));
  }

  /**
   * Get all the fields and values in a hash
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hgetall(String key, Handler<AsyncResult<JsonObject>> handler) { 
    delegate.hgetall(key, handler);
    return this;
  }

  /**
   * Get all the fields and values in a hash
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxHgetall} instead
   */
  @Deprecated()
  public Observable<JsonObject> hgetallObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<JsonObject> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hgetall(key, handler.toHandler());
    return handler;
  }

  /**
   * Get all the fields and values in a hash
   * @param key Key string
   * @return 
   */
  public Single<JsonObject> rxHgetall(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hgetall(key, fut);
    }));
  }

  /**
   * Increment the integer value of a hash field by the given number
   * @param key Key string
   * @param field Field name
   * @param increment Value by which to increment
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hincrby(String key, String field, long increment, Handler<AsyncResult<Long>> handler) { 
    delegate.hincrby(key, field, increment, handler);
    return this;
  }

  /**
   * Increment the integer value of a hash field by the given number
   * @param key Key string
   * @param field Field name
   * @param increment Value by which to increment
   * @return 
   * @deprecated use {@link #rxHincrby} instead
   */
  @Deprecated()
  public Observable<Long> hincrbyObservable(String key, String field, long increment) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hincrby(key, field, increment, handler.toHandler());
    return handler;
  }

  /**
   * Increment the integer value of a hash field by the given number
   * @param key Key string
   * @param field Field name
   * @param increment Value by which to increment
   * @return 
   */
  public Single<Long> rxHincrby(String key, String field, long increment) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hincrby(key, field, increment, fut);
    }));
  }

  /**
   * Increment the float value of a hash field by the given amount
   * @param key Key string
   * @param field Field name
   * @param increment Value by which to increment
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hincrbyfloat(String key, String field, double increment, Handler<AsyncResult<String>> handler) { 
    delegate.hincrbyfloat(key, field, increment, handler);
    return this;
  }

  /**
   * Increment the float value of a hash field by the given amount
   * @param key Key string
   * @param field Field name
   * @param increment Value by which to increment
   * @return 
   * @deprecated use {@link #rxHincrbyfloat} instead
   */
  @Deprecated()
  public Observable<String> hincrbyfloatObservable(String key, String field, double increment) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hincrbyfloat(key, field, increment, handler.toHandler());
    return handler;
  }

  /**
   * Increment the float value of a hash field by the given amount
   * @param key Key string
   * @param field Field name
   * @param increment Value by which to increment
   * @return 
   */
  public Single<String> rxHincrbyfloat(String key, String field, double increment) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hincrbyfloat(key, field, increment, fut);
    }));
  }

  /**
   * Get all the fields in a hash
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hkeys(String key, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.hkeys(key, handler);
    return this;
  }

  /**
   * Get all the fields in a hash
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxHkeys} instead
   */
  @Deprecated()
  public Observable<JsonArray> hkeysObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hkeys(key, handler.toHandler());
    return handler;
  }

  /**
   * Get all the fields in a hash
   * @param key Key string
   * @return 
   */
  public Single<JsonArray> rxHkeys(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hkeys(key, fut);
    }));
  }

  /**
   * Get the number of fields in a hash
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hlen(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.hlen(key, handler);
    return this;
  }

  /**
   * Get the number of fields in a hash
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxHlen} instead
   */
  @Deprecated()
  public Observable<Long> hlenObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hlen(key, handler.toHandler());
    return handler;
  }

  /**
   * Get the number of fields in a hash
   * @param key Key string
   * @return 
   */
  public Single<Long> rxHlen(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hlen(key, fut);
    }));
  }

  /**
   * Get the values of all the given hash fields
   * @param key Key string
   * @param fields Field names
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hmget(String key, List<String> fields, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.hmget(key, fields, handler);
    return this;
  }

  /**
   * Get the values of all the given hash fields
   * @param key Key string
   * @param fields Field names
   * @return 
   * @deprecated use {@link #rxHmget} instead
   */
  @Deprecated()
  public Observable<JsonArray> hmgetObservable(String key, List<String> fields) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hmget(key, fields, handler.toHandler());
    return handler;
  }

  /**
   * Get the values of all the given hash fields
   * @param key Key string
   * @param fields Field names
   * @return 
   */
  public Single<JsonArray> rxHmget(String key, List<String> fields) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hmget(key, fields, fut);
    }));
  }

  /**
   * Set multiple hash fields to multiple values
   * @param key Key string
   * @param values Map of field:value pairs
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hmset(String key, JsonObject values, Handler<AsyncResult<String>> handler) { 
    delegate.hmset(key, values, handler);
    return this;
  }

  /**
   * Set multiple hash fields to multiple values
   * @param key Key string
   * @param values Map of field:value pairs
   * @return 
   * @deprecated use {@link #rxHmset} instead
   */
  @Deprecated()
  public Observable<String> hmsetObservable(String key, JsonObject values) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hmset(key, values, handler.toHandler());
    return handler;
  }

  /**
   * Set multiple hash fields to multiple values
   * @param key Key string
   * @param values Map of field:value pairs
   * @return 
   */
  public Single<String> rxHmset(String key, JsonObject values) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hmset(key, values, fut);
    }));
  }

  /**
   * Set the string value of a hash field
   * @param key Key string
   * @param field Field name
   * @param value New value
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hset(String key, String field, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.hset(key, field, value, handler);
    return this;
  }

  /**
   * Set the string value of a hash field
   * @param key Key string
   * @param field Field name
   * @param value New value
   * @return 
   * @deprecated use {@link #rxHset} instead
   */
  @Deprecated()
  public Observable<Long> hsetObservable(String key, String field, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hset(key, field, value, handler.toHandler());
    return handler;
  }

  /**
   * Set the string value of a hash field
   * @param key Key string
   * @param field Field name
   * @param value New value
   * @return 
   */
  public Single<Long> rxHset(String key, String field, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hset(key, field, value, fut);
    }));
  }

  /**
   * Set the value of a hash field, only if the field does not exist
   * @param key Key string
   * @param field Field name
   * @param value New value
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hsetnx(String key, String field, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.hsetnx(key, field, value, handler);
    return this;
  }

  /**
   * Set the value of a hash field, only if the field does not exist
   * @param key Key string
   * @param field Field name
   * @param value New value
   * @return 
   * @deprecated use {@link #rxHsetnx} instead
   */
  @Deprecated()
  public Observable<Long> hsetnxObservable(String key, String field, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hsetnx(key, field, value, handler.toHandler());
    return handler;
  }

  /**
   * Set the value of a hash field, only if the field does not exist
   * @param key Key string
   * @param field Field name
   * @param value New value
   * @return 
   */
  public Single<Long> rxHsetnx(String key, String field, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hsetnx(key, field, value, fut);
    }));
  }

  /**
   * Get all the values in a hash
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hvals(String key, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.hvals(key, handler);
    return this;
  }

  /**
   * Get all the values in a hash
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxHvals} instead
   */
  @Deprecated()
  public Observable<JsonArray> hvalsObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hvals(key, handler.toHandler());
    return handler;
  }

  /**
   * Get all the values in a hash
   * @param key Key string
   * @return 
   */
  public Single<JsonArray> rxHvals(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hvals(key, fut);
    }));
  }

  /**
   * Increment the integer value of a key by one
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient incr(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.incr(key, handler);
    return this;
  }

  /**
   * Increment the integer value of a key by one
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxIncr} instead
   */
  @Deprecated()
  public Observable<Long> incrObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    incr(key, handler.toHandler());
    return handler;
  }

  /**
   * Increment the integer value of a key by one
   * @param key Key string
   * @return 
   */
  public Single<Long> rxIncr(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      incr(key, fut);
    }));
  }

  /**
   * Increment the integer value of a key by the given amount
   * @param key Key string
   * @param increment Value by which to increment
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient incrby(String key, long increment, Handler<AsyncResult<Long>> handler) { 
    delegate.incrby(key, increment, handler);
    return this;
  }

  /**
   * Increment the integer value of a key by the given amount
   * @param key Key string
   * @param increment Value by which to increment
   * @return 
   * @deprecated use {@link #rxIncrby} instead
   */
  @Deprecated()
  public Observable<Long> incrbyObservable(String key, long increment) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    incrby(key, increment, handler.toHandler());
    return handler;
  }

  /**
   * Increment the integer value of a key by the given amount
   * @param key Key string
   * @param increment Value by which to increment
   * @return 
   */
  public Single<Long> rxIncrby(String key, long increment) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      incrby(key, increment, fut);
    }));
  }

  /**
   * Increment the float value of a key by the given amount
   * @param key Key string
   * @param increment Value by which to increment
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient incrbyfloat(String key, double increment, Handler<AsyncResult<String>> handler) { 
    delegate.incrbyfloat(key, increment, handler);
    return this;
  }

  /**
   * Increment the float value of a key by the given amount
   * @param key Key string
   * @param increment Value by which to increment
   * @return 
   * @deprecated use {@link #rxIncrbyfloat} instead
   */
  @Deprecated()
  public Observable<String> incrbyfloatObservable(String key, double increment) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    incrbyfloat(key, increment, handler.toHandler());
    return handler;
  }

  /**
   * Increment the float value of a key by the given amount
   * @param key Key string
   * @param increment Value by which to increment
   * @return 
   */
  public Single<String> rxIncrbyfloat(String key, double increment) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      incrbyfloat(key, increment, fut);
    }));
  }

  /**
   * Get information and statistics about the server
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient info(Handler<AsyncResult<JsonObject>> handler) { 
    delegate.info(handler);
    return this;
  }

  /**
   * Get information and statistics about the server
   * @return 
   * @deprecated use {@link #rxInfo} instead
   */
  @Deprecated()
  public Observable<JsonObject> infoObservable() { 
    io.vertx.rx.java.ObservableFuture<JsonObject> handler = io.vertx.rx.java.RxHelper.observableFuture();
    info(handler.toHandler());
    return handler;
  }

  /**
   * Get information and statistics about the server
   * @return 
   */
  public Single<JsonObject> rxInfo() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      info(fut);
    }));
  }

  /**
   * Get information and statistics about the server
   * @param section Specific section of information to return
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient infoSection(String section, Handler<AsyncResult<JsonObject>> handler) { 
    delegate.infoSection(section, handler);
    return this;
  }

  /**
   * Get information and statistics about the server
   * @param section Specific section of information to return
   * @return 
   * @deprecated use {@link #rxInfoSection} instead
   */
  @Deprecated()
  public Observable<JsonObject> infoSectionObservable(String section) { 
    io.vertx.rx.java.ObservableFuture<JsonObject> handler = io.vertx.rx.java.RxHelper.observableFuture();
    infoSection(section, handler.toHandler());
    return handler;
  }

  /**
   * Get information and statistics about the server
   * @param section Specific section of information to return
   * @return 
   */
  public Single<JsonObject> rxInfoSection(String section) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      infoSection(section, fut);
    }));
  }

  /**
   * Find all keys matching the given pattern
   * @param pattern Pattern to limit the keys returned
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient keys(String pattern, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.keys(pattern, handler);
    return this;
  }

  /**
   * Find all keys matching the given pattern
   * @param pattern Pattern to limit the keys returned
   * @return 
   * @deprecated use {@link #rxKeys} instead
   */
  @Deprecated()
  public Observable<JsonArray> keysObservable(String pattern) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    keys(pattern, handler.toHandler());
    return handler;
  }

  /**
   * Find all keys matching the given pattern
   * @param pattern Pattern to limit the keys returned
   * @return 
   */
  public Single<JsonArray> rxKeys(String pattern) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      keys(pattern, fut);
    }));
  }

  /**
   * Get the UNIX time stamp of the last successful save to disk
   * @param handler 
   * @return 
   */
  public RedisClient lastsave(Handler<AsyncResult<Long>> handler) { 
    delegate.lastsave(handler);
    return this;
  }

  /**
   * Get the UNIX time stamp of the last successful save to disk
   * @return 
   * @deprecated use {@link #rxLastsave} instead
   */
  @Deprecated()
  public Observable<Long> lastsaveObservable() { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    lastsave(handler.toHandler());
    return handler;
  }

  /**
   * Get the UNIX time stamp of the last successful save to disk
   * @return 
   */
  public Single<Long> rxLastsave() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      lastsave(fut);
    }));
  }

  /**
   * Get an element from a list by its index
   * @param key Key string
   * @param index Index of list element to get
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient lindex(String key, int index, Handler<AsyncResult<String>> handler) { 
    delegate.lindex(key, index, handler);
    return this;
  }

  /**
   * Get an element from a list by its index
   * @param key Key string
   * @param index Index of list element to get
   * @return 
   * @deprecated use {@link #rxLindex} instead
   */
  @Deprecated()
  public Observable<String> lindexObservable(String key, int index) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    lindex(key, index, handler.toHandler());
    return handler;
  }

  /**
   * Get an element from a list by its index
   * @param key Key string
   * @param index Index of list element to get
   * @return 
   */
  public Single<String> rxLindex(String key, int index) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      lindex(key, index, fut);
    }));
  }

  /**
   * Insert an element before or after another element in a list
   * @param key Key string
   * @param option BEFORE or AFTER
   * @param pivot Key to use as a pivot
   * @param value Value to be inserted before or after the pivot
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient linsert(String key, InsertOptions option, String pivot, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.linsert(key, option, pivot, value, handler);
    return this;
  }

  /**
   * Insert an element before or after another element in a list
   * @param key Key string
   * @param option BEFORE or AFTER
   * @param pivot Key to use as a pivot
   * @param value Value to be inserted before or after the pivot
   * @return 
   * @deprecated use {@link #rxLinsert} instead
   */
  @Deprecated()
  public Observable<Long> linsertObservable(String key, InsertOptions option, String pivot, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    linsert(key, option, pivot, value, handler.toHandler());
    return handler;
  }

  /**
   * Insert an element before or after another element in a list
   * @param key Key string
   * @param option BEFORE or AFTER
   * @param pivot Key to use as a pivot
   * @param value Value to be inserted before or after the pivot
   * @return 
   */
  public Single<Long> rxLinsert(String key, InsertOptions option, String pivot, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      linsert(key, option, pivot, value, fut);
    }));
  }

  /**
   * Get the length of a list
   * @param key String key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient llen(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.llen(key, handler);
    return this;
  }

  /**
   * Get the length of a list
   * @param key String key
   * @return 
   * @deprecated use {@link #rxLlen} instead
   */
  @Deprecated()
  public Observable<Long> llenObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    llen(key, handler.toHandler());
    return handler;
  }

  /**
   * Get the length of a list
   * @param key String key
   * @return 
   */
  public Single<Long> rxLlen(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      llen(key, fut);
    }));
  }

  /**
   * Remove and get the first element in a list
   * @param key String key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient lpop(String key, Handler<AsyncResult<String>> handler) { 
    delegate.lpop(key, handler);
    return this;
  }

  /**
   * Remove and get the first element in a list
   * @param key String key
   * @return 
   * @deprecated use {@link #rxLpop} instead
   */
  @Deprecated()
  public Observable<String> lpopObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    lpop(key, handler.toHandler());
    return handler;
  }

  /**
   * Remove and get the first element in a list
   * @param key String key
   * @return 
   */
  public Single<String> rxLpop(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      lpop(key, fut);
    }));
  }

  /**
   * Prepend one or multiple values to a list
   * @param key Key string
   * @param values Values to be added at the beginning of the list, one by one
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient lpushMany(String key, List<String> values, Handler<AsyncResult<Long>> handler) { 
    delegate.lpushMany(key, values, handler);
    return this;
  }

  /**
   * Prepend one or multiple values to a list
   * @param key Key string
   * @param values Values to be added at the beginning of the list, one by one
   * @return 
   * @deprecated use {@link #rxLpushMany} instead
   */
  @Deprecated()
  public Observable<Long> lpushManyObservable(String key, List<String> values) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    lpushMany(key, values, handler.toHandler());
    return handler;
  }

  /**
   * Prepend one or multiple values to a list
   * @param key Key string
   * @param values Values to be added at the beginning of the list, one by one
   * @return 
   */
  public Single<Long> rxLpushMany(String key, List<String> values) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      lpushMany(key, values, fut);
    }));
  }

  /**
   * Prepend one value to a list
   * @param key Key string
   * @param value Value to be added at the beginning of the list
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient lpush(String key, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.lpush(key, value, handler);
    return this;
  }

  /**
   * Prepend one value to a list
   * @param key Key string
   * @param value Value to be added at the beginning of the list
   * @return 
   * @deprecated use {@link #rxLpush} instead
   */
  @Deprecated()
  public Observable<Long> lpushObservable(String key, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    lpush(key, value, handler.toHandler());
    return handler;
  }

  /**
   * Prepend one value to a list
   * @param key Key string
   * @param value Value to be added at the beginning of the list
   * @return 
   */
  public Single<Long> rxLpush(String key, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      lpush(key, value, fut);
    }));
  }

  /**
   * Prepend a value to a list, only if the list exists
   * @param key Key string
   * @param value Value to add at the beginning of the list
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient lpushx(String key, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.lpushx(key, value, handler);
    return this;
  }

  /**
   * Prepend a value to a list, only if the list exists
   * @param key Key string
   * @param value Value to add at the beginning of the list
   * @return 
   * @deprecated use {@link #rxLpushx} instead
   */
  @Deprecated()
  public Observable<Long> lpushxObservable(String key, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    lpushx(key, value, handler.toHandler());
    return handler;
  }

  /**
   * Prepend a value to a list, only if the list exists
   * @param key Key string
   * @param value Value to add at the beginning of the list
   * @return 
   */
  public Single<Long> rxLpushx(String key, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      lpushx(key, value, fut);
    }));
  }

  /**
   * Get a range of elements from a list
   * @param key Key string
   * @param from Start index
   * @param to Stop index
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient lrange(String key, long from, long to, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.lrange(key, from, to, handler);
    return this;
  }

  /**
   * Get a range of elements from a list
   * @param key Key string
   * @param from Start index
   * @param to Stop index
   * @return 
   * @deprecated use {@link #rxLrange} instead
   */
  @Deprecated()
  public Observable<JsonArray> lrangeObservable(String key, long from, long to) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    lrange(key, from, to, handler.toHandler());
    return handler;
  }

  /**
   * Get a range of elements from a list
   * @param key Key string
   * @param from Start index
   * @param to Stop index
   * @return 
   */
  public Single<JsonArray> rxLrange(String key, long from, long to) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      lrange(key, from, to, fut);
    }));
  }

  /**
   * Remove elements from a list
   * @param key Key string
   * @param count Number of first found occurrences equal to $value to remove from the list
   * @param value Value to be removed
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient lrem(String key, long count, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.lrem(key, count, value, handler);
    return this;
  }

  /**
   * Remove elements from a list
   * @param key Key string
   * @param count Number of first found occurrences equal to $value to remove from the list
   * @param value Value to be removed
   * @return 
   * @deprecated use {@link #rxLrem} instead
   */
  @Deprecated()
  public Observable<Long> lremObservable(String key, long count, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    lrem(key, count, value, handler.toHandler());
    return handler;
  }

  /**
   * Remove elements from a list
   * @param key Key string
   * @param count Number of first found occurrences equal to $value to remove from the list
   * @param value Value to be removed
   * @return 
   */
  public Single<Long> rxLrem(String key, long count, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      lrem(key, count, value, fut);
    }));
  }

  /**
   * Set the value of an element in a list by its index
   * @param key Key string
   * @param index Position within list
   * @param value New value
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient lset(String key, long index, String value, Handler<AsyncResult<String>> handler) { 
    delegate.lset(key, index, value, handler);
    return this;
  }

  /**
   * Set the value of an element in a list by its index
   * @param key Key string
   * @param index Position within list
   * @param value New value
   * @return 
   * @deprecated use {@link #rxLset} instead
   */
  @Deprecated()
  public Observable<String> lsetObservable(String key, long index, String value) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    lset(key, index, value, handler.toHandler());
    return handler;
  }

  /**
   * Set the value of an element in a list by its index
   * @param key Key string
   * @param index Position within list
   * @param value New value
   * @return 
   */
  public Single<String> rxLset(String key, long index, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      lset(key, index, value, fut);
    }));
  }

  /**
   * Trim a list to the specified range
   * @param key Key string
   * @param from Start index
   * @param to Stop index
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient ltrim(String key, long from, long to, Handler<AsyncResult<String>> handler) { 
    delegate.ltrim(key, from, to, handler);
    return this;
  }

  /**
   * Trim a list to the specified range
   * @param key Key string
   * @param from Start index
   * @param to Stop index
   * @return 
   * @deprecated use {@link #rxLtrim} instead
   */
  @Deprecated()
  public Observable<String> ltrimObservable(String key, long from, long to) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    ltrim(key, from, to, handler.toHandler());
    return handler;
  }

  /**
   * Trim a list to the specified range
   * @param key Key string
   * @param from Start index
   * @param to Stop index
   * @return 
   */
  public Single<String> rxLtrim(String key, long from, long to) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      ltrim(key, from, to, fut);
    }));
  }

  /**
   * Get the value of the given key
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient mget(String key, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.mget(key, handler);
    return this;
  }

  /**
   * Get the value of the given key
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxMget} instead
   */
  @Deprecated()
  public Observable<JsonArray> mgetObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    mget(key, handler.toHandler());
    return handler;
  }

  /**
   * Get the value of the given key
   * @param key Key string
   * @return 
   */
  public Single<JsonArray> rxMget(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      mget(key, fut);
    }));
  }

  /**
   * Get the values of all the given keys
   * @param keys List of keys to get
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient mgetMany(List<String> keys, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.mgetMany(keys, handler);
    return this;
  }

  /**
   * Get the values of all the given keys
   * @param keys List of keys to get
   * @return 
   * @deprecated use {@link #rxMgetMany} instead
   */
  @Deprecated()
  public Observable<JsonArray> mgetManyObservable(List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    mgetMany(keys, handler.toHandler());
    return handler;
  }

  /**
   * Get the values of all the given keys
   * @param keys List of keys to get
   * @return 
   */
  public Single<JsonArray> rxMgetMany(List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      mgetMany(keys, fut);
    }));
  }

  /**
   * Atomically transfer a key from a Redis instance to another one.
   * @param host Destination host
   * @param port Destination port
   * @param key Key to migrate
   * @param destdb Destination database index
   * @param timeout 
   * @param options Migrate options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient migrate(String host, int port, String key, int destdb, long timeout, MigrateOptions options, Handler<AsyncResult<String>> handler) { 
    delegate.migrate(host, port, key, destdb, timeout, options, handler);
    return this;
  }

  /**
   * Atomically transfer a key from a Redis instance to another one.
   * @param host Destination host
   * @param port Destination port
   * @param key Key to migrate
   * @param destdb Destination database index
   * @param timeout 
   * @param options Migrate options
   * @return 
   * @deprecated use {@link #rxMigrate} instead
   */
  @Deprecated()
  public Observable<String> migrateObservable(String host, int port, String key, int destdb, long timeout, MigrateOptions options) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    migrate(host, port, key, destdb, timeout, options, handler.toHandler());
    return handler;
  }

  /**
   * Atomically transfer a key from a Redis instance to another one.
   * @param host Destination host
   * @param port Destination port
   * @param key Key to migrate
   * @param destdb Destination database index
   * @param timeout 
   * @param options Migrate options
   * @return 
   */
  public Single<String> rxMigrate(String host, int port, String key, int destdb, long timeout, MigrateOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      migrate(host, port, key, destdb, timeout, options, fut);
    }));
  }

  /**
   * Listen for all requests received by the server in real time
   * @param handler 
   * @return 
   */
  public RedisClient monitor(Handler<AsyncResult<Void>> handler) { 
    delegate.monitor(handler);
    return this;
  }

  /**
   * Listen for all requests received by the server in real time
   * @return 
   * @deprecated use {@link #rxMonitor} instead
   */
  @Deprecated()
  public Observable<Void> monitorObservable() { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    monitor(handler.toHandler());
    return handler;
  }

  /**
   * Listen for all requests received by the server in real time
   * @return 
   */
  public Single<Void> rxMonitor() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      monitor(fut);
    }));
  }

  /**
   * Move a key to another database
   * @param key Key to migrate
   * @param destdb Destination database index
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient move(String key, int destdb, Handler<AsyncResult<Long>> handler) { 
    delegate.move(key, destdb, handler);
    return this;
  }

  /**
   * Move a key to another database
   * @param key Key to migrate
   * @param destdb Destination database index
   * @return 
   * @deprecated use {@link #rxMove} instead
   */
  @Deprecated()
  public Observable<Long> moveObservable(String key, int destdb) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    move(key, destdb, handler.toHandler());
    return handler;
  }

  /**
   * Move a key to another database
   * @param key Key to migrate
   * @param destdb Destination database index
   * @return 
   */
  public Single<Long> rxMove(String key, int destdb) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      move(key, destdb, fut);
    }));
  }

  /**
   * Set multiple keys to multiple values
   * @param keyvals Key value pairs to set
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient mset(JsonObject keyvals, Handler<AsyncResult<String>> handler) { 
    delegate.mset(keyvals, handler);
    return this;
  }

  /**
   * Set multiple keys to multiple values
   * @param keyvals Key value pairs to set
   * @return 
   * @deprecated use {@link #rxMset} instead
   */
  @Deprecated()
  public Observable<String> msetObservable(JsonObject keyvals) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    mset(keyvals, handler.toHandler());
    return handler;
  }

  /**
   * Set multiple keys to multiple values
   * @param keyvals Key value pairs to set
   * @return 
   */
  public Single<String> rxMset(JsonObject keyvals) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      mset(keyvals, fut);
    }));
  }

  /**
   * Set multiple keys to multiple values, only if none of the keys exist
   * @param keyvals Key value pairs to set
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient msetnx(JsonObject keyvals, Handler<AsyncResult<Long>> handler) { 
    delegate.msetnx(keyvals, handler);
    return this;
  }

  /**
   * Set multiple keys to multiple values, only if none of the keys exist
   * @param keyvals Key value pairs to set
   * @return 
   * @deprecated use {@link #rxMsetnx} instead
   */
  @Deprecated()
  public Observable<Long> msetnxObservable(JsonObject keyvals) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    msetnx(keyvals, handler.toHandler());
    return handler;
  }

  /**
   * Set multiple keys to multiple values, only if none of the keys exist
   * @param keyvals Key value pairs to set
   * @return 
   */
  public Single<Long> rxMsetnx(JsonObject keyvals) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      msetnx(keyvals, fut);
    }));
  }

  /**
   * Inspect the internals of Redis objects
   * @param key Key string
   * @param cmd Object sub command
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient object(String key, ObjectCmd cmd, Handler<AsyncResult<Void>> handler) { 
    delegate.object(key, cmd, handler);
    return this;
  }

  /**
   * Inspect the internals of Redis objects
   * @param key Key string
   * @param cmd Object sub command
   * @return 
   * @deprecated use {@link #rxObject} instead
   */
  @Deprecated()
  public Observable<Void> objectObservable(String key, ObjectCmd cmd) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    object(key, cmd, handler.toHandler());
    return handler;
  }

  /**
   * Inspect the internals of Redis objects
   * @param key Key string
   * @param cmd Object sub command
   * @return 
   */
  public Single<Void> rxObject(String key, ObjectCmd cmd) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      object(key, cmd, fut);
    }));
  }

  /**
   * Remove the expiration from a key
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient persist(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.persist(key, handler);
    return this;
  }

  /**
   * Remove the expiration from a key
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxPersist} instead
   */
  @Deprecated()
  public Observable<Long> persistObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    persist(key, handler.toHandler());
    return handler;
  }

  /**
   * Remove the expiration from a key
   * @param key Key string
   * @return 
   */
  public Single<Long> rxPersist(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      persist(key, fut);
    }));
  }

  /**
   * Set a key's time to live in milliseconds
   * @param key String key
   * @param millis Time to live in milliseconds
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pexpire(String key, long millis, Handler<AsyncResult<Long>> handler) { 
    delegate.pexpire(key, millis, handler);
    return this;
  }

  /**
   * Set a key's time to live in milliseconds
   * @param key String key
   * @param millis Time to live in milliseconds
   * @return 
   * @deprecated use {@link #rxPexpire} instead
   */
  @Deprecated()
  public Observable<Long> pexpireObservable(String key, long millis) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pexpire(key, millis, handler.toHandler());
    return handler;
  }

  /**
   * Set a key's time to live in milliseconds
   * @param key String key
   * @param millis Time to live in milliseconds
   * @return 
   */
  public Single<Long> rxPexpire(String key, long millis) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pexpire(key, millis, fut);
    }));
  }

  /**
   * Set the expiration for a key as a UNIX timestamp specified in milliseconds
   * @param key Key string
   * @param millis Expiry time as Unix timestamp in milliseconds
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pexpireat(String key, long millis, Handler<AsyncResult<Long>> handler) { 
    delegate.pexpireat(key, millis, handler);
    return this;
  }

  /**
   * Set the expiration for a key as a UNIX timestamp specified in milliseconds
   * @param key Key string
   * @param millis Expiry time as Unix timestamp in milliseconds
   * @return 
   * @deprecated use {@link #rxPexpireat} instead
   */
  @Deprecated()
  public Observable<Long> pexpireatObservable(String key, long millis) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pexpireat(key, millis, handler.toHandler());
    return handler;
  }

  /**
   * Set the expiration for a key as a UNIX timestamp specified in milliseconds
   * @param key Key string
   * @param millis Expiry time as Unix timestamp in milliseconds
   * @return 
   */
  public Single<Long> rxPexpireat(String key, long millis) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pexpireat(key, millis, fut);
    }));
  }

  /**
   * Adds the specified element to the specified HyperLogLog.
   * @param key Key string
   * @param element Element to add
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pfadd(String key, String element, Handler<AsyncResult<Long>> handler) { 
    delegate.pfadd(key, element, handler);
    return this;
  }

  /**
   * Adds the specified element to the specified HyperLogLog.
   * @param key Key string
   * @param element Element to add
   * @return 
   * @deprecated use {@link #rxPfadd} instead
   */
  @Deprecated()
  public Observable<Long> pfaddObservable(String key, String element) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pfadd(key, element, handler.toHandler());
    return handler;
  }

  /**
   * Adds the specified element to the specified HyperLogLog.
   * @param key Key string
   * @param element Element to add
   * @return 
   */
  public Single<Long> rxPfadd(String key, String element) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pfadd(key, element, fut);
    }));
  }

  /**
   * Adds the specified elements to the specified HyperLogLog.
   * @param key Key string
   * @param elements Elementa to add
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pfaddMany(String key, List<String> elements, Handler<AsyncResult<Long>> handler) { 
    delegate.pfaddMany(key, elements, handler);
    return this;
  }

  /**
   * Adds the specified elements to the specified HyperLogLog.
   * @param key Key string
   * @param elements Elementa to add
   * @return 
   * @deprecated use {@link #rxPfaddMany} instead
   */
  @Deprecated()
  public Observable<Long> pfaddManyObservable(String key, List<String> elements) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pfaddMany(key, elements, handler.toHandler());
    return handler;
  }

  /**
   * Adds the specified elements to the specified HyperLogLog.
   * @param key Key string
   * @param elements Elementa to add
   * @return 
   */
  public Single<Long> rxPfaddMany(String key, List<String> elements) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pfaddMany(key, elements, fut);
    }));
  }

  /**
   * Return the approximated cardinality of the set observed by the HyperLogLog at key.
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pfcount(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.pfcount(key, handler);
    return this;
  }

  /**
   * Return the approximated cardinality of the set observed by the HyperLogLog at key.
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxPfcount} instead
   */
  @Deprecated()
  public Observable<Long> pfcountObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pfcount(key, handler.toHandler());
    return handler;
  }

  /**
   * Return the approximated cardinality of the set observed by the HyperLogLog at key.
   * @param key Key string
   * @return 
   */
  public Single<Long> rxPfcount(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pfcount(key, fut);
    }));
  }

  /**
   * Return the approximated cardinality of the set(s) observed by the HyperLogLog at key(s).
   * @param keys List of keys
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pfcountMany(List<String> keys, Handler<AsyncResult<Long>> handler) { 
    delegate.pfcountMany(keys, handler);
    return this;
  }

  /**
   * Return the approximated cardinality of the set(s) observed by the HyperLogLog at key(s).
   * @param keys List of keys
   * @return 
   * @deprecated use {@link #rxPfcountMany} instead
   */
  @Deprecated()
  public Observable<Long> pfcountManyObservable(List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pfcountMany(keys, handler.toHandler());
    return handler;
  }

  /**
   * Return the approximated cardinality of the set(s) observed by the HyperLogLog at key(s).
   * @param keys List of keys
   * @return 
   */
  public Single<Long> rxPfcountMany(List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pfcountMany(keys, fut);
    }));
  }

  /**
   * Merge N different HyperLogLogs into a single one.
   * @param destkey Destination key
   * @param keys List of source keys
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pfmerge(String destkey, List<String> keys, Handler<AsyncResult<String>> handler) { 
    delegate.pfmerge(destkey, keys, handler);
    return this;
  }

  /**
   * Merge N different HyperLogLogs into a single one.
   * @param destkey Destination key
   * @param keys List of source keys
   * @return 
   * @deprecated use {@link #rxPfmerge} instead
   */
  @Deprecated()
  public Observable<String> pfmergeObservable(String destkey, List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pfmerge(destkey, keys, handler.toHandler());
    return handler;
  }

  /**
   * Merge N different HyperLogLogs into a single one.
   * @param destkey Destination key
   * @param keys List of source keys
   * @return 
   */
  public Single<String> rxPfmerge(String destkey, List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pfmerge(destkey, keys, fut);
    }));
  }

  /**
   * Ping the server
   * @param handler 
   * @return 
   */
  public RedisClient ping(Handler<AsyncResult<String>> handler) { 
    delegate.ping(handler);
    return this;
  }

  /**
   * Ping the server
   * @return 
   * @deprecated use {@link #rxPing} instead
   */
  @Deprecated()
  public Observable<String> pingObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    ping(handler.toHandler());
    return handler;
  }

  /**
   * Ping the server
   * @return 
   */
  public Single<String> rxPing() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      ping(fut);
    }));
  }

  /**
   * Set the value and expiration in milliseconds of a key
   * @param key Key string
   * @param millis Number of milliseconds until the key expires
   * @param value New value for key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient psetex(String key, long millis, String value, Handler<AsyncResult<Void>> handler) { 
    delegate.psetex(key, millis, value, handler);
    return this;
  }

  /**
   * Set the value and expiration in milliseconds of a key
   * @param key Key string
   * @param millis Number of milliseconds until the key expires
   * @param value New value for key
   * @return 
   * @deprecated use {@link #rxPsetex} instead
   */
  @Deprecated()
  public Observable<Void> psetexObservable(String key, long millis, String value) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    psetex(key, millis, value, handler.toHandler());
    return handler;
  }

  /**
   * Set the value and expiration in milliseconds of a key
   * @param key Key string
   * @param millis Number of milliseconds until the key expires
   * @param value New value for key
   * @return 
   */
  public Single<Void> rxPsetex(String key, long millis, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      psetex(key, millis, value, fut);
    }));
  }

  /**
   * Listen for messages published to channels matching the given pattern
   * @param pattern Pattern string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient psubscribe(String pattern, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.psubscribe(pattern, handler);
    return this;
  }

  /**
   * Listen for messages published to channels matching the given pattern
   * @param pattern Pattern string
   * @return 
   * @deprecated use {@link #rxPsubscribe} instead
   */
  @Deprecated()
  public Observable<JsonArray> psubscribeObservable(String pattern) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    psubscribe(pattern, handler.toHandler());
    return handler;
  }

  /**
   * Listen for messages published to channels matching the given pattern
   * @param pattern Pattern string
   * @return 
   */
  public Single<JsonArray> rxPsubscribe(String pattern) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      psubscribe(pattern, fut);
    }));
  }

  /**
   * Listen for messages published to channels matching the given patterns
   * @param patterns List of patterns
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient psubscribeMany(List<String> patterns, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.psubscribeMany(patterns, handler);
    return this;
  }

  /**
   * Listen for messages published to channels matching the given patterns
   * @param patterns List of patterns
   * @return 
   * @deprecated use {@link #rxPsubscribeMany} instead
   */
  @Deprecated()
  public Observable<JsonArray> psubscribeManyObservable(List<String> patterns) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    psubscribeMany(patterns, handler.toHandler());
    return handler;
  }

  /**
   * Listen for messages published to channels matching the given patterns
   * @param patterns List of patterns
   * @return 
   */
  public Single<JsonArray> rxPsubscribeMany(List<String> patterns) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      psubscribeMany(patterns, fut);
    }));
  }

  /**
   * Lists the currently active channels - only those matching the pattern
   * @param pattern A glob-style pattern - an empty string means no pattern
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pubsubChannels(String pattern, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.pubsubChannels(pattern, handler);
    return this;
  }

  /**
   * Lists the currently active channels - only those matching the pattern
   * @param pattern A glob-style pattern - an empty string means no pattern
   * @return 
   * @deprecated use {@link #rxPubsubChannels} instead
   */
  @Deprecated()
  public Observable<JsonArray> pubsubChannelsObservable(String pattern) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pubsubChannels(pattern, handler.toHandler());
    return handler;
  }

  /**
   * Lists the currently active channels - only those matching the pattern
   * @param pattern A glob-style pattern - an empty string means no pattern
   * @return 
   */
  public Single<JsonArray> rxPubsubChannels(String pattern) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pubsubChannels(pattern, fut);
    }));
  }

  /**
   * Returns the number of subscribers (not counting clients subscribed to patterns) for the specified channels
   * @param channels List of channels
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pubsubNumsub(List<String> channels, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.pubsubNumsub(channels, handler);
    return this;
  }

  /**
   * Returns the number of subscribers (not counting clients subscribed to patterns) for the specified channels
   * @param channels List of channels
   * @return 
   * @deprecated use {@link #rxPubsubNumsub} instead
   */
  @Deprecated()
  public Observable<JsonArray> pubsubNumsubObservable(List<String> channels) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pubsubNumsub(channels, handler.toHandler());
    return handler;
  }

  /**
   * Returns the number of subscribers (not counting clients subscribed to patterns) for the specified channels
   * @param channels List of channels
   * @return 
   */
  public Single<JsonArray> rxPubsubNumsub(List<String> channels) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pubsubNumsub(channels, fut);
    }));
  }

  /**
   * Returns the number of subscriptions to patterns (that are performed using the PSUBSCRIBE command)
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pubsubNumpat(Handler<AsyncResult<Long>> handler) { 
    delegate.pubsubNumpat(handler);
    return this;
  }

  /**
   * Returns the number of subscriptions to patterns (that are performed using the PSUBSCRIBE command)
   * @return 
   * @deprecated use {@link #rxPubsubNumpat} instead
   */
  @Deprecated()
  public Observable<Long> pubsubNumpatObservable() { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pubsubNumpat(handler.toHandler());
    return handler;
  }

  /**
   * Returns the number of subscriptions to patterns (that are performed using the PSUBSCRIBE command)
   * @return 
   */
  public Single<Long> rxPubsubNumpat() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pubsubNumpat(fut);
    }));
  }

  /**
   * Get the time to live for a key in milliseconds
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient pttl(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.pttl(key, handler);
    return this;
  }

  /**
   * Get the time to live for a key in milliseconds
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxPttl} instead
   */
  @Deprecated()
  public Observable<Long> pttlObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    pttl(key, handler.toHandler());
    return handler;
  }

  /**
   * Get the time to live for a key in milliseconds
   * @param key Key string
   * @return 
   */
  public Single<Long> rxPttl(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      pttl(key, fut);
    }));
  }

  /**
   * Post a message to a channel
   * @param channel Channel key
   * @param message Message to send to channel
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient publish(String channel, String message, Handler<AsyncResult<Long>> handler) { 
    delegate.publish(channel, message, handler);
    return this;
  }

  /**
   * Post a message to a channel
   * @param channel Channel key
   * @param message Message to send to channel
   * @return 
   * @deprecated use {@link #rxPublish} instead
   */
  @Deprecated()
  public Observable<Long> publishObservable(String channel, String message) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    publish(channel, message, handler.toHandler());
    return handler;
  }

  /**
   * Post a message to a channel
   * @param channel Channel key
   * @param message Message to send to channel
   * @return 
   */
  public Single<Long> rxPublish(String channel, String message) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      publish(channel, message, fut);
    }));
  }

  /**
   * Stop listening for messages posted to channels matching the given patterns
   * @param patterns List of patterns to match against
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient punsubscribe(List<String> patterns, Handler<AsyncResult<Void>> handler) { 
    delegate.punsubscribe(patterns, handler);
    return this;
  }

  /**
   * Stop listening for messages posted to channels matching the given patterns
   * @param patterns List of patterns to match against
   * @return 
   * @deprecated use {@link #rxPunsubscribe} instead
   */
  @Deprecated()
  public Observable<Void> punsubscribeObservable(List<String> patterns) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    punsubscribe(patterns, handler.toHandler());
    return handler;
  }

  /**
   * Stop listening for messages posted to channels matching the given patterns
   * @param patterns List of patterns to match against
   * @return 
   */
  public Single<Void> rxPunsubscribe(List<String> patterns) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      punsubscribe(patterns, fut);
    }));
  }

  /**
   * Return a random key from the keyspace
   * @param handler 
   * @return 
   */
  public RedisClient randomkey(Handler<AsyncResult<String>> handler) { 
    delegate.randomkey(handler);
    return this;
  }

  /**
   * Return a random key from the keyspace
   * @return 
   * @deprecated use {@link #rxRandomkey} instead
   */
  @Deprecated()
  public Observable<String> randomkeyObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    randomkey(handler.toHandler());
    return handler;
  }

  /**
   * Return a random key from the keyspace
   * @return 
   */
  public Single<String> rxRandomkey() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      randomkey(fut);
    }));
  }

  /**
   * Rename a key
   * @param key Key string to be renamed
   * @param newkey New key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient rename(String key, String newkey, Handler<AsyncResult<String>> handler) { 
    delegate.rename(key, newkey, handler);
    return this;
  }

  /**
   * Rename a key
   * @param key Key string to be renamed
   * @param newkey New key string
   * @return 
   * @deprecated use {@link #rxRename} instead
   */
  @Deprecated()
  public Observable<String> renameObservable(String key, String newkey) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    rename(key, newkey, handler.toHandler());
    return handler;
  }

  /**
   * Rename a key
   * @param key Key string to be renamed
   * @param newkey New key string
   * @return 
   */
  public Single<String> rxRename(String key, String newkey) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      rename(key, newkey, fut);
    }));
  }

  /**
   * Rename a key, only if the new key does not exist
   * @param key Key string to be renamed
   * @param newkey New key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient renamenx(String key, String newkey, Handler<AsyncResult<Long>> handler) { 
    delegate.renamenx(key, newkey, handler);
    return this;
  }

  /**
   * Rename a key, only if the new key does not exist
   * @param key Key string to be renamed
   * @param newkey New key string
   * @return 
   * @deprecated use {@link #rxRenamenx} instead
   */
  @Deprecated()
  public Observable<Long> renamenxObservable(String key, String newkey) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    renamenx(key, newkey, handler.toHandler());
    return handler;
  }

  /**
   * Rename a key, only if the new key does not exist
   * @param key Key string to be renamed
   * @param newkey New key string
   * @return 
   */
  public Single<Long> rxRenamenx(String key, String newkey) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      renamenx(key, newkey, fut);
    }));
  }

  /**
   * Create a key using the provided serialized value, previously obtained using DUMP.
   * @param key Key string
   * @param millis Expiry time in milliseconds to set on the key
   * @param serialized Serialized form of the key value as obtained using DUMP
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient restore(String key, long millis, String serialized, Handler<AsyncResult<String>> handler) { 
    delegate.restore(key, millis, serialized, handler);
    return this;
  }

  /**
   * Create a key using the provided serialized value, previously obtained using DUMP.
   * @param key Key string
   * @param millis Expiry time in milliseconds to set on the key
   * @param serialized Serialized form of the key value as obtained using DUMP
   * @return 
   * @deprecated use {@link #rxRestore} instead
   */
  @Deprecated()
  public Observable<String> restoreObservable(String key, long millis, String serialized) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    restore(key, millis, serialized, handler.toHandler());
    return handler;
  }

  /**
   * Create a key using the provided serialized value, previously obtained using DUMP.
   * @param key Key string
   * @param millis Expiry time in milliseconds to set on the key
   * @param serialized Serialized form of the key value as obtained using DUMP
   * @return 
   */
  public Single<String> rxRestore(String key, long millis, String serialized) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      restore(key, millis, serialized, fut);
    }));
  }

  /**
   * Return the role of the instance in the context of replication
   * @param handler 
   * @return 
   */
  public RedisClient role(Handler<AsyncResult<JsonArray>> handler) { 
    delegate.role(handler);
    return this;
  }

  /**
   * Return the role of the instance in the context of replication
   * @return 
   * @deprecated use {@link #rxRole} instead
   */
  @Deprecated()
  public Observable<JsonArray> roleObservable() { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    role(handler.toHandler());
    return handler;
  }

  /**
   * Return the role of the instance in the context of replication
   * @return 
   */
  public Single<JsonArray> rxRole() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      role(fut);
    }));
  }

  /**
   * Remove and get the last element in a list
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient rpop(String key, Handler<AsyncResult<String>> handler) { 
    delegate.rpop(key, handler);
    return this;
  }

  /**
   * Remove and get the last element in a list
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxRpop} instead
   */
  @Deprecated()
  public Observable<String> rpopObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    rpop(key, handler.toHandler());
    return handler;
  }

  /**
   * Remove and get the last element in a list
   * @param key Key string
   * @return 
   */
  public Single<String> rxRpop(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      rpop(key, fut);
    }));
  }

  /**
   * Remove the last element in a list, append it to another list and return it
   * @param key Key string identifying source list
   * @param destkey Key string identifying destination list
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient rpoplpush(String key, String destkey, Handler<AsyncResult<String>> handler) { 
    delegate.rpoplpush(key, destkey, handler);
    return this;
  }

  /**
   * Remove the last element in a list, append it to another list and return it
   * @param key Key string identifying source list
   * @param destkey Key string identifying destination list
   * @return 
   * @deprecated use {@link #rxRpoplpush} instead
   */
  @Deprecated()
  public Observable<String> rpoplpushObservable(String key, String destkey) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    rpoplpush(key, destkey, handler.toHandler());
    return handler;
  }

  /**
   * Remove the last element in a list, append it to another list and return it
   * @param key Key string identifying source list
   * @param destkey Key string identifying destination list
   * @return 
   */
  public Single<String> rxRpoplpush(String key, String destkey) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      rpoplpush(key, destkey, fut);
    }));
  }

  /**
   * Append one or multiple values to a list
   * @param key Key string
   * @param values List of values to add to the end of the list
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient rpushMany(String key, List<String> values, Handler<AsyncResult<Long>> handler) { 
    delegate.rpushMany(key, values, handler);
    return this;
  }

  /**
   * Append one or multiple values to a list
   * @param key Key string
   * @param values List of values to add to the end of the list
   * @return 
   * @deprecated use {@link #rxRpushMany} instead
   */
  @Deprecated()
  public Observable<Long> rpushManyObservable(String key, List<String> values) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    rpushMany(key, values, handler.toHandler());
    return handler;
  }

  /**
   * Append one or multiple values to a list
   * @param key Key string
   * @param values List of values to add to the end of the list
   * @return 
   */
  public Single<Long> rxRpushMany(String key, List<String> values) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      rpushMany(key, values, fut);
    }));
  }

  /**
   * Append one or multiple values to a list
   * @param key Key string
   * @param value Value to be added to the end of the list
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient rpush(String key, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.rpush(key, value, handler);
    return this;
  }

  /**
   * Append one or multiple values to a list
   * @param key Key string
   * @param value Value to be added to the end of the list
   * @return 
   * @deprecated use {@link #rxRpush} instead
   */
  @Deprecated()
  public Observable<Long> rpushObservable(String key, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    rpush(key, value, handler.toHandler());
    return handler;
  }

  /**
   * Append one or multiple values to a list
   * @param key Key string
   * @param value Value to be added to the end of the list
   * @return 
   */
  public Single<Long> rxRpush(String key, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      rpush(key, value, fut);
    }));
  }

  /**
   * Append a value to a list, only if the list exists
   * @param key Key string
   * @param value Value to be added to the end of the list
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient rpushx(String key, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.rpushx(key, value, handler);
    return this;
  }

  /**
   * Append a value to a list, only if the list exists
   * @param key Key string
   * @param value Value to be added to the end of the list
   * @return 
   * @deprecated use {@link #rxRpushx} instead
   */
  @Deprecated()
  public Observable<Long> rpushxObservable(String key, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    rpushx(key, value, handler.toHandler());
    return handler;
  }

  /**
   * Append a value to a list, only if the list exists
   * @param key Key string
   * @param value Value to be added to the end of the list
   * @return 
   */
  public Single<Long> rxRpushx(String key, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      rpushx(key, value, fut);
    }));
  }

  /**
   * Add a member to a set
   * @param key Key string
   * @param member Value to be added to the set
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sadd(String key, String member, Handler<AsyncResult<Long>> handler) { 
    delegate.sadd(key, member, handler);
    return this;
  }

  /**
   * Add a member to a set
   * @param key Key string
   * @param member Value to be added to the set
   * @return 
   * @deprecated use {@link #rxSadd} instead
   */
  @Deprecated()
  public Observable<Long> saddObservable(String key, String member) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sadd(key, member, handler.toHandler());
    return handler;
  }

  /**
   * Add a member to a set
   * @param key Key string
   * @param member Value to be added to the set
   * @return 
   */
  public Single<Long> rxSadd(String key, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sadd(key, member, fut);
    }));
  }

  /**
   * Add one or more members to a set
   * @param key Key string
   * @param members Values to be added to the set
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient saddMany(String key, List<String> members, Handler<AsyncResult<Long>> handler) { 
    delegate.saddMany(key, members, handler);
    return this;
  }

  /**
   * Add one or more members to a set
   * @param key Key string
   * @param members Values to be added to the set
   * @return 
   * @deprecated use {@link #rxSaddMany} instead
   */
  @Deprecated()
  public Observable<Long> saddManyObservable(String key, List<String> members) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    saddMany(key, members, handler.toHandler());
    return handler;
  }

  /**
   * Add one or more members to a set
   * @param key Key string
   * @param members Values to be added to the set
   * @return 
   */
  public Single<Long> rxSaddMany(String key, List<String> members) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      saddMany(key, members, fut);
    }));
  }

  /**
   * Synchronously save the dataset to disk
   * @param handler 
   * @return 
   */
  public RedisClient save(Handler<AsyncResult<String>> handler) { 
    delegate.save(handler);
    return this;
  }

  /**
   * Synchronously save the dataset to disk
   * @return 
   * @deprecated use {@link #rxSave} instead
   */
  @Deprecated()
  public Observable<String> saveObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    save(handler.toHandler());
    return handler;
  }

  /**
   * Synchronously save the dataset to disk
   * @return 
   */
  public Single<String> rxSave() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      save(fut);
    }));
  }

  /**
   * Get the number of members in a set
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient scard(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.scard(key, handler);
    return this;
  }

  /**
   * Get the number of members in a set
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxScard} instead
   */
  @Deprecated()
  public Observable<Long> scardObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    scard(key, handler.toHandler());
    return handler;
  }

  /**
   * Get the number of members in a set
   * @param key Key string
   * @return 
   */
  public Single<Long> rxScard(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      scard(key, fut);
    }));
  }

  /**
   * Check existence of script in the script cache.
   * @param script SHA1 digest identifying a script in the script cache
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient scriptExists(String script, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.scriptExists(script, handler);
    return this;
  }

  /**
   * Check existence of script in the script cache.
   * @param script SHA1 digest identifying a script in the script cache
   * @return 
   * @deprecated use {@link #rxScriptExists} instead
   */
  @Deprecated()
  public Observable<JsonArray> scriptExistsObservable(String script) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    scriptExists(script, handler.toHandler());
    return handler;
  }

  /**
   * Check existence of script in the script cache.
   * @param script SHA1 digest identifying a script in the script cache
   * @return 
   */
  public Single<JsonArray> rxScriptExists(String script) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      scriptExists(script, fut);
    }));
  }

  /**
   * Check existence of scripts in the script cache.
   * @param scripts List of SHA1 digests identifying scripts in the script cache
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient scriptExistsMany(List<String> scripts, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.scriptExistsMany(scripts, handler);
    return this;
  }

  /**
   * Check existence of scripts in the script cache.
   * @param scripts List of SHA1 digests identifying scripts in the script cache
   * @return 
   * @deprecated use {@link #rxScriptExistsMany} instead
   */
  @Deprecated()
  public Observable<JsonArray> scriptExistsManyObservable(List<String> scripts) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    scriptExistsMany(scripts, handler.toHandler());
    return handler;
  }

  /**
   * Check existence of scripts in the script cache.
   * @param scripts List of SHA1 digests identifying scripts in the script cache
   * @return 
   */
  public Single<JsonArray> rxScriptExistsMany(List<String> scripts) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      scriptExistsMany(scripts, fut);
    }));
  }

  /**
   * Remove all the scripts from the script cache.
   * @param handler 
   * @return 
   */
  public RedisClient scriptFlush(Handler<AsyncResult<String>> handler) { 
    delegate.scriptFlush(handler);
    return this;
  }

  /**
   * Remove all the scripts from the script cache.
   * @return 
   * @deprecated use {@link #rxScriptFlush} instead
   */
  @Deprecated()
  public Observable<String> scriptFlushObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    scriptFlush(handler.toHandler());
    return handler;
  }

  /**
   * Remove all the scripts from the script cache.
   * @return 
   */
  public Single<String> rxScriptFlush() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      scriptFlush(fut);
    }));
  }

  /**
   * Kill the script currently in execution.
   * @param handler 
   * @return 
   */
  public RedisClient scriptKill(Handler<AsyncResult<String>> handler) { 
    delegate.scriptKill(handler);
    return this;
  }

  /**
   * Kill the script currently in execution.
   * @return 
   * @deprecated use {@link #rxScriptKill} instead
   */
  @Deprecated()
  public Observable<String> scriptKillObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    scriptKill(handler.toHandler());
    return handler;
  }

  /**
   * Kill the script currently in execution.
   * @return 
   */
  public Single<String> rxScriptKill() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      scriptKill(fut);
    }));
  }

  /**
   * Load the specified Lua script into the script cache.
   * @param script Lua script
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient scriptLoad(String script, Handler<AsyncResult<String>> handler) { 
    delegate.scriptLoad(script, handler);
    return this;
  }

  /**
   * Load the specified Lua script into the script cache.
   * @param script Lua script
   * @return 
   * @deprecated use {@link #rxScriptLoad} instead
   */
  @Deprecated()
  public Observable<String> scriptLoadObservable(String script) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    scriptLoad(script, handler.toHandler());
    return handler;
  }

  /**
   * Load the specified Lua script into the script cache.
   * @param script Lua script
   * @return 
   */
  public Single<String> rxScriptLoad(String script) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      scriptLoad(script, fut);
    }));
  }

  /**
   * Subtract multiple sets
   * @param key Key identifying the set to compare with all other sets combined
   * @param cmpkeys List of keys identifying sets to subtract from the key set
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sdiff(String key, List<String> cmpkeys, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.sdiff(key, cmpkeys, handler);
    return this;
  }

  /**
   * Subtract multiple sets
   * @param key Key identifying the set to compare with all other sets combined
   * @param cmpkeys List of keys identifying sets to subtract from the key set
   * @return 
   * @deprecated use {@link #rxSdiff} instead
   */
  @Deprecated()
  public Observable<JsonArray> sdiffObservable(String key, List<String> cmpkeys) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sdiff(key, cmpkeys, handler.toHandler());
    return handler;
  }

  /**
   * Subtract multiple sets
   * @param key Key identifying the set to compare with all other sets combined
   * @param cmpkeys List of keys identifying sets to subtract from the key set
   * @return 
   */
  public Single<JsonArray> rxSdiff(String key, List<String> cmpkeys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sdiff(key, cmpkeys, fut);
    }));
  }

  /**
   * Subtract multiple sets and store the resulting set in a key
   * @param destkey Destination key where the result should be stored
   * @param key Key identifying the set to compare with all other sets combined
   * @param cmpkeys List of keys identifying sets to subtract from the key set
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sdiffstore(String destkey, String key, List<String> cmpkeys, Handler<AsyncResult<Long>> handler) { 
    delegate.sdiffstore(destkey, key, cmpkeys, handler);
    return this;
  }

  /**
   * Subtract multiple sets and store the resulting set in a key
   * @param destkey Destination key where the result should be stored
   * @param key Key identifying the set to compare with all other sets combined
   * @param cmpkeys List of keys identifying sets to subtract from the key set
   * @return 
   * @deprecated use {@link #rxSdiffstore} instead
   */
  @Deprecated()
  public Observable<Long> sdiffstoreObservable(String destkey, String key, List<String> cmpkeys) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sdiffstore(destkey, key, cmpkeys, handler.toHandler());
    return handler;
  }

  /**
   * Subtract multiple sets and store the resulting set in a key
   * @param destkey Destination key where the result should be stored
   * @param key Key identifying the set to compare with all other sets combined
   * @param cmpkeys List of keys identifying sets to subtract from the key set
   * @return 
   */
  public Single<Long> rxSdiffstore(String destkey, String key, List<String> cmpkeys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sdiffstore(destkey, key, cmpkeys, fut);
    }));
  }

  /**
   * Change the selected database for the current connection
   * @param dbindex Index identifying the new active database
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient select(int dbindex, Handler<AsyncResult<String>> handler) { 
    delegate.select(dbindex, handler);
    return this;
  }

  /**
   * Change the selected database for the current connection
   * @param dbindex Index identifying the new active database
   * @return 
   * @deprecated use {@link #rxSelect} instead
   */
  @Deprecated()
  public Observable<String> selectObservable(int dbindex) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    select(dbindex, handler.toHandler());
    return handler;
  }

  /**
   * Change the selected database for the current connection
   * @param dbindex Index identifying the new active database
   * @return 
   */
  public Single<String> rxSelect(int dbindex) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      select(dbindex, fut);
    }));
  }

  /**
   * Set the string value of a key
   * @param key Key of which value to set
   * @param value New value for the key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient set(String key, String value, Handler<AsyncResult<Void>> handler) { 
    delegate.set(key, value, handler);
    return this;
  }

  /**
   * Set the string value of a key
   * @param key Key of which value to set
   * @param value New value for the key
   * @return 
   * @deprecated use {@link #rxSet} instead
   */
  @Deprecated()
  public Observable<Void> setObservable(String key, String value) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    set(key, value, handler.toHandler());
    return handler;
  }

  /**
   * Set the string value of a key
   * @param key Key of which value to set
   * @param value New value for the key
   * @return 
   */
  public Single<Void> rxSet(String key, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      set(key, value, fut);
    }));
  }

  /**
   * Set the string value of a key
   * @param key Key of which value to set
   * @param value New value for the key
   * @param options Set options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient setWithOptions(String key, String value, SetOptions options, Handler<AsyncResult<String>> handler) { 
    delegate.setWithOptions(key, value, options, handler);
    return this;
  }

  /**
   * Set the string value of a key
   * @param key Key of which value to set
   * @param value New value for the key
   * @param options Set options
   * @return 
   * @deprecated use {@link #rxSetWithOptions} instead
   */
  @Deprecated()
  public Observable<String> setWithOptionsObservable(String key, String value, SetOptions options) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    setWithOptions(key, value, options, handler.toHandler());
    return handler;
  }

  /**
   * Set the string value of a key
   * @param key Key of which value to set
   * @param value New value for the key
   * @param options Set options
   * @return 
   */
  public Single<String> rxSetWithOptions(String key, String value, SetOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      setWithOptions(key, value, options, fut);
    }));
  }

  /**
   * Set the binary string value of a key - without encoding as utf-8
   * @param key Key of which value to set
   * @param value New value for the key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient setBinary(String key, Buffer value, Handler<AsyncResult<Void>> handler) { 
    delegate.setBinary(key, value.getDelegate(), handler);
    return this;
  }

  /**
   * Set the binary string value of a key - without encoding as utf-8
   * @param key Key of which value to set
   * @param value New value for the key
   * @return 
   * @deprecated use {@link #rxSetBinary} instead
   */
  @Deprecated()
  public Observable<Void> setBinaryObservable(String key, Buffer value) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    setBinary(key, value, handler.toHandler());
    return handler;
  }

  /**
   * Set the binary string value of a key - without encoding as utf-8
   * @param key Key of which value to set
   * @param value New value for the key
   * @return 
   */
  public Single<Void> rxSetBinary(String key, Buffer value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      setBinary(key, value, fut);
    }));
  }

  /**
   * Set the string value of a key
   * @param key Key of which value to set
   * @param value New value for the key
   * @param options Set options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient setBinaryWithOptions(String key, Buffer value, SetOptions options, Handler<AsyncResult<Void>> handler) { 
    delegate.setBinaryWithOptions(key, value.getDelegate(), options, handler);
    return this;
  }

  /**
   * Set the string value of a key
   * @param key Key of which value to set
   * @param value New value for the key
   * @param options Set options
   * @return 
   * @deprecated use {@link #rxSetBinaryWithOptions} instead
   */
  @Deprecated()
  public Observable<Void> setBinaryWithOptionsObservable(String key, Buffer value, SetOptions options) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    setBinaryWithOptions(key, value, options, handler.toHandler());
    return handler;
  }

  /**
   * Set the string value of a key
   * @param key Key of which value to set
   * @param value New value for the key
   * @param options Set options
   * @return 
   */
  public Single<Void> rxSetBinaryWithOptions(String key, Buffer value, SetOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      setBinaryWithOptions(key, value, options, fut);
    }));
  }

  /**
   * Sets or clears the bit at offset in the string value stored at key
   * @param key Key string
   * @param offset Bit offset
   * @param bit New value - must be 1 or 0
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient setbit(String key, long offset, int bit, Handler<AsyncResult<Long>> handler) { 
    delegate.setbit(key, offset, bit, handler);
    return this;
  }

  /**
   * Sets or clears the bit at offset in the string value stored at key
   * @param key Key string
   * @param offset Bit offset
   * @param bit New value - must be 1 or 0
   * @return 
   * @deprecated use {@link #rxSetbit} instead
   */
  @Deprecated()
  public Observable<Long> setbitObservable(String key, long offset, int bit) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    setbit(key, offset, bit, handler.toHandler());
    return handler;
  }

  /**
   * Sets or clears the bit at offset in the string value stored at key
   * @param key Key string
   * @param offset Bit offset
   * @param bit New value - must be 1 or 0
   * @return 
   */
  public Single<Long> rxSetbit(String key, long offset, int bit) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      setbit(key, offset, bit, fut);
    }));
  }

  /**
   * Set the value and expiration of a key
   * @param key Key string
   * @param seconds Number of seconds until the key expires
   * @param value New value for key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient setex(String key, long seconds, String value, Handler<AsyncResult<String>> handler) { 
    delegate.setex(key, seconds, value, handler);
    return this;
  }

  /**
   * Set the value and expiration of a key
   * @param key Key string
   * @param seconds Number of seconds until the key expires
   * @param value New value for key
   * @return 
   * @deprecated use {@link #rxSetex} instead
   */
  @Deprecated()
  public Observable<String> setexObservable(String key, long seconds, String value) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    setex(key, seconds, value, handler.toHandler());
    return handler;
  }

  /**
   * Set the value and expiration of a key
   * @param key Key string
   * @param seconds Number of seconds until the key expires
   * @param value New value for key
   * @return 
   */
  public Single<String> rxSetex(String key, long seconds, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      setex(key, seconds, value, fut);
    }));
  }

  /**
   * Set the value of a key, only if the key does not exist
   * @param key Key of which value to set
   * @param value New value for the key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient setnx(String key, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.setnx(key, value, handler);
    return this;
  }

  /**
   * Set the value of a key, only if the key does not exist
   * @param key Key of which value to set
   * @param value New value for the key
   * @return 
   * @deprecated use {@link #rxSetnx} instead
   */
  @Deprecated()
  public Observable<Long> setnxObservable(String key, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    setnx(key, value, handler.toHandler());
    return handler;
  }

  /**
   * Set the value of a key, only if the key does not exist
   * @param key Key of which value to set
   * @param value New value for the key
   * @return 
   */
  public Single<Long> rxSetnx(String key, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      setnx(key, value, fut);
    }));
  }

  /**
   * Overwrite part of a string at key starting at the specified offset
   * @param key Key string
   * @param offset Offset - the maximum offset that you can set is 2^29 -1 (536870911), as Redis Strings are limited to 512 megabytes
   * @param value Value to overwrite with
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient setrange(String key, int offset, String value, Handler<AsyncResult<Long>> handler) { 
    delegate.setrange(key, offset, value, handler);
    return this;
  }

  /**
   * Overwrite part of a string at key starting at the specified offset
   * @param key Key string
   * @param offset Offset - the maximum offset that you can set is 2^29 -1 (536870911), as Redis Strings are limited to 512 megabytes
   * @param value Value to overwrite with
   * @return 
   * @deprecated use {@link #rxSetrange} instead
   */
  @Deprecated()
  public Observable<Long> setrangeObservable(String key, int offset, String value) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    setrange(key, offset, value, handler.toHandler());
    return handler;
  }

  /**
   * Overwrite part of a string at key starting at the specified offset
   * @param key Key string
   * @param offset Offset - the maximum offset that you can set is 2^29 -1 (536870911), as Redis Strings are limited to 512 megabytes
   * @param value Value to overwrite with
   * @return 
   */
  public Single<Long> rxSetrange(String key, int offset, String value) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      setrange(key, offset, value, fut);
    }));
  }

  /**
   * Intersect multiple sets
   * @param keys List of keys to perform intersection on
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sinter(List<String> keys, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.sinter(keys, handler);
    return this;
  }

  /**
   * Intersect multiple sets
   * @param keys List of keys to perform intersection on
   * @return 
   * @deprecated use {@link #rxSinter} instead
   */
  @Deprecated()
  public Observable<JsonArray> sinterObservable(List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sinter(keys, handler.toHandler());
    return handler;
  }

  /**
   * Intersect multiple sets
   * @param keys List of keys to perform intersection on
   * @return 
   */
  public Single<JsonArray> rxSinter(List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sinter(keys, fut);
    }));
  }

  /**
   * Intersect multiple sets and store the resulting set in a key
   * @param destkey Key where to store the results
   * @param keys List of keys to perform intersection on
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sinterstore(String destkey, List<String> keys, Handler<AsyncResult<Long>> handler) { 
    delegate.sinterstore(destkey, keys, handler);
    return this;
  }

  /**
   * Intersect multiple sets and store the resulting set in a key
   * @param destkey Key where to store the results
   * @param keys List of keys to perform intersection on
   * @return 
   * @deprecated use {@link #rxSinterstore} instead
   */
  @Deprecated()
  public Observable<Long> sinterstoreObservable(String destkey, List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sinterstore(destkey, keys, handler.toHandler());
    return handler;
  }

  /**
   * Intersect multiple sets and store the resulting set in a key
   * @param destkey Key where to store the results
   * @param keys List of keys to perform intersection on
   * @return 
   */
  public Single<Long> rxSinterstore(String destkey, List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sinterstore(destkey, keys, fut);
    }));
  }

  /**
   * Determine if a given value is a member of a set
   * @param key Key string
   * @param member Member to look for
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sismember(String key, String member, Handler<AsyncResult<Long>> handler) { 
    delegate.sismember(key, member, handler);
    return this;
  }

  /**
   * Determine if a given value is a member of a set
   * @param key Key string
   * @param member Member to look for
   * @return 
   * @deprecated use {@link #rxSismember} instead
   */
  @Deprecated()
  public Observable<Long> sismemberObservable(String key, String member) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sismember(key, member, handler.toHandler());
    return handler;
  }

  /**
   * Determine if a given value is a member of a set
   * @param key Key string
   * @param member Member to look for
   * @return 
   */
  public Single<Long> rxSismember(String key, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sismember(key, member, fut);
    }));
  }

  /**
   * Make the server a slave of another instance
   * @param host Host to become this server's master
   * @param port Port of our new master
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient slaveof(String host, int port, Handler<AsyncResult<String>> handler) { 
    delegate.slaveof(host, port, handler);
    return this;
  }

  /**
   * Make the server a slave of another instance
   * @param host Host to become this server's master
   * @param port Port of our new master
   * @return 
   * @deprecated use {@link #rxSlaveof} instead
   */
  @Deprecated()
  public Observable<String> slaveofObservable(String host, int port) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    slaveof(host, port, handler.toHandler());
    return handler;
  }

  /**
   * Make the server a slave of another instance
   * @param host Host to become this server's master
   * @param port Port of our new master
   * @return 
   */
  public Single<String> rxSlaveof(String host, int port) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      slaveof(host, port, fut);
    }));
  }

  /**
   * Make this server a master
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient slaveofNoone(Handler<AsyncResult<String>> handler) { 
    delegate.slaveofNoone(handler);
    return this;
  }

  /**
   * Make this server a master
   * @return 
   * @deprecated use {@link #rxSlaveofNoone} instead
   */
  @Deprecated()
  public Observable<String> slaveofNooneObservable() { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    slaveofNoone(handler.toHandler());
    return handler;
  }

  /**
   * Make this server a master
   * @return 
   */
  public Single<String> rxSlaveofNoone() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      slaveofNoone(fut);
    }));
  }

  /**
   * Read the Redis slow queries log
   * @param limit Number of log entries to return. If value is less than zero all entries are returned
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient slowlogGet(int limit, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.slowlogGet(limit, handler);
    return this;
  }

  /**
   * Read the Redis slow queries log
   * @param limit Number of log entries to return. If value is less than zero all entries are returned
   * @return 
   * @deprecated use {@link #rxSlowlogGet} instead
   */
  @Deprecated()
  public Observable<JsonArray> slowlogGetObservable(int limit) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    slowlogGet(limit, handler.toHandler());
    return handler;
  }

  /**
   * Read the Redis slow queries log
   * @param limit Number of log entries to return. If value is less than zero all entries are returned
   * @return 
   */
  public Single<JsonArray> rxSlowlogGet(int limit) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      slowlogGet(limit, fut);
    }));
  }

  /**
   * Get the length of the Redis slow queries log
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient slowlogLen(Handler<AsyncResult<Long>> handler) { 
    delegate.slowlogLen(handler);
    return this;
  }

  /**
   * Get the length of the Redis slow queries log
   * @return 
   * @deprecated use {@link #rxSlowlogLen} instead
   */
  @Deprecated()
  public Observable<Long> slowlogLenObservable() { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    slowlogLen(handler.toHandler());
    return handler;
  }

  /**
   * Get the length of the Redis slow queries log
   * @return 
   */
  public Single<Long> rxSlowlogLen() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      slowlogLen(fut);
    }));
  }

  /**
   * Reset the Redis slow queries log
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient slowlogReset(Handler<AsyncResult<Void>> handler) { 
    delegate.slowlogReset(handler);
    return this;
  }

  /**
   * Reset the Redis slow queries log
   * @return 
   * @deprecated use {@link #rxSlowlogReset} instead
   */
  @Deprecated()
  public Observable<Void> slowlogResetObservable() { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    slowlogReset(handler.toHandler());
    return handler;
  }

  /**
   * Reset the Redis slow queries log
   * @return 
   */
  public Single<Void> rxSlowlogReset() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      slowlogReset(fut);
    }));
  }

  /**
   * Get all the members in a set
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient smembers(String key, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.smembers(key, handler);
    return this;
  }

  /**
   * Get all the members in a set
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxSmembers} instead
   */
  @Deprecated()
  public Observable<JsonArray> smembersObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    smembers(key, handler.toHandler());
    return handler;
  }

  /**
   * Get all the members in a set
   * @param key Key string
   * @return 
   */
  public Single<JsonArray> rxSmembers(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      smembers(key, fut);
    }));
  }

  /**
   * Move a member from one set to another
   * @param key Key of source set currently containing the member
   * @param destkey Key identifying the destination set
   * @param member Member to move
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient smove(String key, String destkey, String member, Handler<AsyncResult<Long>> handler) { 
    delegate.smove(key, destkey, member, handler);
    return this;
  }

  /**
   * Move a member from one set to another
   * @param key Key of source set currently containing the member
   * @param destkey Key identifying the destination set
   * @param member Member to move
   * @return 
   * @deprecated use {@link #rxSmove} instead
   */
  @Deprecated()
  public Observable<Long> smoveObservable(String key, String destkey, String member) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    smove(key, destkey, member, handler.toHandler());
    return handler;
  }

  /**
   * Move a member from one set to another
   * @param key Key of source set currently containing the member
   * @param destkey Key identifying the destination set
   * @param member Member to move
   * @return 
   */
  public Single<Long> rxSmove(String key, String destkey, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      smove(key, destkey, member, fut);
    }));
  }

  /**
   * Sort the elements in a list, set or sorted set
   * @param key Key string
   * @param options Sort options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sort(String key, SortOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.sort(key, options, handler);
    return this;
  }

  /**
   * Sort the elements in a list, set or sorted set
   * @param key Key string
   * @param options Sort options
   * @return 
   * @deprecated use {@link #rxSort} instead
   */
  @Deprecated()
  public Observable<JsonArray> sortObservable(String key, SortOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sort(key, options, handler.toHandler());
    return handler;
  }

  /**
   * Sort the elements in a list, set or sorted set
   * @param key Key string
   * @param options Sort options
   * @return 
   */
  public Single<JsonArray> rxSort(String key, SortOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sort(key, options, fut);
    }));
  }

  /**
   * Remove and return a random member from a set
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient spop(String key, Handler<AsyncResult<String>> handler) { 
    delegate.spop(key, handler);
    return this;
  }

  /**
   * Remove and return a random member from a set
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxSpop} instead
   */
  @Deprecated()
  public Observable<String> spopObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    spop(key, handler.toHandler());
    return handler;
  }

  /**
   * Remove and return a random member from a set
   * @param key Key string
   * @return 
   */
  public Single<String> rxSpop(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      spop(key, fut);
    }));
  }

  /**
   * Remove and return random members from a set
   * @param key Key string
   * @param count Number of members to remove
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient spopMany(String key, int count, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.spopMany(key, count, handler);
    return this;
  }

  /**
   * Remove and return random members from a set
   * @param key Key string
   * @param count Number of members to remove
   * @return 
   * @deprecated use {@link #rxSpopMany} instead
   */
  @Deprecated()
  public Observable<JsonArray> spopManyObservable(String key, int count) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    spopMany(key, count, handler.toHandler());
    return handler;
  }

  /**
   * Remove and return random members from a set
   * @param key Key string
   * @param count Number of members to remove
   * @return 
   */
  public Single<JsonArray> rxSpopMany(String key, int count) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      spopMany(key, count, fut);
    }));
  }

  /**
   * Get one or multiple random members from a set
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient srandmember(String key, Handler<AsyncResult<String>> handler) { 
    delegate.srandmember(key, handler);
    return this;
  }

  /**
   * Get one or multiple random members from a set
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxSrandmember} instead
   */
  @Deprecated()
  public Observable<String> srandmemberObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    srandmember(key, handler.toHandler());
    return handler;
  }

  /**
   * Get one or multiple random members from a set
   * @param key Key string
   * @return 
   */
  public Single<String> rxSrandmember(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      srandmember(key, fut);
    }));
  }

  /**
   * Get one or multiple random members from a set
   * @param key Key string
   * @param count Number of members to get
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient srandmemberCount(String key, int count, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.srandmemberCount(key, count, handler);
    return this;
  }

  /**
   * Get one or multiple random members from a set
   * @param key Key string
   * @param count Number of members to get
   * @return 
   * @deprecated use {@link #rxSrandmemberCount} instead
   */
  @Deprecated()
  public Observable<JsonArray> srandmemberCountObservable(String key, int count) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    srandmemberCount(key, count, handler.toHandler());
    return handler;
  }

  /**
   * Get one or multiple random members from a set
   * @param key Key string
   * @param count Number of members to get
   * @return 
   */
  public Single<JsonArray> rxSrandmemberCount(String key, int count) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      srandmemberCount(key, count, fut);
    }));
  }

  /**
   * Remove one member from a set
   * @param key Key string
   * @param member Member to remove
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient srem(String key, String member, Handler<AsyncResult<Long>> handler) { 
    delegate.srem(key, member, handler);
    return this;
  }

  /**
   * Remove one member from a set
   * @param key Key string
   * @param member Member to remove
   * @return 
   * @deprecated use {@link #rxSrem} instead
   */
  @Deprecated()
  public Observable<Long> sremObservable(String key, String member) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    srem(key, member, handler.toHandler());
    return handler;
  }

  /**
   * Remove one member from a set
   * @param key Key string
   * @param member Member to remove
   * @return 
   */
  public Single<Long> rxSrem(String key, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      srem(key, member, fut);
    }));
  }

  /**
   * Remove one or more members from a set
   * @param key Key string
   * @param members Members to remove
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sremMany(String key, List<String> members, Handler<AsyncResult<Long>> handler) { 
    delegate.sremMany(key, members, handler);
    return this;
  }

  /**
   * Remove one or more members from a set
   * @param key Key string
   * @param members Members to remove
   * @return 
   * @deprecated use {@link #rxSremMany} instead
   */
  @Deprecated()
  public Observable<Long> sremManyObservable(String key, List<String> members) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sremMany(key, members, handler.toHandler());
    return handler;
  }

  /**
   * Remove one or more members from a set
   * @param key Key string
   * @param members Members to remove
   * @return 
   */
  public Single<Long> rxSremMany(String key, List<String> members) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sremMany(key, members, fut);
    }));
  }

  /**
   * Get the length of the value stored in a key
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient strlen(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.strlen(key, handler);
    return this;
  }

  /**
   * Get the length of the value stored in a key
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxStrlen} instead
   */
  @Deprecated()
  public Observable<Long> strlenObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    strlen(key, handler.toHandler());
    return handler;
  }

  /**
   * Get the length of the value stored in a key
   * @param key Key string
   * @return 
   */
  public Single<Long> rxStrlen(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      strlen(key, fut);
    }));
  }

  /**
   * Listen for messages published to the given channels
   * @param channel Channel to subscribe to
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient subscribe(String channel, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.subscribe(channel, handler);
    return this;
  }

  /**
   * Listen for messages published to the given channels
   * @param channel Channel to subscribe to
   * @return 
   * @deprecated use {@link #rxSubscribe} instead
   */
  @Deprecated()
  public Observable<JsonArray> subscribeObservable(String channel) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    subscribe(channel, handler.toHandler());
    return handler;
  }

  /**
   * Listen for messages published to the given channels
   * @param channel Channel to subscribe to
   * @return 
   */
  public Single<JsonArray> rxSubscribe(String channel) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      subscribe(channel, fut);
    }));
  }

  /**
   * Listen for messages published to the given channels
   * @param channels List of channels to subscribe to
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient subscribeMany(List<String> channels, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.subscribeMany(channels, handler);
    return this;
  }

  /**
   * Listen for messages published to the given channels
   * @param channels List of channels to subscribe to
   * @return 
   * @deprecated use {@link #rxSubscribeMany} instead
   */
  @Deprecated()
  public Observable<JsonArray> subscribeManyObservable(List<String> channels) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    subscribeMany(channels, handler.toHandler());
    return handler;
  }

  /**
   * Listen for messages published to the given channels
   * @param channels List of channels to subscribe to
   * @return 
   */
  public Single<JsonArray> rxSubscribeMany(List<String> channels) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      subscribeMany(channels, fut);
    }));
  }

  /**
   * Add multiple sets
   * @param keys List of keys identifying sets to add up
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sunion(List<String> keys, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.sunion(keys, handler);
    return this;
  }

  /**
   * Add multiple sets
   * @param keys List of keys identifying sets to add up
   * @return 
   * @deprecated use {@link #rxSunion} instead
   */
  @Deprecated()
  public Observable<JsonArray> sunionObservable(List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sunion(keys, handler.toHandler());
    return handler;
  }

  /**
   * Add multiple sets
   * @param keys List of keys identifying sets to add up
   * @return 
   */
  public Single<JsonArray> rxSunion(List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sunion(keys, fut);
    }));
  }

  /**
   * Add multiple sets and store the resulting set in a key
   * @param destkey Destination key
   * @param keys List of keys identifying sets to add up
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sunionstore(String destkey, List<String> keys, Handler<AsyncResult<Long>> handler) { 
    delegate.sunionstore(destkey, keys, handler);
    return this;
  }

  /**
   * Add multiple sets and store the resulting set in a key
   * @param destkey Destination key
   * @param keys List of keys identifying sets to add up
   * @return 
   * @deprecated use {@link #rxSunionstore} instead
   */
  @Deprecated()
  public Observable<Long> sunionstoreObservable(String destkey, List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sunionstore(destkey, keys, handler.toHandler());
    return handler;
  }

  /**
   * Add multiple sets and store the resulting set in a key
   * @param destkey Destination key
   * @param keys List of keys identifying sets to add up
   * @return 
   */
  public Single<Long> rxSunionstore(String destkey, List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sunionstore(destkey, keys, fut);
    }));
  }

  /**
   * Internal command used for replication
   * @param handler 
   * @return 
   */
  public RedisClient sync(Handler<AsyncResult<Void>> handler) { 
    delegate.sync(handler);
    return this;
  }

  /**
   * Internal command used for replication
   * @return 
   * @deprecated use {@link #rxSync} instead
   */
  @Deprecated()
  public Observable<Void> syncObservable() { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sync(handler.toHandler());
    return handler;
  }

  /**
   * Internal command used for replication
   * @return 
   */
  public Single<Void> rxSync() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sync(fut);
    }));
  }

  /**
   * Return the current server time
   * @param handler 
   * @return 
   */
  public RedisClient time(Handler<AsyncResult<JsonArray>> handler) { 
    delegate.time(handler);
    return this;
  }

  /**
   * Return the current server time
   * @return 
   * @deprecated use {@link #rxTime} instead
   */
  @Deprecated()
  public Observable<JsonArray> timeObservable() { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    time(handler.toHandler());
    return handler;
  }

  /**
   * Return the current server time
   * @return 
   */
  public Single<JsonArray> rxTime() { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      time(fut);
    }));
  }

  /**
   * Return a RedisTransaction instance
   * @return transaction instance
   */
  public RedisTransaction transaction() { 
    RedisTransaction ret = RedisTransaction.newInstance(delegate.transaction());
    return ret;
  }

  /**
   * Get the time to live for a key
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient ttl(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.ttl(key, handler);
    return this;
  }

  /**
   * Get the time to live for a key
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxTtl} instead
   */
  @Deprecated()
  public Observable<Long> ttlObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    ttl(key, handler.toHandler());
    return handler;
  }

  /**
   * Get the time to live for a key
   * @param key Key string
   * @return 
   */
  public Single<Long> rxTtl(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      ttl(key, fut);
    }));
  }

  /**
   * Determine the type stored at key
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient type(String key, Handler<AsyncResult<String>> handler) { 
    delegate.type(key, handler);
    return this;
  }

  /**
   * Determine the type stored at key
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxType} instead
   */
  @Deprecated()
  public Observable<String> typeObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    type(key, handler.toHandler());
    return handler;
  }

  /**
   * Determine the type stored at key
   * @param key Key string
   * @return 
   */
  public Single<String> rxType(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      type(key, fut);
    }));
  }

  /**
   * Stop listening for messages posted to the given channels
   * @param channels List of channels to subscribe to
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient unsubscribe(List<String> channels, Handler<AsyncResult<Void>> handler) { 
    delegate.unsubscribe(channels, handler);
    return this;
  }

  /**
   * Stop listening for messages posted to the given channels
   * @param channels List of channels to subscribe to
   * @return 
   * @deprecated use {@link #rxUnsubscribe} instead
   */
  @Deprecated()
  public Observable<Void> unsubscribeObservable(List<String> channels) { 
    io.vertx.rx.java.ObservableFuture<Void> handler = io.vertx.rx.java.RxHelper.observableFuture();
    unsubscribe(channels, handler.toHandler());
    return handler;
  }

  /**
   * Stop listening for messages posted to the given channels
   * @param channels List of channels to subscribe to
   * @return 
   */
  public Single<Void> rxUnsubscribe(List<String> channels) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      unsubscribe(channels, fut);
    }));
  }

  /**
   * Wait for the synchronous replication of all the write commands sent in the context of the current connection.
   * @param numSlaves 
   * @param timeout 
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient wait(long numSlaves, long timeout, Handler<AsyncResult<String>> handler) { 
    delegate.wait(numSlaves, timeout, handler);
    return this;
  }

  /**
   * Wait for the synchronous replication of all the write commands sent in the context of the current connection.
   * @param numSlaves 
   * @param timeout 
   * @return 
   * @deprecated use {@link #rxWait} instead
   */
  @Deprecated()
  public Observable<String> waitObservable(long numSlaves, long timeout) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    wait(numSlaves, timeout, handler.toHandler());
    return handler;
  }

  /**
   * Wait for the synchronous replication of all the write commands sent in the context of the current connection.
   * @param numSlaves 
   * @param timeout 
   * @return 
   */
  public Single<String> rxWait(long numSlaves, long timeout) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      wait(numSlaves, timeout, fut);
    }));
  }

  /**
   * Add one or more members to a sorted set, or update its score if it already exists
   * @param key Key string
   * @param score Score used for sorting
   * @param member New member key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zadd(String key, double score, String member, Handler<AsyncResult<Long>> handler) { 
    delegate.zadd(key, score, member, handler);
    return this;
  }

  /**
   * Add one or more members to a sorted set, or update its score if it already exists
   * @param key Key string
   * @param score Score used for sorting
   * @param member New member key
   * @return 
   * @deprecated use {@link #rxZadd} instead
   */
  @Deprecated()
  public Observable<Long> zaddObservable(String key, double score, String member) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zadd(key, score, member, handler.toHandler());
    return handler;
  }

  /**
   * Add one or more members to a sorted set, or update its score if it already exists
   * @param key Key string
   * @param score Score used for sorting
   * @param member New member key
   * @return 
   */
  public Single<Long> rxZadd(String key, double score, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zadd(key, score, member, fut);
    }));
  }

  /**
   * Add one or more members to a sorted set, or update its score if it already exists
   * @param key Key string
   * @param members New member keys and their scores
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zaddMany(String key, Map<String,Double> members, Handler<AsyncResult<Long>> handler) { 
    delegate.zaddMany(key, members, handler);
    return this;
  }

  /**
   * Add one or more members to a sorted set, or update its score if it already exists
   * @param key Key string
   * @param members New member keys and their scores
   * @return 
   * @deprecated use {@link #rxZaddMany} instead
   */
  @Deprecated()
  public Observable<Long> zaddManyObservable(String key, Map<String,Double> members) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zaddMany(key, members, handler.toHandler());
    return handler;
  }

  /**
   * Add one or more members to a sorted set, or update its score if it already exists
   * @param key Key string
   * @param members New member keys and their scores
   * @return 
   */
  public Single<Long> rxZaddMany(String key, Map<String,Double> members) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zaddMany(key, members, fut);
    }));
  }

  /**
   * Get the number of members in a sorted set
   * @param key Key string
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zcard(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.zcard(key, handler);
    return this;
  }

  /**
   * Get the number of members in a sorted set
   * @param key Key string
   * @return 
   * @deprecated use {@link #rxZcard} instead
   */
  @Deprecated()
  public Observable<Long> zcardObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zcard(key, handler.toHandler());
    return handler;
  }

  /**
   * Get the number of members in a sorted set
   * @param key Key string
   * @return 
   */
  public Single<Long> rxZcard(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zcard(key, fut);
    }));
  }

  /**
   * Count the members in a sorted set with scores within the given values
   * @param key Key string
   * @param min Minimum score
   * @param max Maximum score
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zcount(String key, double min, double max, Handler<AsyncResult<Long>> handler) { 
    delegate.zcount(key, min, max, handler);
    return this;
  }

  /**
   * Count the members in a sorted set with scores within the given values
   * @param key Key string
   * @param min Minimum score
   * @param max Maximum score
   * @return 
   * @deprecated use {@link #rxZcount} instead
   */
  @Deprecated()
  public Observable<Long> zcountObservable(String key, double min, double max) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zcount(key, min, max, handler.toHandler());
    return handler;
  }

  /**
   * Count the members in a sorted set with scores within the given values
   * @param key Key string
   * @param min Minimum score
   * @param max Maximum score
   * @return 
   */
  public Single<Long> rxZcount(String key, double min, double max) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zcount(key, min, max, fut);
    }));
  }

  /**
   * Increment the score of a member in a sorted set
   * @param key Key string
   * @param increment Increment amount
   * @param member Member key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zincrby(String key, double increment, String member, Handler<AsyncResult<String>> handler) { 
    delegate.zincrby(key, increment, member, handler);
    return this;
  }

  /**
   * Increment the score of a member in a sorted set
   * @param key Key string
   * @param increment Increment amount
   * @param member Member key
   * @return 
   * @deprecated use {@link #rxZincrby} instead
   */
  @Deprecated()
  public Observable<String> zincrbyObservable(String key, double increment, String member) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zincrby(key, increment, member, handler.toHandler());
    return handler;
  }

  /**
   * Increment the score of a member in a sorted set
   * @param key Key string
   * @param increment Increment amount
   * @param member Member key
   * @return 
   */
  public Single<String> rxZincrby(String key, double increment, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zincrby(key, increment, member, fut);
    }));
  }

  /**
   * Intersect multiple sorted sets and store the resulting sorted set in a new key
   * @param destkey Destination key
   * @param sets List of keys identifying sorted sets to intersect
   * @param options Aggregation options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zinterstore(String destkey, List<String> sets, AggregateOptions options, Handler<AsyncResult<Long>> handler) { 
    delegate.zinterstore(destkey, sets, options, handler);
    return this;
  }

  /**
   * Intersect multiple sorted sets and store the resulting sorted set in a new key
   * @param destkey Destination key
   * @param sets List of keys identifying sorted sets to intersect
   * @param options Aggregation options
   * @return 
   * @deprecated use {@link #rxZinterstore} instead
   */
  @Deprecated()
  public Observable<Long> zinterstoreObservable(String destkey, List<String> sets, AggregateOptions options) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zinterstore(destkey, sets, options, handler.toHandler());
    return handler;
  }

  /**
   * Intersect multiple sorted sets and store the resulting sorted set in a new key
   * @param destkey Destination key
   * @param sets List of keys identifying sorted sets to intersect
   * @param options Aggregation options
   * @return 
   */
  public Single<Long> rxZinterstore(String destkey, List<String> sets, AggregateOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zinterstore(destkey, sets, options, fut);
    }));
  }

  /**
   * Intersect multiple sorted sets and store the resulting sorted set in a new key using weights for scoring
   * @param destkey Destination key
   * @param sets List of keys identifying sorted sets to intersect
   * @param options Aggregation options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zinterstoreWeighed(String destkey, Map<String,Double> sets, AggregateOptions options, Handler<AsyncResult<Long>> handler) { 
    delegate.zinterstoreWeighed(destkey, sets, options, handler);
    return this;
  }

  /**
   * Intersect multiple sorted sets and store the resulting sorted set in a new key using weights for scoring
   * @param destkey Destination key
   * @param sets List of keys identifying sorted sets to intersect
   * @param options Aggregation options
   * @return 
   * @deprecated use {@link #rxZinterstoreWeighed} instead
   */
  @Deprecated()
  public Observable<Long> zinterstoreWeighedObservable(String destkey, Map<String,Double> sets, AggregateOptions options) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zinterstoreWeighed(destkey, sets, options, handler.toHandler());
    return handler;
  }

  /**
   * Intersect multiple sorted sets and store the resulting sorted set in a new key using weights for scoring
   * @param destkey Destination key
   * @param sets List of keys identifying sorted sets to intersect
   * @param options Aggregation options
   * @return 
   */
  public Single<Long> rxZinterstoreWeighed(String destkey, Map<String,Double> sets, AggregateOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zinterstoreWeighed(destkey, sets, options, fut);
    }));
  }

  /**
   * Count the number of members in a sorted set between a given lexicographical range
   * @param key Key string
   * @param min Pattern to compare against for minimum value
   * @param max Pattern to compare against for maximum value
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zlexcount(String key, String min, String max, Handler<AsyncResult<Long>> handler) { 
    delegate.zlexcount(key, min, max, handler);
    return this;
  }

  /**
   * Count the number of members in a sorted set between a given lexicographical range
   * @param key Key string
   * @param min Pattern to compare against for minimum value
   * @param max Pattern to compare against for maximum value
   * @return 
   * @deprecated use {@link #rxZlexcount} instead
   */
  @Deprecated()
  public Observable<Long> zlexcountObservable(String key, String min, String max) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zlexcount(key, min, max, handler.toHandler());
    return handler;
  }

  /**
   * Count the number of members in a sorted set between a given lexicographical range
   * @param key Key string
   * @param min Pattern to compare against for minimum value
   * @param max Pattern to compare against for maximum value
   * @return 
   */
  public Single<Long> rxZlexcount(String key, String min, String max) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zlexcount(key, min, max, fut);
    }));
  }

  /**
   * Return a range of members in a sorted set, by index
   * @param key Key string
   * @param start Start index for the range
   * @param stop Stop index for the range - inclusive
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zrange(String key, long start, long stop, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.zrange(key, start, stop, handler);
    return this;
  }

  /**
   * Return a range of members in a sorted set, by index
   * @param key Key string
   * @param start Start index for the range
   * @param stop Stop index for the range - inclusive
   * @return 
   * @deprecated use {@link #rxZrange} instead
   */
  @Deprecated()
  public Observable<JsonArray> zrangeObservable(String key, long start, long stop) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zrange(key, start, stop, handler.toHandler());
    return handler;
  }

  /**
   * Return a range of members in a sorted set, by index
   * @param key Key string
   * @param start Start index for the range
   * @param stop Stop index for the range - inclusive
   * @return 
   */
  public Single<JsonArray> rxZrange(String key, long start, long stop) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zrange(key, start, stop, fut);
    }));
  }

  /**
   * Return a range of members in a sorted set, by index
   * @param key Key string
   * @param start Start index for the range
   * @param stop Stop index for the range - inclusive
   * @param options Range options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zrangeWithOptions(String key, long start, long stop, RangeOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.zrangeWithOptions(key, start, stop, options, handler);
    return this;
  }

  /**
   * Return a range of members in a sorted set, by index
   * @param key Key string
   * @param start Start index for the range
   * @param stop Stop index for the range - inclusive
   * @param options Range options
   * @return 
   * @deprecated use {@link #rxZrangeWithOptions} instead
   */
  @Deprecated()
  public Observable<JsonArray> zrangeWithOptionsObservable(String key, long start, long stop, RangeOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zrangeWithOptions(key, start, stop, options, handler.toHandler());
    return handler;
  }

  /**
   * Return a range of members in a sorted set, by index
   * @param key Key string
   * @param start Start index for the range
   * @param stop Stop index for the range - inclusive
   * @param options Range options
   * @return 
   */
  public Single<JsonArray> rxZrangeWithOptions(String key, long start, long stop, RangeOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zrangeWithOptions(key, start, stop, options, fut);
    }));
  }

  /**
   * Return a range of members in a sorted set, by lexicographical range
   * @param key Key string
   * @param min Pattern representing a minimum allowed value
   * @param max Pattern representing a maximum allowed value
   * @param options Limit options where limit can be specified
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zrangebylex(String key, String min, String max, LimitOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.zrangebylex(key, min, max, options, handler);
    return this;
  }

  /**
   * Return a range of members in a sorted set, by lexicographical range
   * @param key Key string
   * @param min Pattern representing a minimum allowed value
   * @param max Pattern representing a maximum allowed value
   * @param options Limit options where limit can be specified
   * @return 
   * @deprecated use {@link #rxZrangebylex} instead
   */
  @Deprecated()
  public Observable<JsonArray> zrangebylexObservable(String key, String min, String max, LimitOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zrangebylex(key, min, max, options, handler.toHandler());
    return handler;
  }

  /**
   * Return a range of members in a sorted set, by lexicographical range
   * @param key Key string
   * @param min Pattern representing a minimum allowed value
   * @param max Pattern representing a maximum allowed value
   * @param options Limit options where limit can be specified
   * @return 
   */
  public Single<JsonArray> rxZrangebylex(String key, String min, String max, LimitOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zrangebylex(key, min, max, options, fut);
    }));
  }

  /**
   * Return a range of members in a sorted set, by score
   * @param key Key string
   * @param min Pattern defining a minimum value
   * @param max Pattern defining a maximum value
   * @param options Range and limit options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zrangebyscore(String key, String min, String max, RangeLimitOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.zrangebyscore(key, min, max, options, handler);
    return this;
  }

  /**
   * Return a range of members in a sorted set, by score
   * @param key Key string
   * @param min Pattern defining a minimum value
   * @param max Pattern defining a maximum value
   * @param options Range and limit options
   * @return 
   * @deprecated use {@link #rxZrangebyscore} instead
   */
  @Deprecated()
  public Observable<JsonArray> zrangebyscoreObservable(String key, String min, String max, RangeLimitOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zrangebyscore(key, min, max, options, handler.toHandler());
    return handler;
  }

  /**
   * Return a range of members in a sorted set, by score
   * @param key Key string
   * @param min Pattern defining a minimum value
   * @param max Pattern defining a maximum value
   * @param options Range and limit options
   * @return 
   */
  public Single<JsonArray> rxZrangebyscore(String key, String min, String max, RangeLimitOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zrangebyscore(key, min, max, options, fut);
    }));
  }

  /**
   * Determine the index of a member in a sorted set
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zrank(String key, String member, Handler<AsyncResult<Long>> handler) { 
    delegate.zrank(key, member, handler);
    return this;
  }

  /**
   * Determine the index of a member in a sorted set
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @return 
   * @deprecated use {@link #rxZrank} instead
   */
  @Deprecated()
  public Observable<Long> zrankObservable(String key, String member) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zrank(key, member, handler.toHandler());
    return handler;
  }

  /**
   * Determine the index of a member in a sorted set
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @return 
   */
  public Single<Long> rxZrank(String key, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zrank(key, member, fut);
    }));
  }

  /**
   * Remove one member from a sorted set
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zrem(String key, String member, Handler<AsyncResult<Long>> handler) { 
    delegate.zrem(key, member, handler);
    return this;
  }

  /**
   * Remove one member from a sorted set
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @return 
   * @deprecated use {@link #rxZrem} instead
   */
  @Deprecated()
  public Observable<Long> zremObservable(String key, String member) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zrem(key, member, handler.toHandler());
    return handler;
  }

  /**
   * Remove one member from a sorted set
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @return 
   */
  public Single<Long> rxZrem(String key, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zrem(key, member, fut);
    }));
  }

  /**
   * Remove one or more members from a sorted set
   * @param key Key string
   * @param members Members in the sorted set identified by key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zremMany(String key, List<String> members, Handler<AsyncResult<Long>> handler) { 
    delegate.zremMany(key, members, handler);
    return this;
  }

  /**
   * Remove one or more members from a sorted set
   * @param key Key string
   * @param members Members in the sorted set identified by key
   * @return 
   * @deprecated use {@link #rxZremMany} instead
   */
  @Deprecated()
  public Observable<Long> zremManyObservable(String key, List<String> members) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zremMany(key, members, handler.toHandler());
    return handler;
  }

  /**
   * Remove one or more members from a sorted set
   * @param key Key string
   * @param members Members in the sorted set identified by key
   * @return 
   */
  public Single<Long> rxZremMany(String key, List<String> members) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zremMany(key, members, fut);
    }));
  }

  /**
   * Remove all members in a sorted set between the given lexicographical range
   * @param key Key string
   * @param min Pattern defining a minimum value
   * @param max Pattern defining a maximum value
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zremrangebylex(String key, String min, String max, Handler<AsyncResult<Long>> handler) { 
    delegate.zremrangebylex(key, min, max, handler);
    return this;
  }

  /**
   * Remove all members in a sorted set between the given lexicographical range
   * @param key Key string
   * @param min Pattern defining a minimum value
   * @param max Pattern defining a maximum value
   * @return 
   * @deprecated use {@link #rxZremrangebylex} instead
   */
  @Deprecated()
  public Observable<Long> zremrangebylexObservable(String key, String min, String max) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zremrangebylex(key, min, max, handler.toHandler());
    return handler;
  }

  /**
   * Remove all members in a sorted set between the given lexicographical range
   * @param key Key string
   * @param min Pattern defining a minimum value
   * @param max Pattern defining a maximum value
   * @return 
   */
  public Single<Long> rxZremrangebylex(String key, String min, String max) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zremrangebylex(key, min, max, fut);
    }));
  }

  /**
   * Remove all members in a sorted set within the given indexes
   * @param key Key string
   * @param start Start index
   * @param stop Stop index
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zremrangebyrank(String key, long start, long stop, Handler<AsyncResult<Long>> handler) { 
    delegate.zremrangebyrank(key, start, stop, handler);
    return this;
  }

  /**
   * Remove all members in a sorted set within the given indexes
   * @param key Key string
   * @param start Start index
   * @param stop Stop index
   * @return 
   * @deprecated use {@link #rxZremrangebyrank} instead
   */
  @Deprecated()
  public Observable<Long> zremrangebyrankObservable(String key, long start, long stop) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zremrangebyrank(key, start, stop, handler.toHandler());
    return handler;
  }

  /**
   * Remove all members in a sorted set within the given indexes
   * @param key Key string
   * @param start Start index
   * @param stop Stop index
   * @return 
   */
  public Single<Long> rxZremrangebyrank(String key, long start, long stop) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zremrangebyrank(key, start, stop, fut);
    }));
  }

  /**
   * Remove all members in a sorted set within the given scores
   * @param key Key string
   * @param min Pattern defining a minimum value
   * @param max Pattern defining a maximum value
   * @param handler 
   * @return 
   */
  public RedisClient zremrangebyscore(String key, String min, String max, Handler<AsyncResult<Long>> handler) { 
    delegate.zremrangebyscore(key, min, max, handler);
    return this;
  }

  /**
   * Remove all members in a sorted set within the given scores
   * @param key Key string
   * @param min Pattern defining a minimum value
   * @param max Pattern defining a maximum value
   * @return 
   * @deprecated use {@link #rxZremrangebyscore} instead
   */
  @Deprecated()
  public Observable<Long> zremrangebyscoreObservable(String key, String min, String max) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zremrangebyscore(key, min, max, handler.toHandler());
    return handler;
  }

  /**
   * Remove all members in a sorted set within the given scores
   * @param key Key string
   * @param min Pattern defining a minimum value
   * @param max Pattern defining a maximum value
   * @return 
   */
  public Single<Long> rxZremrangebyscore(String key, String min, String max) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zremrangebyscore(key, min, max, fut);
    }));
  }

  /**
   * Return a range of members in a sorted set, by index, with scores ordered from high to low
   * @param key Key string
   * @param start Start index for the range
   * @param stop Stop index for the range - inclusive
   * @param options Range options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zrevrange(String key, long start, long stop, RangeOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.zrevrange(key, start, stop, options, handler);
    return this;
  }

  /**
   * Return a range of members in a sorted set, by index, with scores ordered from high to low
   * @param key Key string
   * @param start Start index for the range
   * @param stop Stop index for the range - inclusive
   * @param options Range options
   * @return 
   * @deprecated use {@link #rxZrevrange} instead
   */
  @Deprecated()
  public Observable<JsonArray> zrevrangeObservable(String key, long start, long stop, RangeOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zrevrange(key, start, stop, options, handler.toHandler());
    return handler;
  }

  /**
   * Return a range of members in a sorted set, by index, with scores ordered from high to low
   * @param key Key string
   * @param start Start index for the range
   * @param stop Stop index for the range - inclusive
   * @param options Range options
   * @return 
   */
  public Single<JsonArray> rxZrevrange(String key, long start, long stop, RangeOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zrevrange(key, start, stop, options, fut);
    }));
  }

  /**
   * Return a range of members in a sorted set, by score, between the given lexicographical range with scores ordered from high to low
   * @param key Key string
   * @param max Pattern defining a maximum value
   * @param min Pattern defining a minimum value
   * @param options Limit options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zrevrangebylex(String key, String max, String min, LimitOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.zrevrangebylex(key, max, min, options, handler);
    return this;
  }

  /**
   * Return a range of members in a sorted set, by score, between the given lexicographical range with scores ordered from high to low
   * @param key Key string
   * @param max Pattern defining a maximum value
   * @param min Pattern defining a minimum value
   * @param options Limit options
   * @return 
   * @deprecated use {@link #rxZrevrangebylex} instead
   */
  @Deprecated()
  public Observable<JsonArray> zrevrangebylexObservable(String key, String max, String min, LimitOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zrevrangebylex(key, max, min, options, handler.toHandler());
    return handler;
  }

  /**
   * Return a range of members in a sorted set, by score, between the given lexicographical range with scores ordered from high to low
   * @param key Key string
   * @param max Pattern defining a maximum value
   * @param min Pattern defining a minimum value
   * @param options Limit options
   * @return 
   */
  public Single<JsonArray> rxZrevrangebylex(String key, String max, String min, LimitOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zrevrangebylex(key, max, min, options, fut);
    }));
  }

  /**
   * Return a range of members in a sorted set, by score, with scores ordered from high to low
   * @param key Key string
   * @param max Pattern defining a maximum value
   * @param min Pattern defining a minimum value
   * @param options Range and limit options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zrevrangebyscore(String key, String max, String min, RangeLimitOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.zrevrangebyscore(key, max, min, options, handler);
    return this;
  }

  /**
   * Return a range of members in a sorted set, by score, with scores ordered from high to low
   * @param key Key string
   * @param max Pattern defining a maximum value
   * @param min Pattern defining a minimum value
   * @param options Range and limit options
   * @return 
   * @deprecated use {@link #rxZrevrangebyscore} instead
   */
  @Deprecated()
  public Observable<JsonArray> zrevrangebyscoreObservable(String key, String max, String min, RangeLimitOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zrevrangebyscore(key, max, min, options, handler.toHandler());
    return handler;
  }

  /**
   * Return a range of members in a sorted set, by score, with scores ordered from high to low
   * @param key Key string
   * @param max Pattern defining a maximum value
   * @param min Pattern defining a minimum value
   * @param options Range and limit options
   * @return 
   */
  public Single<JsonArray> rxZrevrangebyscore(String key, String max, String min, RangeLimitOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zrevrangebyscore(key, max, min, options, fut);
    }));
  }

  /**
   * Determine the index of a member in a sorted set, with scores ordered from high to low
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zrevrank(String key, String member, Handler<AsyncResult<Long>> handler) { 
    delegate.zrevrank(key, member, handler);
    return this;
  }

  /**
   * Determine the index of a member in a sorted set, with scores ordered from high to low
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @return 
   * @deprecated use {@link #rxZrevrank} instead
   */
  @Deprecated()
  public Observable<Long> zrevrankObservable(String key, String member) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zrevrank(key, member, handler.toHandler());
    return handler;
  }

  /**
   * Determine the index of a member in a sorted set, with scores ordered from high to low
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @return 
   */
  public Single<Long> rxZrevrank(String key, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zrevrank(key, member, fut);
    }));
  }

  /**
   * Get the score associated with the given member in a sorted set
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zscore(String key, String member, Handler<AsyncResult<String>> handler) { 
    delegate.zscore(key, member, handler);
    return this;
  }

  /**
   * Get the score associated with the given member in a sorted set
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @return 
   * @deprecated use {@link #rxZscore} instead
   */
  @Deprecated()
  public Observable<String> zscoreObservable(String key, String member) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zscore(key, member, handler.toHandler());
    return handler;
  }

  /**
   * Get the score associated with the given member in a sorted set
   * @param key Key string
   * @param member Member in the sorted set identified by key
   * @return 
   */
  public Single<String> rxZscore(String key, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zscore(key, member, fut);
    }));
  }

  /**
   * Add multiple sorted sets and store the resulting sorted set in a new key
   * @param destkey Destination key
   * @param sets List of keys identifying sorted sets
   * @param options Aggregation options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zunionstore(String destkey, List<String> sets, AggregateOptions options, Handler<AsyncResult<Long>> handler) { 
    delegate.zunionstore(destkey, sets, options, handler);
    return this;
  }

  /**
   * Add multiple sorted sets and store the resulting sorted set in a new key
   * @param destkey Destination key
   * @param sets List of keys identifying sorted sets
   * @param options Aggregation options
   * @return 
   * @deprecated use {@link #rxZunionstore} instead
   */
  @Deprecated()
  public Observable<Long> zunionstoreObservable(String destkey, List<String> sets, AggregateOptions options) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zunionstore(destkey, sets, options, handler.toHandler());
    return handler;
  }

  /**
   * Add multiple sorted sets and store the resulting sorted set in a new key
   * @param destkey Destination key
   * @param sets List of keys identifying sorted sets
   * @param options Aggregation options
   * @return 
   */
  public Single<Long> rxZunionstore(String destkey, List<String> sets, AggregateOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zunionstore(destkey, sets, options, fut);
    }));
  }

  /**
   * Add multiple sorted sets using weights, and store the resulting sorted set in a new key
   * @param key Destination key
   * @param sets Map containing set-key:weight pairs
   * @param options Aggregation options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zunionstoreWeighed(String key, Map<String,Double> sets, AggregateOptions options, Handler<AsyncResult<Long>> handler) { 
    delegate.zunionstoreWeighed(key, sets, options, handler);
    return this;
  }

  /**
   * Add multiple sorted sets using weights, and store the resulting sorted set in a new key
   * @param key Destination key
   * @param sets Map containing set-key:weight pairs
   * @param options Aggregation options
   * @return 
   * @deprecated use {@link #rxZunionstoreWeighed} instead
   */
  @Deprecated()
  public Observable<Long> zunionstoreWeighedObservable(String key, Map<String,Double> sets, AggregateOptions options) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zunionstoreWeighed(key, sets, options, handler.toHandler());
    return handler;
  }

  /**
   * Add multiple sorted sets using weights, and store the resulting sorted set in a new key
   * @param key Destination key
   * @param sets Map containing set-key:weight pairs
   * @param options Aggregation options
   * @return 
   */
  public Single<Long> rxZunionstoreWeighed(String key, Map<String,Double> sets, AggregateOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zunionstoreWeighed(key, sets, options, fut);
    }));
  }

  /**
   * Incrementally iterate the keys space
   * @param cursor Cursor id
   * @param options Scan options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient scan(String cursor, ScanOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.scan(cursor, options, handler);
    return this;
  }

  /**
   * Incrementally iterate the keys space
   * @param cursor Cursor id
   * @param options Scan options
   * @return 
   * @deprecated use {@link #rxScan} instead
   */
  @Deprecated()
  public Observable<JsonArray> scanObservable(String cursor, ScanOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    scan(cursor, options, handler.toHandler());
    return handler;
  }

  /**
   * Incrementally iterate the keys space
   * @param cursor Cursor id
   * @param options Scan options
   * @return 
   */
  public Single<JsonArray> rxScan(String cursor, ScanOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      scan(cursor, options, fut);
    }));
  }

  /**
   * Incrementally iterate Set elements
   * @param key Key string
   * @param cursor Cursor id
   * @param options Scan options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient sscan(String key, String cursor, ScanOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.sscan(key, cursor, options, handler);
    return this;
  }

  /**
   * Incrementally iterate Set elements
   * @param key Key string
   * @param cursor Cursor id
   * @param options Scan options
   * @return 
   * @deprecated use {@link #rxSscan} instead
   */
  @Deprecated()
  public Observable<JsonArray> sscanObservable(String key, String cursor, ScanOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    sscan(key, cursor, options, handler.toHandler());
    return handler;
  }

  /**
   * Incrementally iterate Set elements
   * @param key Key string
   * @param cursor Cursor id
   * @param options Scan options
   * @return 
   */
  public Single<JsonArray> rxSscan(String key, String cursor, ScanOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      sscan(key, cursor, options, fut);
    }));
  }

  /**
   * Incrementally iterate hash fields and associated values
   * @param key Key string
   * @param cursor Cursor id
   * @param options Scan options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient hscan(String key, String cursor, ScanOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.hscan(key, cursor, options, handler);
    return this;
  }

  /**
   * Incrementally iterate hash fields and associated values
   * @param key Key string
   * @param cursor Cursor id
   * @param options Scan options
   * @return 
   * @deprecated use {@link #rxHscan} instead
   */
  @Deprecated()
  public Observable<JsonArray> hscanObservable(String key, String cursor, ScanOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hscan(key, cursor, options, handler.toHandler());
    return handler;
  }

  /**
   * Incrementally iterate hash fields and associated values
   * @param key Key string
   * @param cursor Cursor id
   * @param options Scan options
   * @return 
   */
  public Single<JsonArray> rxHscan(String key, String cursor, ScanOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hscan(key, cursor, options, fut);
    }));
  }

  /**
   * Incrementally iterate sorted sets elements and associated scores
   * @param key Key string
   * @param cursor Cursor id
   * @param options Scan options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient zscan(String key, String cursor, ScanOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.zscan(key, cursor, options, handler);
    return this;
  }

  /**
   * Incrementally iterate sorted sets elements and associated scores
   * @param key Key string
   * @param cursor Cursor id
   * @param options Scan options
   * @return 
   * @deprecated use {@link #rxZscan} instead
   */
  @Deprecated()
  public Observable<JsonArray> zscanObservable(String key, String cursor, ScanOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    zscan(key, cursor, options, handler.toHandler());
    return handler;
  }

  /**
   * Incrementally iterate sorted sets elements and associated scores
   * @param key Key string
   * @param cursor Cursor id
   * @param options Scan options
   * @return 
   */
  public Single<JsonArray> rxZscan(String key, String cursor, ScanOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      zscan(key, cursor, options, fut);
    }));
  }

  /**
   * Add one or more geospatial items in the geospatial index represented using a sorted set.
   * @param key Key string
   * @param longitude longitude
   * @param latitude latitude
   * @param member member
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient geoadd(String key, double longitude, double latitude, String member, Handler<AsyncResult<Long>> handler) { 
    delegate.geoadd(key, longitude, latitude, member, handler);
    return this;
  }

  /**
   * Add one or more geospatial items in the geospatial index represented using a sorted set.
   * @param key Key string
   * @param longitude longitude
   * @param latitude latitude
   * @param member member
   * @return 
   * @deprecated use {@link #rxGeoadd} instead
   */
  @Deprecated()
  public Observable<Long> geoaddObservable(String key, double longitude, double latitude, String member) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    geoadd(key, longitude, latitude, member, handler.toHandler());
    return handler;
  }

  /**
   * Add one or more geospatial items in the geospatial index represented using a sorted set.
   * @param key Key string
   * @param longitude longitude
   * @param latitude latitude
   * @param member member
   * @return 
   */
  public Single<Long> rxGeoadd(String key, double longitude, double latitude, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      geoadd(key, longitude, latitude, member, fut);
    }));
  }

  /**
   * Add one or more geospatial items in the geospatial index represented using a sorted set.
   * @param key Key string
   * @param members list of &lt;lon, lat, member&gt;
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient geoaddMany(String key, List<GeoMember> members, Handler<AsyncResult<Long>> handler) { 
    delegate.geoaddMany(key, members, handler);
    return this;
  }

  /**
   * Add one or more geospatial items in the geospatial index represented using a sorted set.
   * @param key Key string
   * @param members list of &lt;lon, lat, member&gt;
   * @return 
   * @deprecated use {@link #rxGeoaddMany} instead
   */
  @Deprecated()
  public Observable<Long> geoaddManyObservable(String key, List<GeoMember> members) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    geoaddMany(key, members, handler.toHandler());
    return handler;
  }

  /**
   * Add one or more geospatial items in the geospatial index represented using a sorted set.
   * @param key Key string
   * @param members list of &lt;lon, lat, member&gt;
   * @return 
   */
  public Single<Long> rxGeoaddMany(String key, List<GeoMember> members) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      geoaddMany(key, members, fut);
    }));
  }

  /**
   * Return valid Geohash strings representing the position of one or more elements in a sorted set value representing
   * a geospatial index (where elements were added using GEOADD).
   * @param key Key string
   * @param member member
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient geohash(String key, String member, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.geohash(key, member, handler);
    return this;
  }

  /**
   * Return valid Geohash strings representing the position of one or more elements in a sorted set value representing
   * a geospatial index (where elements were added using GEOADD).
   * @param key Key string
   * @param member member
   * @return 
   * @deprecated use {@link #rxGeohash} instead
   */
  @Deprecated()
  public Observable<JsonArray> geohashObservable(String key, String member) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    geohash(key, member, handler.toHandler());
    return handler;
  }

  /**
   * Return valid Geohash strings representing the position of one or more elements in a sorted set value representing
   * a geospatial index (where elements were added using GEOADD).
   * @param key Key string
   * @param member member
   * @return 
   */
  public Single<JsonArray> rxGeohash(String key, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      geohash(key, member, fut);
    }));
  }

  /**
   * Return valid Geohash strings representing the position of one or more elements in a sorted set value representing
   * a geospatial index (where elements were added using GEOADD).
   * @param key Key string
   * @param members list of members
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient geohashMany(String key, List<String> members, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.geohashMany(key, members, handler);
    return this;
  }

  /**
   * Return valid Geohash strings representing the position of one or more elements in a sorted set value representing
   * a geospatial index (where elements were added using GEOADD).
   * @param key Key string
   * @param members list of members
   * @return 
   * @deprecated use {@link #rxGeohashMany} instead
   */
  @Deprecated()
  public Observable<JsonArray> geohashManyObservable(String key, List<String> members) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    geohashMany(key, members, handler.toHandler());
    return handler;
  }

  /**
   * Return valid Geohash strings representing the position of one or more elements in a sorted set value representing
   * a geospatial index (where elements were added using GEOADD).
   * @param key Key string
   * @param members list of members
   * @return 
   */
  public Single<JsonArray> rxGeohashMany(String key, List<String> members) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      geohashMany(key, members, fut);
    }));
  }

  /**
   * Return the positions (longitude,latitude) of all the specified members of the geospatial index represented by the
   * sorted set at key.
   * @param key Key string
   * @param member member
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient geopos(String key, String member, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.geopos(key, member, handler);
    return this;
  }

  /**
   * Return the positions (longitude,latitude) of all the specified members of the geospatial index represented by the
   * sorted set at key.
   * @param key Key string
   * @param member member
   * @return 
   * @deprecated use {@link #rxGeopos} instead
   */
  @Deprecated()
  public Observable<JsonArray> geoposObservable(String key, String member) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    geopos(key, member, handler.toHandler());
    return handler;
  }

  /**
   * Return the positions (longitude,latitude) of all the specified members of the geospatial index represented by the
   * sorted set at key.
   * @param key Key string
   * @param member member
   * @return 
   */
  public Single<JsonArray> rxGeopos(String key, String member) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      geopos(key, member, fut);
    }));
  }

  /**
   * Return the positions (longitude,latitude) of all the specified members of the geospatial index represented by the
   * sorted set at key.
   * @param key Key string
   * @param members list of members
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient geoposMany(String key, List<String> members, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.geoposMany(key, members, handler);
    return this;
  }

  /**
   * Return the positions (longitude,latitude) of all the specified members of the geospatial index represented by the
   * sorted set at key.
   * @param key Key string
   * @param members list of members
   * @return 
   * @deprecated use {@link #rxGeoposMany} instead
   */
  @Deprecated()
  public Observable<JsonArray> geoposManyObservable(String key, List<String> members) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    geoposMany(key, members, handler.toHandler());
    return handler;
  }

  /**
   * Return the positions (longitude,latitude) of all the specified members of the geospatial index represented by the
   * sorted set at key.
   * @param key Key string
   * @param members list of members
   * @return 
   */
  public Single<JsonArray> rxGeoposMany(String key, List<String> members) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      geoposMany(key, members, fut);
    }));
  }

  /**
   * Return the distance between two members in the geospatial index represented by the sorted set.
   * @param key Key string
   * @param member1 member 1
   * @param member2 member 2
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient geodist(String key, String member1, String member2, Handler<AsyncResult<String>> handler) { 
    delegate.geodist(key, member1, member2, handler);
    return this;
  }

  /**
   * Return the distance between two members in the geospatial index represented by the sorted set.
   * @param key Key string
   * @param member1 member 1
   * @param member2 member 2
   * @return 
   * @deprecated use {@link #rxGeodist} instead
   */
  @Deprecated()
  public Observable<String> geodistObservable(String key, String member1, String member2) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    geodist(key, member1, member2, handler.toHandler());
    return handler;
  }

  /**
   * Return the distance between two members in the geospatial index represented by the sorted set.
   * @param key Key string
   * @param member1 member 1
   * @param member2 member 2
   * @return 
   */
  public Single<String> rxGeodist(String key, String member1, String member2) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      geodist(key, member1, member2, fut);
    }));
  }

  /**
   * Return the distance between two members in the geospatial index represented by the sorted set.
   * @param key Key string
   * @param member1 member 1
   * @param member2 member 2
   * @param unit geo unit
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient geodistWithUnit(String key, String member1, String member2, GeoUnit unit, Handler<AsyncResult<String>> handler) { 
    delegate.geodistWithUnit(key, member1, member2, unit, handler);
    return this;
  }

  /**
   * Return the distance between two members in the geospatial index represented by the sorted set.
   * @param key Key string
   * @param member1 member 1
   * @param member2 member 2
   * @param unit geo unit
   * @return 
   * @deprecated use {@link #rxGeodistWithUnit} instead
   */
  @Deprecated()
  public Observable<String> geodistWithUnitObservable(String key, String member1, String member2, GeoUnit unit) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    geodistWithUnit(key, member1, member2, unit, handler.toHandler());
    return handler;
  }

  /**
   * Return the distance between two members in the geospatial index represented by the sorted set.
   * @param key Key string
   * @param member1 member 1
   * @param member2 member 2
   * @param unit geo unit
   * @return 
   */
  public Single<String> rxGeodistWithUnit(String key, String member1, String member2, GeoUnit unit) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      geodistWithUnit(key, member1, member2, unit, fut);
    }));
  }

  /**
   * Return the members of a sorted set populated with geospatial information using GEOADD, which are within the borders
   * of the area specified with the center location and the maximum distance from the center (the radius).
   * @param key Key string
   * @param longitude longitude
   * @param latitude latitude
   * @param radius radius
   * @param unit geo unit
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient georadius(String key, double longitude, double latitude, double radius, GeoUnit unit, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.georadius(key, longitude, latitude, radius, unit, handler);
    return this;
  }

  /**
   * Return the members of a sorted set populated with geospatial information using GEOADD, which are within the borders
   * of the area specified with the center location and the maximum distance from the center (the radius).
   * @param key Key string
   * @param longitude longitude
   * @param latitude latitude
   * @param radius radius
   * @param unit geo unit
   * @return 
   * @deprecated use {@link #rxGeoradius} instead
   */
  @Deprecated()
  public Observable<JsonArray> georadiusObservable(String key, double longitude, double latitude, double radius, GeoUnit unit) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    georadius(key, longitude, latitude, radius, unit, handler.toHandler());
    return handler;
  }

  /**
   * Return the members of a sorted set populated with geospatial information using GEOADD, which are within the borders
   * of the area specified with the center location and the maximum distance from the center (the radius).
   * @param key Key string
   * @param longitude longitude
   * @param latitude latitude
   * @param radius radius
   * @param unit geo unit
   * @return 
   */
  public Single<JsonArray> rxGeoradius(String key, double longitude, double latitude, double radius, GeoUnit unit) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      georadius(key, longitude, latitude, radius, unit, fut);
    }));
  }

  /**
   * Return the members of a sorted set populated with geospatial information using GEOADD, which are within the borders
   * of the area specified with the center location and the maximum distance from the center (the radius).
   * @param key Key string
   * @param longitude longitude
   * @param latitude latitude
   * @param radius radius
   * @param unit geo unit
   * @param options geo radius options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient georadiusWithOptions(String key, double longitude, double latitude, double radius, GeoUnit unit, GeoRadiusOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.georadiusWithOptions(key, longitude, latitude, radius, unit, options, handler);
    return this;
  }

  /**
   * Return the members of a sorted set populated with geospatial information using GEOADD, which are within the borders
   * of the area specified with the center location and the maximum distance from the center (the radius).
   * @param key Key string
   * @param longitude longitude
   * @param latitude latitude
   * @param radius radius
   * @param unit geo unit
   * @param options geo radius options
   * @return 
   * @deprecated use {@link #rxGeoradiusWithOptions} instead
   */
  @Deprecated()
  public Observable<JsonArray> georadiusWithOptionsObservable(String key, double longitude, double latitude, double radius, GeoUnit unit, GeoRadiusOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    georadiusWithOptions(key, longitude, latitude, radius, unit, options, handler.toHandler());
    return handler;
  }

  /**
   * Return the members of a sorted set populated with geospatial information using GEOADD, which are within the borders
   * of the area specified with the center location and the maximum distance from the center (the radius).
   * @param key Key string
   * @param longitude longitude
   * @param latitude latitude
   * @param radius radius
   * @param unit geo unit
   * @param options geo radius options
   * @return 
   */
  public Single<JsonArray> rxGeoradiusWithOptions(String key, double longitude, double latitude, double radius, GeoUnit unit, GeoRadiusOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      georadiusWithOptions(key, longitude, latitude, radius, unit, options, fut);
    }));
  }

  /**
   * This command is exactly like GEORADIUS with the sole difference that instead of taking, as the center of the area
   * to query, a longitude and latitude value, it takes the name of a member already existing inside the geospatial
   * index represented by the sorted set.
   * @param key Key string
   * @param member member
   * @param radius radius
   * @param unit geo unit
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient georadiusbymember(String key, String member, double radius, GeoUnit unit, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.georadiusbymember(key, member, radius, unit, handler);
    return this;
  }

  /**
   * This command is exactly like GEORADIUS with the sole difference that instead of taking, as the center of the area
   * to query, a longitude and latitude value, it takes the name of a member already existing inside the geospatial
   * index represented by the sorted set.
   * @param key Key string
   * @param member member
   * @param radius radius
   * @param unit geo unit
   * @return 
   * @deprecated use {@link #rxGeoradiusbymember} instead
   */
  @Deprecated()
  public Observable<JsonArray> georadiusbymemberObservable(String key, String member, double radius, GeoUnit unit) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    georadiusbymember(key, member, radius, unit, handler.toHandler());
    return handler;
  }

  /**
   * This command is exactly like GEORADIUS with the sole difference that instead of taking, as the center of the area
   * to query, a longitude and latitude value, it takes the name of a member already existing inside the geospatial
   * index represented by the sorted set.
   * @param key Key string
   * @param member member
   * @param radius radius
   * @param unit geo unit
   * @return 
   */
  public Single<JsonArray> rxGeoradiusbymember(String key, String member, double radius, GeoUnit unit) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      georadiusbymember(key, member, radius, unit, fut);
    }));
  }

  /**
   * This command is exactly like GEORADIUS with the sole difference that instead of taking, as the center of the area
   * to query, a longitude and latitude value, it takes the name of a member already existing inside the geospatial
   * index represented by the sorted set.
   * @param key Key string
   * @param member member
   * @param radius radius
   * @param unit geo unit
   * @param options geo radius options
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient georadiusbymemberWithOptions(String key, String member, double radius, GeoUnit unit, GeoRadiusOptions options, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.georadiusbymemberWithOptions(key, member, radius, unit, options, handler);
    return this;
  }

  /**
   * This command is exactly like GEORADIUS with the sole difference that instead of taking, as the center of the area
   * to query, a longitude and latitude value, it takes the name of a member already existing inside the geospatial
   * index represented by the sorted set.
   * @param key Key string
   * @param member member
   * @param radius radius
   * @param unit geo unit
   * @param options geo radius options
   * @return 
   * @deprecated use {@link #rxGeoradiusbymemberWithOptions} instead
   */
  @Deprecated()
  public Observable<JsonArray> georadiusbymemberWithOptionsObservable(String key, String member, double radius, GeoUnit unit, GeoRadiusOptions options) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    georadiusbymemberWithOptions(key, member, radius, unit, options, handler.toHandler());
    return handler;
  }

  /**
   * This command is exactly like GEORADIUS with the sole difference that instead of taking, as the center of the area
   * to query, a longitude and latitude value, it takes the name of a member already existing inside the geospatial
   * index represented by the sorted set.
   * @param key Key string
   * @param member member
   * @param radius radius
   * @param unit geo unit
   * @param options geo radius options
   * @return 
   */
  public Single<JsonArray> rxGeoradiusbymemberWithOptions(String key, String member, double radius, GeoUnit unit, GeoRadiusOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      georadiusbymemberWithOptions(key, member, radius, unit, options, fut);
    }));
  }

  /**
   * Instruct the server whether to reply to commands.
   * @param options 
   * @param handler 
   * @return 
   */
  public RedisClient clientReply(ClientReplyOptions options, Handler<AsyncResult<String>> handler) { 
    delegate.clientReply(options, handler);
    return this;
  }

  /**
   * Instruct the server whether to reply to commands.
   * @param options 
   * @return 
   * @deprecated use {@link #rxClientReply} instead
   */
  @Deprecated()
  public Observable<String> clientReplyObservable(ClientReplyOptions options) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    clientReply(options, handler.toHandler());
    return handler;
  }

  /**
   * Instruct the server whether to reply to commands.
   * @param options 
   * @return 
   */
  public Single<String> rxClientReply(ClientReplyOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      clientReply(options, fut);
    }));
  }

  /**
   * Get the length of the value of a hash field.
   * @param key Key String
   * @param field field
   * @param handler 
   * @return 
   */
  public RedisClient hstrlen(String key, String field, Handler<AsyncResult<Long>> handler) { 
    delegate.hstrlen(key, field, handler);
    return this;
  }

  /**
   * Get the length of the value of a hash field.
   * @param key Key String
   * @param field field
   * @return 
   * @deprecated use {@link #rxHstrlen} instead
   */
  @Deprecated()
  public Observable<Long> hstrlenObservable(String key, String field) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    hstrlen(key, field, handler.toHandler());
    return handler;
  }

  /**
   * Get the length of the value of a hash field.
   * @param key Key String
   * @param field field
   * @return 
   */
  public Single<Long> rxHstrlen(String key, String field) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      hstrlen(key, field, fut);
    }));
  }

  /**
   * Alters the last access time of a key(s). Returns the number of existing keys specified.
   * @param key Key String
   * @param handler 
   * @return 
   */
  public RedisClient touch(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.touch(key, handler);
    return this;
  }

  /**
   * Alters the last access time of a key(s). Returns the number of existing keys specified.
   * @param key Key String
   * @return 
   * @deprecated use {@link #rxTouch} instead
   */
  @Deprecated()
  public Observable<Long> touchObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    touch(key, handler.toHandler());
    return handler;
  }

  /**
   * Alters the last access time of a key(s). Returns the number of existing keys specified.
   * @param key Key String
   * @return 
   */
  public Single<Long> rxTouch(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      touch(key, fut);
    }));
  }

  /**
   * Alters the last access time of a key(s). Returns the number of existing keys specified.
   * @param keys list of keys
   * @param handler 
   * @return 
   */
  public RedisClient touchMany(List<String> keys, Handler<AsyncResult<Long>> handler) { 
    delegate.touchMany(keys, handler);
    return this;
  }

  /**
   * Alters the last access time of a key(s). Returns the number of existing keys specified.
   * @param keys list of keys
   * @return 
   * @deprecated use {@link #rxTouchMany} instead
   */
  @Deprecated()
  public Observable<Long> touchManyObservable(List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    touchMany(keys, handler.toHandler());
    return handler;
  }

  /**
   * Alters the last access time of a key(s). Returns the number of existing keys specified.
   * @param keys list of keys
   * @return 
   */
  public Single<Long> rxTouchMany(List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      touchMany(keys, fut);
    }));
  }

  /**
   * Set the debug mode for executed scripts.
   * @param scriptDebugOptions the option
   * @param handler 
   * @return 
   */
  public RedisClient scriptDebug(ScriptDebugOptions scriptDebugOptions, Handler<AsyncResult<String>> handler) { 
    delegate.scriptDebug(scriptDebugOptions, handler);
    return this;
  }

  /**
   * Set the debug mode for executed scripts.
   * @param scriptDebugOptions the option
   * @return 
   * @deprecated use {@link #rxScriptDebug} instead
   */
  @Deprecated()
  public Observable<String> scriptDebugObservable(ScriptDebugOptions scriptDebugOptions) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    scriptDebug(scriptDebugOptions, handler.toHandler());
    return handler;
  }

  /**
   * Set the debug mode for executed scripts.
   * @param scriptDebugOptions the option
   * @return 
   */
  public Single<String> rxScriptDebug(ScriptDebugOptions scriptDebugOptions) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      scriptDebug(scriptDebugOptions, fut);
    }));
  }

  /**
   * Perform arbitrary bitfield integer operations on strings.
   * @param key Key string
   * @param bitFieldOptions 
   * @param handler 
   * @return 
   */
  public RedisClient bitfield(String key, BitFieldOptions bitFieldOptions, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.bitfield(key, bitFieldOptions, handler);
    return this;
  }

  /**
   * Perform arbitrary bitfield integer operations on strings.
   * @param key Key string
   * @param bitFieldOptions 
   * @return 
   * @deprecated use {@link #rxBitfield} instead
   */
  @Deprecated()
  public Observable<JsonArray> bitfieldObservable(String key, BitFieldOptions bitFieldOptions) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    bitfield(key, bitFieldOptions, handler.toHandler());
    return handler;
  }

  /**
   * Perform arbitrary bitfield integer operations on strings.
   * @param key Key string
   * @param bitFieldOptions 
   * @return 
   */
  public Single<JsonArray> rxBitfield(String key, BitFieldOptions bitFieldOptions) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      bitfield(key, bitFieldOptions, fut);
    }));
  }

  /**
   * Perform arbitrary bitfield integer operations on strings.
   * @param key Key string
   * @param commands 
   * @param overflow 
   * @param handler 
   * @return 
   */
  public RedisClient bitfieldWithOverflow(String key, BitFieldOptions commands, BitFieldOverflowOptions overflow, Handler<AsyncResult<JsonArray>> handler) { 
    delegate.bitfieldWithOverflow(key, commands, overflow, handler);
    return this;
  }

  /**
   * Perform arbitrary bitfield integer operations on strings.
   * @param key Key string
   * @param commands 
   * @param overflow 
   * @return 
   * @deprecated use {@link #rxBitfieldWithOverflow} instead
   */
  @Deprecated()
  public Observable<JsonArray> bitfieldWithOverflowObservable(String key, BitFieldOptions commands, BitFieldOverflowOptions overflow) { 
    io.vertx.rx.java.ObservableFuture<JsonArray> handler = io.vertx.rx.java.RxHelper.observableFuture();
    bitfieldWithOverflow(key, commands, overflow, handler.toHandler());
    return handler;
  }

  /**
   * Perform arbitrary bitfield integer operations on strings.
   * @param key Key string
   * @param commands 
   * @param overflow 
   * @return 
   */
  public Single<JsonArray> rxBitfieldWithOverflow(String key, BitFieldOptions commands, BitFieldOverflowOptions overflow) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      bitfieldWithOverflow(key, commands, overflow, fut);
    }));
  }

  /**
   * Delete a key asynchronously in another thread. Otherwise it is just as DEL, but non blocking.
   * @param key Key to delete
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient unlink(String key, Handler<AsyncResult<Long>> handler) { 
    delegate.unlink(key, handler);
    return this;
  }

  /**
   * Delete a key asynchronously in another thread. Otherwise it is just as DEL, but non blocking.
   * @param key Key to delete
   * @return 
   * @deprecated use {@link #rxUnlink} instead
   */
  @Deprecated()
  public Observable<Long> unlinkObservable(String key) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    unlink(key, handler.toHandler());
    return handler;
  }

  /**
   * Delete a key asynchronously in another thread. Otherwise it is just as DEL, but non blocking.
   * @param key Key to delete
   * @return 
   */
  public Single<Long> rxUnlink(String key) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      unlink(key, fut);
    }));
  }

  /**
   * Delete multiple keys asynchronously in another thread. Otherwise it is just as DEL, but non blocking.
   * @param keys List of keys to delete
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient unlinkMany(List<String> keys, Handler<AsyncResult<Long>> handler) { 
    delegate.unlinkMany(keys, handler);
    return this;
  }

  /**
   * Delete multiple keys asynchronously in another thread. Otherwise it is just as DEL, but non blocking.
   * @param keys List of keys to delete
   * @return 
   * @deprecated use {@link #rxUnlinkMany} instead
   */
  @Deprecated()
  public Observable<Long> unlinkManyObservable(List<String> keys) { 
    io.vertx.rx.java.ObservableFuture<Long> handler = io.vertx.rx.java.RxHelper.observableFuture();
    unlinkMany(keys, handler.toHandler());
    return handler;
  }

  /**
   * Delete multiple keys asynchronously in another thread. Otherwise it is just as DEL, but non blocking.
   * @param keys List of keys to delete
   * @return 
   */
  public Single<Long> rxUnlinkMany(List<String> keys) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      unlinkMany(keys, fut);
    }));
  }

  /**
   * Swaps two Redis databases
   * @param index1 index of first database to swap
   * @param index2 index of second database to swap
   * @param handler Handler for the result of this call.
   * @return 
   */
  public RedisClient swapdb(int index1, int index2, Handler<AsyncResult<String>> handler) { 
    delegate.swapdb(index1, index2, handler);
    return this;
  }

  /**
   * Swaps two Redis databases
   * @param index1 index of first database to swap
   * @param index2 index of second database to swap
   * @return 
   * @deprecated use {@link #rxSwapdb} instead
   */
  @Deprecated()
  public Observable<String> swapdbObservable(int index1, int index2) { 
    io.vertx.rx.java.ObservableFuture<String> handler = io.vertx.rx.java.RxHelper.observableFuture();
    swapdb(index1, index2, handler.toHandler());
    return handler;
  }

  /**
   * Swaps two Redis databases
   * @param index1 index of first database to swap
   * @param index2 index of second database to swap
   * @return 
   */
  public Single<String> rxSwapdb(int index1, int index2) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      swapdb(index1, index2, fut);
    }));
  }


  public static  RedisClient newInstance(io.vertx.redis.RedisClient arg) {
    return arg != null ? new RedisClient(arg) : null;
  }
}
