/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sshd.common.config.keys;

import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StreamCorruptedException;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchProviderException;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPrivateKeySpec;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;
import org.apache.sshd.common.cipher.ECCurves;
import org.apache.sshd.common.config.keys.AbstractPublicKeyEntryDecoder;
import org.apache.sshd.common.util.NumberUtils;
import org.apache.sshd.common.util.SecurityUtils;
import org.apache.sshd.common.util.ValidateUtils;
import org.apache.sshd.common.util.buffer.BufferUtils;

public class ECDSAPublicKeyEntryDecoder
extends AbstractPublicKeyEntryDecoder<ECPublicKey, ECPrivateKey> {
    public static final ECDSAPublicKeyEntryDecoder INSTANCE = new ECDSAPublicKeyEntryDecoder();
    public static final byte ECPOINT_UNCOMPRESSED_FORM_INDICATOR = 4;
    public static final byte ECPOINT_COMPRESSED_VARIANT_2 = 2;
    public static final byte ECPOINT_COMPRESSED_VARIANT_3 = 2;

    public ECDSAPublicKeyEntryDecoder() {
        super(ECPublicKey.class, ECPrivateKey.class, ECCurves.KEY_TYPES);
    }

    @Override
    public ECPublicKey decodePublicKey(String keyType, InputStream keyData) throws IOException, GeneralSecurityException {
        ECPoint w;
        ECCurves curve = ECCurves.fromKeyType(keyType);
        if (curve == null) {
            throw new InvalidKeySpecException("Not an EC curve name: " + keyType);
        }
        if (!SecurityUtils.hasEcc()) {
            throw new NoSuchProviderException("ECC not supported");
        }
        String keyCurveName = curve.getName();
        ECParameterSpec paramSpec = curve.getParameters();
        String encCurveName = ECDSAPublicKeyEntryDecoder.decodeString(keyData);
        if (!keyCurveName.equals(encCurveName)) {
            throw new InvalidKeySpecException("Mismatched key curve name (" + keyCurveName + ") vs. encoded one (" + encCurveName + ")");
        }
        byte[] octets = ECDSAPublicKeyEntryDecoder.readRLEBytes(keyData);
        try {
            w = ECDSAPublicKeyEntryDecoder.octetStringToEcPoint(octets);
            if (w == null) {
                throw new InvalidKeySpecException("No ECPoint generated for curve=" + keyCurveName + " from octets=" + BufferUtils.toHex(':', octets));
            }
        }
        catch (RuntimeException e) {
            throw new InvalidKeySpecException("Failed (" + e.getClass().getSimpleName() + ")" + " to generate ECPoint for curve=" + keyCurveName + " from octets=" + BufferUtils.toHex(':', octets) + ": " + e.getMessage());
        }
        return (ECPublicKey)this.generatePublicKey(new ECPublicKeySpec(w, paramSpec));
    }

    @Override
    public ECPublicKey clonePublicKey(ECPublicKey key) throws GeneralSecurityException {
        if (!SecurityUtils.hasEcc()) {
            throw new NoSuchProviderException("ECC not supported");
        }
        if (key == null) {
            return null;
        }
        ECParameterSpec params = key.getParams();
        if (params == null) {
            throw new InvalidKeyException("Missing parameters in key");
        }
        return (ECPublicKey)this.generatePublicKey(new ECPublicKeySpec(key.getW(), params));
    }

    @Override
    public ECPrivateKey clonePrivateKey(ECPrivateKey key) throws GeneralSecurityException {
        if (!SecurityUtils.hasEcc()) {
            throw new NoSuchProviderException("ECC not supported");
        }
        if (key == null) {
            return null;
        }
        ECParameterSpec params = key.getParams();
        if (params == null) {
            throw new InvalidKeyException("Missing parameters in key");
        }
        return (ECPrivateKey)this.generatePrivateKey(new ECPrivateKeySpec(key.getS(), params));
    }

    @Override
    public String encodePublicKey(OutputStream s, ECPublicKey key) throws IOException {
        ValidateUtils.checkNotNull(key, "No public key provided");
        ECParameterSpec params = ValidateUtils.checkNotNull(key.getParams(), "No EC parameters available");
        ECCurves curve = ValidateUtils.checkNotNull(ECCurves.fromCurveParameters(params), "Cannot determine curve");
        String keyType = curve.getKeyType();
        String curveName = curve.getName();
        ECDSAPublicKeyEntryDecoder.encodeString(s, keyType);
        ECDSAPublicKeyEntryDecoder.encodeString(s, curveName);
        ECPointCompression.UNCOMPRESSED.writeECPoint(s, curveName, key.getW());
        return keyType;
    }

    @Override
    public KeyFactory getKeyFactoryInstance() throws GeneralSecurityException {
        if (SecurityUtils.hasEcc()) {
            return SecurityUtils.getKeyFactory("EC");
        }
        throw new NoSuchProviderException("ECC not supported");
    }

    @Override
    public KeyPair generateKeyPair(int keySize) throws GeneralSecurityException {
        ECCurves curve = ECCurves.fromCurveSize(keySize);
        if (curve == null) {
            throw new InvalidKeySpecException("Unknown curve for key size=" + keySize);
        }
        KeyPairGenerator gen = this.getKeyPairGenerator();
        gen.initialize(curve.getParameters());
        return gen.generateKeyPair();
    }

    @Override
    public KeyPairGenerator getKeyPairGenerator() throws GeneralSecurityException {
        if (SecurityUtils.hasEcc()) {
            return SecurityUtils.getKeyPairGenerator("EC");
        }
        throw new NoSuchProviderException("ECC not supported");
    }

    public static ECPoint octetStringToEcPoint(byte ... octets) {
        if (NumberUtils.isEmpty(octets)) {
            return null;
        }
        int startIndex = ECDSAPublicKeyEntryDecoder.findFirstNonZeroIndex(octets);
        if (startIndex < 0) {
            throw new IllegalArgumentException("All zeroes ECPoint N/A");
        }
        byte indicator = octets[startIndex];
        ECPointCompression compression = ECPointCompression.fromIndicatorValue(indicator);
        if (compression == null) {
            throw new UnsupportedOperationException("Unknown compression indicator value: 0x" + Integer.toHexString(indicator & 0xFF));
        }
        return compression.octetStringToEcPoint(octets, startIndex + 1, octets.length - startIndex - 1);
    }

    private static int findFirstNonZeroIndex(byte ... octets) {
        if (NumberUtils.isEmpty(octets)) {
            return -1;
        }
        for (int index = 0; index < octets.length; ++index) {
            if (octets[index] == 0) continue;
            return index;
        }
        return -1;
    }

    public static enum ECPointCompression {
        VARIANT2(2){

            @Override
            public ECPoint octetStringToEcPoint(byte[] octets, int startIndex, int len) {
                byte[] xp = new byte[len];
                System.arraycopy(octets, startIndex, xp, 0, len);
                BigInteger x = 1.octetStringToInteger(xp);
                throw new UnsupportedOperationException("octetStringToEcPoint(" + this.name() + ")(X=" + x + ") compression support N/A");
            }
        }
        ,
        VARIANT3(3){

            @Override
            public ECPoint octetStringToEcPoint(byte[] octets, int startIndex, int len) {
                byte[] xp = new byte[len];
                System.arraycopy(octets, startIndex, xp, 0, len);
                BigInteger x = 2.octetStringToInteger(xp);
                throw new UnsupportedOperationException("octetStringToEcPoint(" + this.name() + ")(X=" + x + ") compression support N/A");
            }
        }
        ,
        UNCOMPRESSED(4){

            @Override
            public ECPoint octetStringToEcPoint(byte[] octets, int startIndex, int len) {
                int numElements = len / 2;
                if (len != numElements * 2) {
                    throw new IllegalArgumentException("octetStringToEcPoint(" + this.name() + ") " + " invalid remainder octets representation: " + " expected=" + 2 * numElements + ", actual=" + len);
                }
                byte[] xp = new byte[numElements];
                byte[] yp = new byte[numElements];
                System.arraycopy(octets, startIndex, xp, 0, numElements);
                System.arraycopy(octets, startIndex + numElements, yp, 0, numElements);
                BigInteger x = 3.octetStringToInteger(xp);
                BigInteger y = 3.octetStringToInteger(yp);
                return new ECPoint(x, y);
            }

            @Override
            public void writeECPoint(OutputStream s, String curveName, ECPoint p) throws IOException {
                ECCurves curve = ECCurves.fromCurveName(curveName);
                if (curve == null) {
                    throw new StreamCorruptedException("writeECPoint(" + this.name() + ")[" + curveName + "] cannot determine octets count");
                }
                int numElements = curve.getNumPointOctets();
                AbstractPublicKeyEntryDecoder.encodeInt(s, 1 + 2 * numElements);
                s.write(this.getIndicatorValue());
                this.writeCoordinate(s, "X", p.getAffineX(), numElements);
                this.writeCoordinate(s, "Y", p.getAffineY(), numElements);
            }
        };

        public static final Set<ECPointCompression> VALUES;
        private final byte indicatorValue;

        private ECPointCompression(byte indicator) {
            this.indicatorValue = indicator;
        }

        public final byte getIndicatorValue() {
            return this.indicatorValue;
        }

        public abstract ECPoint octetStringToEcPoint(byte[] var1, int var2, int var3);

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        public byte[] ecPointToOctetString(String curveName, ECPoint p) {
            try (ByteArrayOutputStream baos = new ByteArrayOutputStream(196);){
                this.writeECPoint(baos, curveName, p);
                byte[] byArray = baos.toByteArray();
                return byArray;
            }
            catch (IOException e) {
                throw new RuntimeException("ecPointToOctetString(" + curveName + ")" + " failed (" + e.getClass().getSimpleName() + ")" + " to write data: " + e.getMessage(), e);
            }
        }

        public void writeECPoint(OutputStream s, String curveName, ECPoint p) throws IOException {
            if (s == null) {
                throw new EOFException("No output stream");
            }
            throw new StreamCorruptedException("writeECPoint(" + this.name() + ")[" + p + "] N/A");
        }

        protected void writeCoordinate(OutputStream s, String n, BigInteger v, int numElements) throws IOException {
            byte[] vp = v.toByteArray();
            int startIndex = 0;
            int vLen = vp.length;
            if (vLen > numElements && vp[0] == 0) {
                ++startIndex;
                --vLen;
            }
            if (vLen > numElements) {
                throw new StreamCorruptedException("writeCoordinate(" + this.name() + ")[" + n + "]" + " value length (" + vLen + ") exceeds max. (" + numElements + ")" + " for " + v);
            }
            if (vLen < numElements) {
                byte[] tmp = new byte[numElements];
                System.arraycopy(vp, startIndex, tmp, numElements - vLen, vLen);
                vp = tmp;
            }
            s.write(vp, startIndex, vLen);
        }

        public static ECPointCompression fromIndicatorValue(int value) {
            if (value < 0 || value > 255) {
                return null;
            }
            for (ECPointCompression c : VALUES) {
                if (value != c.getIndicatorValue()) continue;
                return c;
            }
            return null;
        }

        public static BigInteger octetStringToInteger(byte ... octets) {
            if (octets == null) {
                return null;
            }
            if (octets.length == 0) {
                return BigInteger.ZERO;
            }
            return new BigInteger(1, octets);
        }

        static {
            VALUES = Collections.unmodifiableSet(EnumSet.allOf(ECPointCompression.class));
        }
    }
}

