/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.ext.web.handler;

import io.vertx.core.Handler;
import io.vertx.core.MultiMap;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpClientRequest;
import io.vertx.core.http.HttpHeaders;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.json.JsonObject;
import io.vertx.ext.web.FileUpload;
import io.vertx.ext.web.Route;
import io.vertx.ext.web.RoutingContext;
import io.vertx.ext.web.WebTestBase;
import io.vertx.ext.web.handler.BodyHandler;
import io.vertx.test.core.TestUtils;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.List;
import java.util.function.Function;
import org.junit.AfterClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class BodyHandlerTest
extends WebTestBase {
    @Rule
    public TemporaryFolder tempUploads = new TemporaryFolder();

    @Override
    public void setUp() throws Exception {
        super.setUp();
        this.router.route().handler((Handler)BodyHandler.create());
    }

    @AfterClass
    public static void oneTimeTearDown() throws Exception {
        BodyHandlerTest.cleanupFileUploadDir();
    }

    @Test
    public void testGETWithoutBody() throws Exception {
        this.router.route().handler(rc -> {
            this.assertNull(rc.body().buffer());
            rc.response().end();
        });
        this.testRequest(HttpMethod.GET, "/", 200, "OK");
    }

    @Test
    public void testHEADWithoutBody() throws Exception {
        this.router.route().handler(rc -> {
            this.assertNull(rc.body().buffer());
            rc.response().end();
        });
        this.testRequest(HttpMethod.HEAD, "/", 200, "OK");
    }

    @Test
    public void testBodyBuffer() throws Exception {
        Buffer buff = TestUtils.randomBuffer((int)1000);
        this.router.route().handler(rc -> {
            this.assertEquals(buff, rc.body().buffer());
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.write((Object)buff);
        }, 200, "OK", null);
    }

    @Test
    public void testBodyString() throws Exception {
        String str = "sausages";
        this.router.route().handler(rc -> {
            this.assertEquals(str, rc.body().asString());
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.write(str);
        }, 200, "OK", null);
    }

    @Test
    public void testBodyStringWithEncoding() throws Exception {
        String str = "\u00ff";
        this.router.route().handler(rc -> {
            this.assertEquals(1L, rc.body().length());
            String decoded = rc.body().asString();
            this.assertEquals(str, decoded);
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.putHeader("content-type", "text/plain;charset=ISO-8859-1");
            byte b = str.getBytes(StandardCharsets.ISO_8859_1)[0];
            req.write((Object)Buffer.buffer((byte[])new byte[]{b}));
        }, 200, "OK", null);
    }

    @Test
    public void testBodyStringEncoding() throws Exception {
        String str = TestUtils.randomUnicodeString((int)100);
        String enc = "UTF-16";
        this.router.route().handler(rc -> {
            this.assertEquals(str, rc.body().asString(enc));
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.write(str, enc);
        }, 200, "OK", null);
    }

    @Test
    public void testBodyJson() throws Exception {
        JsonObject json = new JsonObject().put("foo", (Object)"bar").put("blah", (Object)123);
        this.router.route().handler(rc -> {
            this.assertEquals(json, rc.body().asJsonObject());
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.write(json.encode());
        }, 200, "OK", null);
    }

    @Test
    public void testBodyJsonWithNegativeContentLength() throws Exception {
        JsonObject json = new JsonObject().put("foo", (Object)"bar").put("blah", (Object)123);
        this.router.route().handler(rc -> {
            this.assertEquals(json, rc.body().asJsonObject());
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.putHeader(HttpHeaders.CONTENT_LENGTH, (CharSequence)"-1");
            req.write(json.encode());
        }, 200, "OK", null);
    }

    @Test
    public void testBodyJsonWithEmptyContentLength() throws Exception {
        JsonObject json = new JsonObject().put("foo", (Object)"bar").put("blah", (Object)123);
        this.router.route().handler(rc -> {
            this.assertEquals(json, rc.body().asJsonObject());
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.putHeader(HttpHeaders.CONTENT_LENGTH, (CharSequence)"");
            req.write(json.encode());
        }, 200, "OK", null);
    }

    @Test
    public void testBodyJsonWithHugeContentLength() throws Exception {
        JsonObject json = new JsonObject().put("foo", (Object)"bar").put("blah", (Object)123);
        this.router.route().handler(rc -> {
            this.assertEquals(json, rc.body().asJsonObject());
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.putHeader(HttpHeaders.CONTENT_LENGTH, (CharSequence)String.valueOf(Long.MAX_VALUE));
            req.write(json.encode());
        }, 200, "OK", null);
    }

    @Test
    public void testBodyTooBig() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setBodyLimit(5000L));
        Buffer buff = TestUtils.randomBuffer((int)10000);
        this.router.route().handler(rc -> this.fail("Should not be called"));
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.write((Object)buff);
        }, 413, "Request Entity Too Large", null);
    }

    @Test
    public void testBodyTooBig2() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setBodyLimit(500L));
        Buffer buff = TestUtils.randomBuffer((int)1000);
        this.router.route().handler(rc -> this.fail("Should not be called"));
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.write((Object)buff);
        }, 413, "Request Entity Too Large", null);
    }

    @Test
    public void testFileUploadSmallUpload() throws Exception {
        this.testFileUpload("file-uploads", 50);
    }

    @Test
    public void testFileUpload7990Upload() throws Exception {
        this.testFileUpload("file-uploads", 7990);
    }

    @Test
    public void testFileUploadLargeUpload() throws Exception {
        this.testFileUpload("file-uploads", 20000);
    }

    @Test
    public void testFileUploadDefaultUploadsDir() throws Exception {
        this.testFileUpload("file-uploads", 5000);
    }

    @Test
    public void testFileUploadOtherUploadsDir() throws Exception {
        this.router.clear();
        File dir = this.tempUploads.newFolder();
        this.router.route().handler((Handler)BodyHandler.create().setUploadsDirectory(dir.getPath()));
        this.testFileUpload(dir.getPath(), 5000);
    }

    private void testFileUpload(String uploadsDir, int size) throws Exception {
        String name = "somename";
        String fileName = "somefile.dat";
        String contentType = "application/octet-stream";
        Buffer fileData = TestUtils.randomBuffer((int)size);
        this.router.route().handler(rc -> {
            List fileUploads = rc.fileUploads();
            this.assertNotNull(fileUploads);
            this.assertEquals(1L, fileUploads.size());
            FileUpload upload = (FileUpload)fileUploads.iterator().next();
            this.assertEquals(name, upload.name());
            this.assertEquals(fileName, upload.fileName());
            this.assertEquals(contentType, upload.contentType());
            this.assertEquals("binary", upload.contentTransferEncoding());
            this.assertEquals(fileData.length(), upload.size());
            String uploadedFileName = upload.uploadedFileName();
            this.assertTrue(uploadedFileName.startsWith(uploadsDir + File.separator));
            Buffer uploaded = this.vertx.fileSystem().readFileBlocking(uploadedFileName);
            this.assertEquals(fileData, uploaded);
            Buffer rawBody = rc.body().buffer();
            this.assertNull(rawBody);
            rc.response().end();
        });
        this.sendFileUploadRequest(fileData, 200, "OK");
    }

    @Test
    public void testFileUploadTooBig() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setBodyLimit(20000L));
        Buffer fileData = TestUtils.randomBuffer((int)50000);
        this.router.route().handler(rc -> this.fail("Should not be called"));
        this.sendFileUploadRequest(fileData, 413, "Request Entity Too Large");
    }

    @Test
    public void testFileUploadTooBig2() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setBodyLimit(20000L));
        Buffer fileData = TestUtils.randomBuffer((int)50000);
        this.router.route().handler(rc -> this.fail("Should not be called"));
        this.sendFileUploadRequest(fileData, 413, "Request Entity Too Large");
    }

    @Test
    public void testFileUploadNoFileRemovalOnEnd() throws Exception {
        this.testFileUploadFileRemoval((Handler<RoutingContext>)((Handler)rc -> rc.response().end()), false, 200, "OK");
    }

    @Test
    public void testFileUploadFileRemovalOnEnd() throws Exception {
        this.testFileUploadFileRemoval((Handler<RoutingContext>)((Handler)rc -> rc.response().end()), true, 200, "OK");
    }

    @Test
    public void testFileUploadFileRemovalOnError() throws Exception {
        this.testFileUploadFileRemoval((Handler<RoutingContext>)((Handler)rc -> {
            throw new IllegalStateException();
        }), true, 500, "Internal Server Error");
    }

    @Test
    public void testFileUploadFileRemovalIfAlreadyRemoved() throws Exception {
        this.testFileUploadFileRemoval((Handler<RoutingContext>)((Handler)rc -> {
            this.vertx.fileSystem().deleteBlocking(((FileUpload)rc.fileUploads().iterator().next()).uploadedFileName());
            rc.response().end();
        }), true, 200, "OK");
    }

    @Test
    public void testFileDeleteOnLargeUpload() throws Exception {
        String uploadsDirectory = this.tempUploads.newFolder().getPath();
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setDeleteUploadedFilesOnEnd(true).setBodyLimit(10000L).setUploadsDirectory(uploadsDirectory));
        this.router.route().handler(ctx -> {
            this.fail();
            ctx.fail(500);
        });
        this.sendFileUploadRequest(TestUtils.randomBuffer((int)20000), 413, "Request Entity Too Large");
        BodyHandlerTest.assertWaitUntil(() -> this.vertx.fileSystem().readDirBlocking(uploadsDirectory).isEmpty());
    }

    @Test
    public void testFileUploadFileRemovalOnClientClosesConnection() throws Exception {
        String uploadsDirectory = this.tempUploads.newFolder().getPath();
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setUploadsDirectory(uploadsDirectory));
        this.assertEquals(0L, this.vertx.fileSystem().readDirBlocking(uploadsDirectory).size());
        this.client.request(HttpMethod.POST, "/").onComplete(this.onSuccess(req -> {
            String name = "somename";
            String fileName = "somefile.dat";
            String contentType = "application/octet-stream";
            String boundary = "dLV9Wyq26L_-JQxk6ferf-RT153LhOO";
            Buffer buffer = Buffer.buffer();
            String header = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"" + name + "\"; filename=\"" + fileName + "\"\r\nContent-Type: " + contentType + "\r\nContent-Transfer-Encoding: binary\r\n\r\n";
            buffer.appendString(header);
            buffer.appendBuffer(TestUtils.randomBuffer((int)50));
            req.headers().set("content-length", String.valueOf(buffer.length() + 50));
            req.headers().set("content-type", "multipart/form-data; boundary=" + boundary);
            req.write((Object)buffer);
            this.repeatWhile(100L, i -> i < 100 && this.vertx.fileSystem().readDirBlocking(uploadsDirectory).size() == 0, () -> {
                this.assertEquals(1L, this.vertx.fileSystem().readDirBlocking(uploadsDirectory).size());
                req.connection().close();
                this.repeatWhile(100L, i -> i < 100 && this.vertx.fileSystem().readDirBlocking(uploadsDirectory).size() != 0, () -> {
                    this.assertEquals(0L, this.vertx.fileSystem().readDirBlocking(uploadsDirectory).size());
                    this.testComplete();
                });
            });
        }));
        this.await();
    }

    private <T> void repeatWhile(long time, Function<Integer, Boolean> f, Runnable done) {
        this.repeatWhile(time, 0, f, done);
    }

    private <T> void repeatWhile(long time, int current, Function<Integer, Boolean> f, Runnable done) {
        if (f.apply(current).booleanValue()) {
            this.vertx.setTimer(time, id -> this.repeatWhile(time, current + 1, f, done));
        } else {
            done.run();
        }
    }

    private void testFileUploadFileRemoval(Handler<RoutingContext> requestHandler, boolean deletedUploadedFilesOnEnd, int statusCode, String statusMessage) throws Exception {
        String uploadsDirectory = this.tempUploads.newFolder().getPath();
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setDeleteUploadedFilesOnEnd(deletedUploadedFilesOnEnd).setUploadsDirectory(uploadsDirectory));
        this.router.route().handler(requestHandler);
        this.sendFileUploadRequest(TestUtils.randomBuffer((int)50), statusCode, statusMessage);
        int uploadedFilesAfterEnd = deletedUploadedFilesOnEnd ? 0 : 1;
        BodyHandlerTest.assertWaitUntil(() -> uploadedFilesAfterEnd == this.vertx.fileSystem().readDirBlocking(uploadsDirectory).size());
    }

    private void sendFileUploadRequest(Buffer fileData, int statusCode, String statusMessage) throws Exception {
        String name = "somename";
        String fileName = "somefile.dat";
        String contentType = "application/octet-stream";
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            String boundary = "dLV9Wyq26L_-JQxk6ferf-RT153LhOO";
            Buffer buffer = Buffer.buffer();
            String header = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"" + name + "\"; filename=\"" + fileName + "\"\r\nContent-Type: " + contentType + "\r\nContent-Transfer-Encoding: binary\r\n\r\n";
            buffer.appendString(header);
            buffer.appendBuffer(fileData);
            String footer = "\r\n--" + boundary + "--\r\n";
            buffer.appendString(footer);
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "multipart/form-data; boundary=" + boundary);
            req.setChunked(true);
            req.write((Object)buffer);
        }, statusCode, statusMessage, null);
    }

    @Test
    public void testFormURLEncoded() throws Exception {
        this.router.route().handler(rc -> {
            MultiMap attrs = rc.request().formAttributes();
            this.assertNotNull(attrs);
            this.assertEquals(3L, attrs.size());
            this.assertEquals("junit-testUserAlias", attrs.get("origin"));
            this.assertEquals("admin@foo.bar", attrs.get("login"));
            this.assertEquals("admin", attrs.get("pass word"));
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            Buffer buffer = Buffer.buffer();
            buffer.appendString("origin=junit-testUserAlias&login=admin%40foo.bar&pass+word=admin");
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "application/x-www-form-urlencoded");
            req.write((Object)buffer);
        }, 200, "OK", null);
    }

    @Test
    public void testFormContentTypeIgnoreCase() throws Exception {
        this.router.route().handler(rc -> {
            MultiMap attrs = rc.request().formAttributes();
            this.assertNotNull(attrs);
            this.assertEquals(1L, attrs.size());
            this.assertEquals("junit-testUserAlias", attrs.get("origin"));
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            Buffer buffer = Buffer.buffer();
            buffer.appendString("origin=junit-testUserAlias");
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "ApPlIcAtIoN/x-WwW-fOrM-uRlEnCoDeD");
            req.write((Object)buffer);
        }, 200, "OK", null);
    }

    @Test
    public void testFormMultipartFormDataMergeAttributesDefault() throws Exception {
        this.testFormMultipartFormData(true);
    }

    @Test
    public void testFormMultipartFormDataMergeAttributes() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setMergeFormAttributes(true));
        this.testFormMultipartFormData(true);
    }

    @Test
    public void testFormMultipartFormDataNoMergeAttributes() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setMergeFormAttributes(false));
        this.testFormMultipartFormData(false);
    }

    @Test
    public void testMultiFileUpload() throws Exception {
        int uploads = 1000;
        this.router.route().handler(rc -> {
            this.assertEquals(uploads, rc.fileUploads().size());
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            String boundary = "dLV9Wyq26L_-JQxk6ferf-RT153LhOO";
            Buffer buffer = Buffer.buffer();
            for (int i = 0; i < uploads; ++i) {
                String header = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"file" + i + "\"; filename=\"file" + i + "\"\r\nContent-Type: application/octet-stream\r\nContent-Transfer-Encoding: binary\r\n\r\n";
                buffer.appendString(header);
                buffer.appendBuffer(TestUtils.randomBuffer((int)65536));
                buffer.appendString("\r\n");
            }
            buffer.appendString("--" + boundary + "\r\n");
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "multipart/form-data; boundary=" + boundary);
            req.write((Object)buffer);
        }, 200, "OK", null);
    }

    private void testFormMultipartFormData(boolean mergeAttributes) throws Exception {
        this.router.route().handler(rc -> {
            MultiMap attrs = rc.request().formAttributes();
            this.assertNotNull(attrs);
            this.assertEquals(2L, attrs.size());
            this.assertEquals("Tim", attrs.get("attr1"));
            this.assertEquals("Julien", attrs.get("attr2"));
            MultiMap params = rc.request().params();
            if (mergeAttributes) {
                this.assertNotNull(params);
                this.assertEquals(3L, params.size());
                this.assertEquals("Tim", params.get("attr1"));
                this.assertEquals("Julien", params.get("attr2"));
            } else {
                this.assertNotNull(params);
                this.assertEquals(1L, params.size());
            }
            this.assertEquals("foo", params.get("p1"));
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/?p1=foo", (HttpClientRequest req) -> {
            String boundary = "dLV9Wyq26L_-JQxk6ferf-RT153LhOO";
            Buffer buffer = Buffer.buffer();
            String str = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"attr1\"\r\n\r\nTim\r\n--" + boundary + "\r\nContent-Disposition: form-data; name=\"attr2\"\r\n\r\nJulien\r\n--" + boundary + "--\r\n";
            buffer.appendString(str);
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "multipart/form-data; boundary=" + boundary);
            req.write((Object)buffer);
        }, 200, "OK", null);
    }

    @Test
    public void testMixedUploadAndForm() throws Exception {
        String uploadsDirectory = this.tempUploads.newFolder().getPath();
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setUploadsDirectory(uploadsDirectory));
        this.router.route().handler(ctx -> {
            this.assertNull(ctx.body().buffer());
            this.assertEquals(1L, ctx.fileUploads().size());
            ctx.response().end();
        });
        String name = "somename";
        String fileName = "somefile.dat";
        String contentType = "application/octet-stream";
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            String boundary = "dLV9Wyq26L_-JQxk6ferf-RT153LhOO";
            Buffer buffer = Buffer.buffer();
            String header = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"" + name + "\"; filename=\"" + fileName + "\"\r\nContent-Type: " + contentType + "\r\nContent-Transfer-Encoding: binary\r\n\r\n";
            buffer.appendString(header);
            buffer.appendBuffer(TestUtils.randomBuffer((int)50));
            String footer = "\r\n--" + boundary + "--\r\n";
            buffer.appendString(footer);
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "multipart/form-data; boundary=" + boundary);
            req.write((Object)buffer);
        }, 200, "OK", "");
    }

    @Test
    public void testNoUploadDirMultiPartFormData() throws Exception {
        String dirName = this.getNotCreatedTemporaryFolderName();
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create((boolean)false).setUploadsDirectory(dirName));
        Buffer fileData = TestUtils.randomBuffer((int)50);
        this.router.route().handler(rc -> {
            rc.response().end();
            this.assertFalse("Upload directory must not be created.", this.vertx.fileSystem().existsBlocking(dirName));
        });
        this.sendFileUploadRequest(fileData, 200, "OK");
    }

    @Test
    public void testFormMultipartFormDataWithAllowedFilesUploadFalse1() throws Exception {
        this.testFormMultipartFormDataWithAllowedFilesUploadFalse(true);
    }

    @Test
    public void testFormMultipartFormDataWithAllowedFilesUploadFalse2() throws Exception {
        this.testFormMultipartFormDataWithAllowedFilesUploadFalse(false);
    }

    private void testFormMultipartFormDataWithAllowedFilesUploadFalse(boolean mergeAttributes) throws Exception {
        String fileName = "test.bin";
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create((boolean)false).setMergeFormAttributes(mergeAttributes)).handler(rc -> {
            MultiMap attrs = rc.request().formAttributes();
            this.assertNotNull(attrs);
            this.assertEquals(2L, attrs.size());
            this.assertEquals("Tim", attrs.get("attr1"));
            this.assertEquals("Tommaso", attrs.get("attr2"));
            MultiMap params = rc.request().params();
            this.assertEquals(0L, rc.fileUploads().size());
            if (mergeAttributes) {
                this.assertNotNull(params);
                this.assertEquals(3L, params.size());
                this.assertEquals("Tim", params.get("attr1"));
                this.assertEquals("Tommaso", params.get("attr2"));
                this.assertEquals("foo", params.get("p1"));
            } else {
                this.assertNotNull(params);
                this.assertEquals(1L, params.size());
                this.assertEquals("foo", params.get("p1"));
                this.assertEquals("Tim", rc.request().getFormAttribute("attr1"));
                this.assertEquals("Tommaso", rc.request().getFormAttribute("attr2"));
            }
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/?p1=foo", (HttpClientRequest req) -> {
            Buffer buffer = Buffer.buffer();
            String boundary = "dLV9Wyq26L_-JQxk6ferf-RT153LhOO";
            String header = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"attr1\"\r\n\r\nTim\r\n--" + boundary + "\r\nContent-Disposition: form-data; name=\"attr2\"\r\n\r\nTommaso\r\n--" + boundary + "\r\nContent-Disposition: form-data; name=\"" + this.name + "\"; filename=\"" + fileName + "\"\r\nContent-Type: application/octet-stream\r\nContent-Transfer-Encoding: binary\r\n\r\n";
            buffer.appendString(header);
            buffer.appendBuffer(TestUtils.randomBuffer((int)50));
            buffer.appendString("\r\n--" + boundary + "--\r\n");
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "multipart/form-data; boundary=" + boundary);
            req.write((Object)buffer);
        }, 200, "OK", null);
    }

    @Test
    public void testNoUploadDirFormURLEncoded() throws Exception {
        String dirName = this.getNotCreatedTemporaryFolderName();
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create((boolean)false).setUploadsDirectory(dirName));
        this.testFormURLEncoded();
        this.assertFalse("Upload directory must not be created.", this.vertx.fileSystem().existsBlocking(dirName));
    }

    @Test
    public void testBodyHandlerCreateTrueWorks() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create((boolean)true));
        this.testFormURLEncoded();
    }

    @Test
    public void testSetHandleFileUploads() throws Exception {
        String dirName = this.getNotCreatedTemporaryFolderName();
        this.router.clear();
        BodyHandler bodyHandler = BodyHandler.create().setUploadsDirectory(dirName).setHandleFileUploads(false);
        this.router.route().handler((Handler)bodyHandler);
        Buffer fileData = TestUtils.randomBuffer((int)50);
        Route route = this.router.route().handler(rc -> {
            rc.response().end();
            this.assertFalse("Upload directory must not be created.", this.vertx.fileSystem().existsBlocking(dirName));
        });
        this.sendFileUploadRequest(fileData, 200, "OK");
        route.remove();
        bodyHandler.setHandleFileUploads(true);
        this.router.route().handler(rc -> {
            rc.response().end();
            this.assertTrue("Upload directory must be created.", this.vertx.fileSystem().existsBlocking(dirName));
        });
        this.sendFileUploadRequest(fileData, 200, "OK");
    }

    @Test
    public void testRerouteWithHandleFileUploadsFalse() throws Exception {
        String fileName = "test.bin";
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create((boolean)false).setMergeFormAttributes(true));
        this.router.route("/toBeRerouted").handler(rc -> rc.reroute("/rerouted"));
        this.router.route("/rerouted").handler(rc -> {
            MultiMap attrs = rc.request().formAttributes();
            this.assertNotNull(attrs);
            this.assertEquals(2L, attrs.size());
            this.assertEquals("Tim", attrs.get("attr1"));
            this.assertEquals("Tommaso", attrs.get("attr2"));
            MultiMap params = rc.request().params();
            this.assertEquals(0L, rc.fileUploads().size());
            this.assertNotNull(params);
            this.assertEquals(2L, params.size());
            this.assertEquals("Tim", params.get("attr1"));
            this.assertEquals("Tommaso", params.get("attr2"));
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/toBeRerouted", (HttpClientRequest req) -> {
            Buffer buffer = Buffer.buffer();
            String boundary = "dLV9Wyq26L_-JQxk6ferf-RT153LhOO";
            String header = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"attr1\"\r\n\r\nTim\r\n--" + boundary + "\r\nContent-Disposition: form-data; name=\"attr2\"\r\n\r\nTommaso\r\n--" + boundary + "\r\nContent-Disposition: form-data; name=\"" + this.name + "\"; filename=\"" + fileName + "\"\r\nContent-Type: application/octet-stream\r\nContent-Transfer-Encoding: binary\r\n\r\n";
            buffer.appendString(header);
            buffer.appendBuffer(TestUtils.randomBuffer((int)50));
            buffer.appendString("\r\n--" + boundary + "--\r\n");
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "multipart/form-data; boundary=" + boundary);
            req.write((Object)buffer);
        }, 200, "OK", null);
    }

    @Test
    public void testBodyLimitWithHandleFileUploadsFalse() throws Exception {
        this.router.clear();
        BodyHandler bodyHandler = BodyHandler.create((boolean)false).setBodyLimit(2048L);
        this.router.route().handler((Handler)bodyHandler);
        Buffer fileData = TestUtils.randomBuffer((int)4096);
        this.router.route().handler(rc -> rc.response().end());
        this.sendFileUploadRequest(fileData, 413, "Request Entity Too Large");
    }

    private String getNotCreatedTemporaryFolderName() throws IOException {
        File dir = this.tempUploads.newFolder();
        dir.delete();
        return dir.getPath();
    }

    @Test
    public void testFomWithoutParamNameRequestForm() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create());
        Buffer buffer = Buffer.buffer((String)"a=b&=&c=d");
        this.router.route().handler(rc -> this.fail("Should not be called"));
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.putHeader("content-type", "application/x-www-form-urlencoded");
            req.write((Object)buffer);
        }, 400, "Bad Request", null);
    }

    @Test
    public void testFomWithoutParamRequestForm() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create());
        Buffer buffer = Buffer.buffer((String)"a=b&&c=d");
        this.router.route().handler(RoutingContext::end);
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.putHeader("content-type", "application/x-www-form-urlencoded");
            req.write((Object)buffer);
        }, 200, "OK", null);
    }

    @Test
    public void testJsonLimit() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create());
        Buffer buffer = Buffer.buffer((String)"000000000000000000000000000000000000000000000000");
        this.router.route().handler(rc -> {
            try {
                rc.body().asJsonObject(10);
                rc.fail(500);
            }
            catch (IllegalStateException e) {
                rc.fail(413);
            }
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.putHeader("content-type", "application/json");
            req.write((Object)buffer);
        }, 413, "Request Entity Too Large", null);
    }

    @Test
    public void testJsonLimitOK() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create());
        Buffer buffer = Buffer.buffer((String)"{\"k\":1111}");
        this.router.route().handler(rc -> {
            try {
                rc.body().asJsonObject(10);
                rc.end();
            }
            catch (IllegalStateException e) {
                rc.fail(500);
            }
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            req.setChunked(true);
            req.putHeader("content-type", "application/json");
            req.write((Object)buffer);
        }, 200, "OK", null);
    }

    @Test
    public void testFileUploadUTF8() throws Exception {
        String name = "somename";
        String fileName = "somefile.dat";
        String contentType = "application/octet-stream";
        Buffer fileData = TestUtils.randomBuffer((int)50);
        this.router.route().handler(rc -> {
            List fileUploads = rc.fileUploads();
            this.assertNotNull(fileUploads);
            this.assertEquals(1L, fileUploads.size());
            FileUpload upload = (FileUpload)fileUploads.iterator().next();
            this.assertEquals(name, upload.name());
            this.assertEquals("\u00a3 and \u20ac " + fileName, upload.fileName());
            this.assertEquals(contentType, upload.contentType());
            this.assertEquals("binary", upload.contentTransferEncoding());
            this.assertEquals(fileData.length(), upload.size());
            String uploadedFileName = upload.uploadedFileName();
            this.assertTrue(uploadedFileName.startsWith("file-uploads" + File.separator));
            Buffer uploaded = this.vertx.fileSystem().readFileBlocking(uploadedFileName);
            this.assertEquals(fileData, uploaded);
            Buffer rawBody = rc.body().buffer();
            this.assertNull(rawBody);
            rc.response().end();
        });
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            String boundary = "dLV9Wyq26L_-JQxk6ferf-RT153LhOO";
            Buffer buffer = Buffer.buffer();
            String header = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"" + name + "\"; filename*=\"UTF-8''%c2%a3%20and%20%e2%82%ac%20" + fileName + "\"\r\nContent-Type: " + contentType + "\r\nContent-Transfer-Encoding: binary\r\n\r\n";
            buffer.appendString(header);
            buffer.appendBuffer(fileData);
            String footer = "\r\n--" + boundary + "--\r\n";
            buffer.appendString(footer);
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "multipart/form-data; boundary=" + boundary);
            req.setChunked(true);
            req.write((Object)buffer);
        }, 200, "OK", null);
    }

    @Test
    public void testFormMultipartFormDataLarge() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create());
        this.router.route().handler(rc -> this.fail("Should not get here")).failureHandler(ctx -> {
            this.assertNotNull(ctx.failure());
            this.assertTrue(ctx.failure() instanceof IOException);
            this.assertEquals("Size exceed allowed maximum capacity", ctx.failure().getMessage());
            ctx.next();
        });
        this.testRequest(HttpMethod.POST, "/?p1=foo", (HttpClientRequest req) -> {
            Buffer buffer = Buffer.buffer();
            String boundary = "dLV9Wyq26L_-JQxk6ferf-RT153LhOO";
            String header = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"attr1\"\r\n\r\n" + Base64.getUrlEncoder().encodeToString(TestUtils.randomBuffer((int)8192).getBytes()) + "\r\n--" + boundary + "\r\nContent-Disposition: form-data; name=\"attr2\"\r\n\r\n" + Base64.getUrlEncoder().encodeToString(TestUtils.randomBuffer((int)8192).getBytes()) + "\r\n--" + boundary + "\r\nContent-Disposition: form-data; name=\"" + this.name + "\"; filename=\"file\"\r\nContent-Type: application/octet-stream\r\nContent-Transfer-Encoding: binary\r\n\r\n";
            buffer.appendString(header);
            buffer.appendBuffer(TestUtils.randomBuffer((int)50));
            buffer.appendString("\r\n--" + boundary + "--\r\n");
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "multipart/form-data; boundary=" + boundary);
            req.write((Object)buffer);
        }, 400, "Bad Request", null);
    }

    @Test
    public void testLogExceptions() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create());
        this.router.route().handler(ctx -> {
            throw new NullPointerException();
        });
        this.testRequest(HttpMethod.GET, "/", (HttpClientRequest req) -> {}, 500, "Internal Server Error", null);
    }

    @Test
    public void testFileUploadSize() throws Exception {
        String uploadsDirectory = this.tempUploads.newFolder().getPath();
        this.router.clear();
        this.router.route().handler((Handler)BodyHandler.create().setDeleteUploadedFilesOnEnd(true).setUploadsDirectory(uploadsDirectory));
        int realSize = 20000;
        this.router.route().handler(ctx -> {
            String specData = ctx.request().formAttributes().get("specData");
            System.out.println(specData);
            FileUpload file = (FileUpload)ctx.fileUploads().iterator().next();
            long uploadSize = file.size();
            this.assertEquals(realSize, uploadSize);
            ctx.end();
        });
        String name = "file";
        String fileName = "/C:/Users/vishal.b05/Desktop/1p.png";
        String contentType = "application/octet-stream";
        this.testRequest(HttpMethod.POST, "/", (HttpClientRequest req) -> {
            String boundary = "dLV9Wyq26L_-JQxk6ferf-RT153LhOO";
            Buffer buffer = Buffer.buffer();
            String header = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"" + name + "\"; filename=\"" + fileName + "\"\r\nContent-Type: " + contentType + "\r\n\r\n";
            buffer.appendString(header);
            buffer.appendBuffer(TestUtils.randomBuffer((int)realSize));
            String footer = "\r\n--" + boundary + "\r\n";
            buffer.appendString(footer);
            String extra = "Content-Disposition: form-data; name=\"specData\"\r\n\r\n{\"id\":\"abc@xyz.com\"}\r\n--" + boundary + "--\r\n\r\n";
            buffer.appendString(extra);
            req.headers().set("content-length", String.valueOf(buffer.length()));
            req.headers().set("content-type", "multipart/form-data; boundary=" + boundary);
            req.setChunked(true);
            req.write((Object)buffer);
        }, 200, "OK", null);
        BodyHandlerTest.assertWaitUntil(() -> this.vertx.fileSystem().readDirBlocking(uploadsDirectory).isEmpty());
    }
}

