package com.wepin.android.commonlib.error

import org.json.JSONObject

class WepinError : Exception {
    var errorReason: JSONObject? = null
    var errorMessage: String? = null
    var code: Int = 0

    constructor() : super() {
        this.errorReason = null
        this.errorMessage = null
        this.code = 0
    }

    constructor(response: JSONObject?) : super() {
        this.errorReason = response
        this.errorMessage = null
        this.code = 0
    }

    constructor(exceptionMessage: String?) : super(exceptionMessage) {
        this.errorReason = null
        this.errorMessage = exceptionMessage
        this.code = 0
    }

    constructor(exceptionMessage: String?, reason: Throwable?) : super(exceptionMessage, reason) {
        this.errorReason = null
        this.errorMessage = exceptionMessage
        this.code = 0
    }

    constructor(cause: Throwable?) : super(cause) {
        this.errorReason = null
        this.errorMessage = cause?.message
        this.code = 0
    }

    constructor(code: Int, errorDescription: String) : super(errorDescription) {
        this.code = code
        this.errorMessage = errorDescription
    }

    fun setErrorCode(errCode: Int) {
        this.code = errCode
    }

    fun getErrorCode(): Int = code

    companion object {
        enum class ErrorCode(val message: String) {
            INVALID_APP_KEY("Invalid app key"),
            INVALID_PARAMETER("Invalid parameter"),
            INVALID_LOGIN_PROVIDER("Invalid login provider"),
            INVALID_TOKEN("Token does not exist"),
            INVALID_LOGIN_SESSION("Invalid login session"),
            NOT_INITIALIZED_ERROR("Not initialized error"),
            ALREADY_INITIALIZED_ERROR("Already initialized"),
            USER_CANCELLED("User cancelled"),
            UNKNOWN_ERROR("Unknown error"),
            NOT_CONNECTED_INTERNET("No internet connection"),
            FAILED_LOGIN("Failed OAuth login"),
            ALREADY_LOGOUT("Already logged out"),
            INVALID_EMAIL_DOMAIN("Invalid email domain"),
            FAILED_SEND_EMAIL("Failed to send email"),
            REQUIRED_EMAIL_VERIFIED("Email verification required"),
            INCORRECT_EMAIL_FORM("Incorrect email format"),
            INCORRECT_PASSWORD_FORM("Incorrect password format"),
            NOT_INITIALIZED_NETWORK("Network manager not initialized"),
            REQUIRED_SIGNUP_EMAIL("Email sign-up required"),
            FAILED_EMAIL_VERIFICATION("Failed email verification"),
            FAILED_PASSWORD_STATE_SETTING("Failed password state setting"),
            FAILED_PASSWORD_SETTING("Failed to set password"),
            EXISTED_EMAIL("Email already exists"),
            API_REQUEST_ERROR("API request error"),
            INCORRECT_LIFE_CYCLE("Incorrect life cycle"),
            FAILED_REGISTER("Failed to register"),
            ACCOUNT_NOT_FOUND("Account not found"),
            NFT_NOT_FOUND("NFT not found"),
            BALANCES_NOT_FOUND("Balances not found"),
            FAILED_SEND("Failed to send"),
            FAILED_RECEIVE("Failed to receive"),
            DEPRECATED("This method is deprecated"),
            NOT_ACTIVITY("Context is not an activity"),
            NOT_SUPPORT_NETWORK("Not support network")
        }

        // Static error instances
        val INVALID_APP_KEY = generalEx(ErrorCode.INVALID_APP_KEY)
        val INVALID_PARAMETER = generalEx(ErrorCode.INVALID_PARAMETER)
        val INVALID_LOGIN_PROVIDER = generalEx(ErrorCode.INVALID_LOGIN_PROVIDER)
        val INVALID_TOKEN = generalEx(ErrorCode.INVALID_TOKEN)
        val INVALID_LOGIN_SESSION = generalEx(ErrorCode.INVALID_LOGIN_SESSION)
        val NOT_INITIALIZED_ERROR = generalEx(ErrorCode.NOT_INITIALIZED_ERROR)
        val ALREADY_INITIALIZED_ERROR = generalEx(ErrorCode.ALREADY_INITIALIZED_ERROR)
        val USER_CANCELED = generalEx(ErrorCode.USER_CANCELLED)
        val UNKNOWN_ERROR = generalEx(ErrorCode.UNKNOWN_ERROR)
        val NOT_CONNECTED_INTERNET = generalEx(ErrorCode.NOT_CONNECTED_INTERNET)
        val FAILED_LOGIN = generalEx(ErrorCode.FAILED_LOGIN)
        val ALREADY_LOGOUT = generalEx(ErrorCode.ALREADY_LOGOUT)
        val INVALID_EMAIL_DOMAIN = generalEx(ErrorCode.INVALID_EMAIL_DOMAIN)
        val FAILED_SEND_EMAIL = generalEx(ErrorCode.FAILED_SEND_EMAIL)
        val REQUIRED_EMAIL_VERIFIED = generalEx(ErrorCode.REQUIRED_EMAIL_VERIFIED)
        val INCORRECT_EMAIL_FORM = generalEx(ErrorCode.INCORRECT_EMAIL_FORM)
        val INCORRECT_PASSWORD_FORM = generalEx(ErrorCode.INCORRECT_PASSWORD_FORM)
        val NOT_INITIALIZED_NETWORK = generalEx(ErrorCode.NOT_INITIALIZED_NETWORK)
        val REQUIRED_SIGNUP_EMAIL = generalEx(ErrorCode.REQUIRED_SIGNUP_EMAIL)
        val FAILED_EMAIL_VERIFICATION = generalEx(ErrorCode.FAILED_EMAIL_VERIFICATION)
        val FAILED_PASSWORD_STATE_SETTING = generalEx(ErrorCode.FAILED_PASSWORD_STATE_SETTING)
        val FAILED_PASSWORD_SETTING = generalEx(ErrorCode.FAILED_PASSWORD_SETTING)
        val EXISTED_EMAIL = generalEx(ErrorCode.EXISTED_EMAIL)
        val API_REQUEST_ERROR = generalEx(ErrorCode.API_REQUEST_ERROR)
        val INCORRECT_LIFE_CYCLE = generalEx(ErrorCode.INCORRECT_LIFE_CYCLE)
        val FAILED_REGISTER = generalEx(ErrorCode.FAILED_REGISTER)
        val ACCOUNT_NOT_FOUND = generalEx(ErrorCode.ACCOUNT_NOT_FOUND)
        val NFT_NOT_FOUND = generalEx(ErrorCode.NFT_NOT_FOUND)
        val BALANCES_NOT_FOUND = generalEx(ErrorCode.BALANCES_NOT_FOUND)
        val FAILED_SEND = generalEx(ErrorCode.FAILED_SEND)
        val FAILED_RECEIVE = generalEx(ErrorCode.FAILED_RECEIVE)
        val DEPRECATED = generalEx(ErrorCode.DEPRECATED)
        val NOT_ACTIVITY = generalEx(ErrorCode.NOT_ACTIVITY)
        val NOT_SUPPORT_NETWORK = generalEx(ErrorCode.NOT_SUPPORT_NETWORK)

        fun getError(errorCode: ErrorCode, message: String? = null): String {
            return if (errorCode == ErrorCode.UNKNOWN_ERROR && message != null) {
                "${errorCode.message}: $message"
            } else {
                errorCode.message
            }
        }

        private fun generalEx(errorCode: ErrorCode): WepinError {
            return WepinError(errorCode.ordinal, errorCode.message)
        }

        fun generalApiRequestError(message: String?): WepinError {
            return WepinError(
                ErrorCode.API_REQUEST_ERROR.ordinal,
                message ?: ErrorCode.API_REQUEST_ERROR.message
            )
        }

        fun generalIncorrectLifeCycle(message: String?): WepinError {
            return WepinError(
                ErrorCode.INCORRECT_LIFE_CYCLE.ordinal,
                message ?: ErrorCode.INCORRECT_LIFE_CYCLE.message
            )
        }

        fun generalAccountNotFound(message: String?): WepinError {
            return WepinError(
                ErrorCode.ACCOUNT_NOT_FOUND.ordinal,
                message ?: ErrorCode.ACCOUNT_NOT_FOUND.message
            )
        }

        fun generalInvalidParameter(message: String?): WepinError {
            return WepinError(
                ErrorCode.INVALID_PARAMETER.ordinal,
                message ?: ErrorCode.INVALID_PARAMETER.message
            )
        }

        fun generalUnKnownEx(message: String?): WepinError {
            return WepinError(
                ErrorCode.UNKNOWN_ERROR.ordinal,
                getError(ErrorCode.UNKNOWN_ERROR, message)
            )
        }

        // Map Webview error messages to WepinError
        fun mapWebviewErrorToWepinError(errorMessage: String): WepinError {
            return when (errorMessage) {
                "User Cancel" -> USER_CANCELED
                "Invalid App Key" -> INVALID_APP_KEY
                "Invalid Parameter" -> INVALID_PARAMETER
                "Invalid Login Session" -> INVALID_LOGIN_SESSION
                "Not Initialized" -> NOT_INITIALIZED_ERROR
                "Already Initialized" -> ALREADY_INITIALIZED_ERROR
                "Network Error" -> NOT_CONNECTED_INTERNET
                "Failed Login" -> FAILED_LOGIN
                else -> generalUnKnownEx(errorMessage)
            }
        }
    }
}