/*******************************************************************************
 * Copyright 2017 xlate.io LLC, http://www.xlate.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 ******************************************************************************/
package io.xlate.edi.internal.stream.validation;

import java.util.List;

import io.xlate.edi.internal.stream.tokenization.Dialect;
import io.xlate.edi.internal.stream.tokenization.EDIException;
import io.xlate.edi.schema.EDISimpleType;
import io.xlate.edi.stream.EDIStreamEvent;
import io.xlate.edi.stream.EDIStreamValidationError;
import io.xlate.edi.stream.EDIValidationException;

abstract class ElementValidator {

    static ElementValidator getInstance(EDISimpleType.Base type) {
        ElementValidator instance;

        switch (type) {
        case IDENTIFIER:
        case STRING:
            instance = AlphaNumericValidator.getInstance();
            break;
        case NUMERIC:
            instance = NumericValidator.getInstance();
            break;
        case DECIMAL:
            instance = DecimalValidator.getInstance();
            break;
        case DATE:
            instance = DateValidator.getInstance();
            break;
        case TIME:
            instance = TimeValidator.getInstance();
            break;
        case BINARY:
        default:
            instance = null;
            break;
        }

        return instance;
    }

    protected static boolean validateLength(Dialect dialect,
                                            EDISimpleType element,
                                            int length,
                                            List<EDIStreamValidationError> errors) {

        final String version = dialect.getTransactionVersionString();

        if (length > element.getMaxLength(version)) {
            errors.add(EDIStreamValidationError.DATA_ELEMENT_TOO_LONG);
            return false;
        } else if (length < element.getMinLength(version)) {
            errors.add(EDIStreamValidationError.DATA_ELEMENT_TOO_SHORT);
            return false;
        }

        return true;
    }

    protected static void assertMinLength(EDISimpleType element, CharSequence value) {
        if (value.length() < element.getMinLength()) {
            throw new EDIValidationException(EDIStreamEvent.ELEMENT_DATA, EDIStreamValidationError.DATA_ELEMENT_TOO_SHORT, null, value);
        }
    }

    protected static void assertMaxLength(EDISimpleType element, int length, CharSequence value) {
        if (length > element.getMaxLength()) {
            throw new EDIValidationException(EDIStreamEvent.ELEMENT_DATA, EDIStreamValidationError.DATA_ELEMENT_TOO_LONG, null, value);
        }
    }

    protected static void assertMaxLength(EDISimpleType element, CharSequence value) {
        assertMaxLength(element, value.length(), value);
    }

    abstract void validate(Dialect dialect,
                           EDISimpleType element,
                           CharSequence value,
                           List<EDIStreamValidationError> errors);

    abstract void format(Dialect dialect,
                         EDISimpleType element,
                         CharSequence value,
                         Appendable result)
            throws EDIException;
}
