/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.engine.processor;

import io.zeebe.engine.processor.SnapshotMetrics;
import io.zeebe.engine.processor.StreamProcessor;
import io.zeebe.logstreams.impl.Loggers;
import io.zeebe.logstreams.log.LogStream;
import io.zeebe.logstreams.spi.SnapshotController;
import io.zeebe.util.sched.Actor;
import io.zeebe.util.sched.ActorCondition;
import io.zeebe.util.sched.ActorControl;
import io.zeebe.util.sched.SchedulingHints;
import io.zeebe.util.sched.future.ActorFuture;
import io.zeebe.util.sched.future.CompletableActorFuture;
import java.time.Duration;
import org.slf4j.Logger;

public class AsyncSnapshotDirector
extends Actor {
    private static final Logger LOG = Loggers.SNAPSHOT_LOGGER;
    private static final String LOG_MSG_WAIT_UNTIL_COMMITTED = "Finished taking snapshot, need to wait until last written event position {} is committed, current commit position is {}. After that snapshot can be marked as valid.";
    private static final String ERROR_MSG_ON_RESOLVE_PROCESSED_POS = "Unexpected error in resolving last processed position.";
    private static final String ERROR_MSG_ON_RESOLVE_WRITTEN_POS = "Unexpected error in resolving last written position.";
    private static final String ERROR_MSG_MOVE_SNAPSHOT = "Unexpected exception occurred on moving valid snapshot.";
    private static final String LOG_MSG_ENFORCE_SNAPSHOT = "Enforce snapshot creation. Last successful processed position is {}.";
    private static final String ERROR_MSG_ENFORCED_SNAPSHOT = "Unexpected exception occurred on creating snapshot, was enforced to do so.";
    private static final int INITIAL_POSITION = -1;
    private final Runnable prepareTakingSnapshot = this::prepareTakingSnapshot;
    private final SnapshotController snapshotController;
    private final LogStream logStream;
    private final String name;
    private final Duration snapshotRate;
    private final SnapshotMetrics metrics;
    private final String processorName;
    private final StreamProcessor streamProcessor;
    private ActorCondition commitCondition;
    private long lastWrittenEventPosition = -1L;
    private boolean pendingSnapshot;
    private long lowerBoundSnapshotPosition;
    private long lastValidSnapshotPosition;

    public AsyncSnapshotDirector(StreamProcessor streamProcessor, SnapshotController snapshotController, LogStream logStream, Duration snapshotRate, SnapshotMetrics metrics) {
        this.streamProcessor = streamProcessor;
        this.snapshotController = snapshotController;
        this.logStream = logStream;
        this.processorName = streamProcessor.getName();
        this.name = this.processorName + "-snapshot-director";
        this.snapshotRate = snapshotRate;
        this.metrics = metrics;
    }

    protected void onActorStarting() {
        this.actor.setSchedulingHints(SchedulingHints.ioBound());
        this.actor.runAtFixedRate(this.snapshotRate, this.prepareTakingSnapshot);
        this.commitCondition = this.actor.onCondition(this.getConditionNameForPosition(), this::onCommitCheck);
        this.logStream.registerOnCommitPositionUpdatedCondition(this.commitCondition);
        this.lastValidSnapshotPosition = this.snapshotController.getLastValidSnapshotPosition();
        LOG.debug("The position of the last valid snapshot is '{}'. Taking snapshots beyond this position.", (Object)this.lastValidSnapshotPosition);
    }

    public String getName() {
        return this.name;
    }

    private String getConditionNameForPosition() {
        return this.getName() + "-wait-for-endPosition-committed";
    }

    private void prepareTakingSnapshot() {
        if (this.pendingSnapshot) {
            return;
        }
        ActorFuture<Long> lastProcessedPosition = this.streamProcessor.getLastProcessedPositionAsync();
        this.actor.runOnCompletion(lastProcessedPosition, (lowerBoundSnapshotPosition, error) -> {
            if (error == null) {
                if (lowerBoundSnapshotPosition > this.lastValidSnapshotPosition) {
                    this.lowerBoundSnapshotPosition = lowerBoundSnapshotPosition;
                    this.takeSnapshot();
                } else {
                    LOG.debug("No changes since last snapshot we will skip snapshot creation. Last valid snapshot position {}, new lower bound position {}", (Object)this.lastValidSnapshotPosition, lowerBoundSnapshotPosition);
                }
            } else {
                LOG.error(ERROR_MSG_ON_RESOLVE_PROCESSED_POS, error);
            }
        });
    }

    private void takeSnapshot() {
        this.pendingSnapshot = true;
        this.createSnapshot(() -> ((SnapshotController)this.snapshotController).takeTempSnapshot());
        ActorFuture<Long> lastWrittenPosition = this.streamProcessor.getLastWrittenPositionAsync();
        this.actor.runOnCompletion(lastWrittenPosition, (endPosition, error) -> {
            if (error == null) {
                long commitPosition = this.logStream.getCommitPosition();
                this.lastWrittenEventPosition = endPosition;
                LOG.debug(LOG_MSG_WAIT_UNTIL_COMMITTED, endPosition, (Object)commitPosition);
                this.onCommitCheck();
            } else {
                this.pendingSnapshot = false;
                LOG.error(ERROR_MSG_ON_RESOLVE_WRITTEN_POS, error);
            }
        });
    }

    private void createSnapshot(Runnable snapshotCreation) {
        long start = System.currentTimeMillis();
        snapshotCreation.run();
        long end = System.currentTimeMillis();
        long snapshotCreationTime = end - start;
        LOG.info("Creation of snapshot for {} took {} ms.", (Object)this.processorName, (Object)snapshotCreationTime);
        this.metrics.recordSnapshotCreationTime(snapshotCreationTime);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void onCommitCheck() {
        long currentCommitPosition = this.logStream.getCommitPosition();
        if (this.pendingSnapshot && currentCommitPosition >= this.lastWrittenEventPosition) {
            try {
                this.lastValidSnapshotPosition = this.lowerBoundSnapshotPosition;
                this.snapshotController.moveValidSnapshot(this.lowerBoundSnapshotPosition);
                this.snapshotController.replicateLatestSnapshot(arg_0 -> ((ActorControl)this.actor).submit(arg_0));
            }
            catch (Exception ex) {
                LOG.error(ERROR_MSG_MOVE_SNAPSHOT, (Throwable)ex);
            }
            finally {
                this.pendingSnapshot = false;
            }
        }
    }

    protected void enforceSnapshotCreation(long lastWrittenPosition, long lastProcessedPosition) {
        long commitPosition = this.logStream.getCommitPosition();
        if (commitPosition >= lastWrittenPosition && lastProcessedPosition > this.lastValidSnapshotPosition) {
            LOG.debug(LOG_MSG_ENFORCE_SNAPSHOT, (Object)lastProcessedPosition);
            try {
                this.createSnapshot(() -> this.snapshotController.takeSnapshot(lastProcessedPosition));
            }
            catch (Exception ex) {
                LOG.error(ERROR_MSG_ENFORCED_SNAPSHOT, (Throwable)ex);
            }
        }
    }

    protected void onActorCloseRequested() {
        this.logStream.removeOnCommitPositionUpdatedCondition(this.commitCondition);
    }

    public ActorFuture<Void> closeAsync() {
        CompletableActorFuture future = new CompletableActorFuture();
        this.actor.call(() -> this.actor.runOnCompletion(this.streamProcessor.getLastWrittenPositionAsync(), (writtenPosition, ex1) -> {
            if (ex1 == null) {
                this.actor.runOnCompletion(this.streamProcessor.getLastProcessedPositionAsync(), (processedPosition, ex2) -> {
                    if (ex2 == null) {
                        this.enforceSnapshotCreation((long)writtenPosition, (long)processedPosition);
                        this.close();
                        future.complete(null);
                    } else {
                        LOG.error(ERROR_MSG_ON_RESOLVE_PROCESSED_POS, ex2);
                        this.close();
                        future.completeExceptionally(ex2);
                    }
                });
            } else {
                LOG.error(ERROR_MSG_ON_RESOLVE_WRITTEN_POS, ex1);
                this.close();
                future.completeExceptionally(ex1);
            }
        }));
        return future;
    }

    private void close() {
        this.metrics.close();
        this.actor.close();
    }
}

