/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.engine.processor;

import io.zeebe.db.DbContext;
import io.zeebe.db.ZeebeDbTransaction;
import io.zeebe.engine.processor.EventFilter;
import io.zeebe.engine.processor.ProcessingContext;
import io.zeebe.engine.processor.RecordProcessorMap;
import io.zeebe.engine.processor.SideEffectProducer;
import io.zeebe.engine.processor.StreamProcessorMetrics;
import io.zeebe.engine.processor.TypedEventImpl;
import io.zeebe.engine.processor.TypedRecordProcessor;
import io.zeebe.engine.processor.TypedResponseWriterImpl;
import io.zeebe.engine.processor.TypedStreamWriter;
import io.zeebe.engine.state.ZeebeState;
import io.zeebe.logstreams.impl.Loggers;
import io.zeebe.logstreams.log.LogStream;
import io.zeebe.logstreams.log.LogStreamReader;
import io.zeebe.logstreams.log.LoggedEvent;
import io.zeebe.msgpack.UnpackedObject;
import io.zeebe.protocol.impl.record.RecordMetadata;
import io.zeebe.protocol.impl.record.UnifiedRecordValue;
import io.zeebe.protocol.impl.record.value.error.ErrorRecord;
import io.zeebe.protocol.record.RecordType;
import io.zeebe.protocol.record.RejectionType;
import io.zeebe.protocol.record.ValueType;
import io.zeebe.protocol.record.intent.ErrorIntent;
import io.zeebe.protocol.record.intent.Intent;
import io.zeebe.util.buffer.BufferReader;
import io.zeebe.util.exception.RecoverableException;
import io.zeebe.util.retry.AbortableRetryStrategy;
import io.zeebe.util.retry.RecoverableRetryStrategy;
import io.zeebe.util.retry.RetryStrategy;
import io.zeebe.util.sched.ActorControl;
import io.zeebe.util.sched.future.ActorFuture;
import java.time.Duration;
import java.util.Map;
import java.util.function.BooleanSupplier;
import org.slf4j.Logger;

public final class ProcessingStateMachine {
    private static final Logger LOG = Loggers.PROCESSOR_LOGGER;
    public static final String ERROR_MESSAGE_WRITE_EVENT_ABORTED = "Expected to write one or more follow up events for event '{}' without errors, but exception was thrown.";
    private static final String ERROR_MESSAGE_ROLLBACK_ABORTED = "Expected to roll back the current transaction for event '{}' successfully, but exception was thrown.";
    private static final String ERROR_MESSAGE_EXECUTE_SIDE_EFFECT_ABORTED = "Expected to execute side effects for event '{}' successfully, but exception was thrown.";
    private static final String ERROR_MESSAGE_UPDATE_STATE_FAILED = "Expected to successfully update state for event '{}' with processor '{}', but caught an exception. Retry.";
    private static final String ERROR_MESSAGE_ON_EVENT_FAILED_SKIP_EVENT = "Expected to find event processor for event '{}' with processor '{}', but caught an exception. Skip this event.";
    private static final String ERROR_MESSAGE_PROCESSING_FAILED_SKIP_EVENT = "Expected to successfully process event '{}' with processor '{}', but caught an exception. Skip this event.";
    private static final String ERROR_MESSAGE_PROCESSING_FAILED_RETRY_PROCESSING = "Expected to process event '{}' successfully on stream processor '{}', but caught recoverable exception. Retry processing.";
    private static final String PROCESSING_ERROR_MESSAGE = "Expected to process event '%s' without errors, but exception occurred with message '%s' .";
    private static final String LOG_ERROR_EVENT_COMMITTED = "Error event was committed, we continue with processing.";
    private static final String LOG_ERROR_EVENT_WRITTEN = "Error record was written at {}, we will continue with processing if event was committed. Current commit position is {}.";
    private static final Duration PROCESSING_RETRY_DELAY = Duration.ofMillis(250L);
    private final ActorControl actor;
    private final int producerId;
    private final String streamProcessorName;
    private final StreamProcessorMetrics metrics;
    private final EventFilter eventFilter;
    private final LogStream logStream;
    private final LogStreamReader logStreamReader;
    private final TypedStreamWriter logStreamWriter;
    private final DbContext dbContext;
    private final RetryStrategy writeRetryStrategy;
    private final RetryStrategy sideEffectsRetryStrategy;
    private final RetryStrategy updateStateRetryStrategy;
    private final BooleanSupplier shouldProcessNext;
    private final BooleanSupplier abortCondition;
    protected final ZeebeState zeebeState;
    private final ErrorRecord errorRecord = new ErrorRecord();
    protected final RecordMetadata metadata = new RecordMetadata();
    private final Map<ValueType, UnifiedRecordValue> eventCache;
    private final RecordProcessorMap recordProcessorMap;
    private final TypedEventImpl typedEvent = new TypedEventImpl();
    protected final TypedResponseWriterImpl responseWriter;
    private SideEffectProducer sideEffectProducer;
    private LoggedEvent currentEvent;
    private TypedRecordProcessor<?> currentProcessor;
    private ZeebeDbTransaction zeebeDbTransaction;
    private long writtenEventPosition = -1L;
    private long lastSuccessfulProcessedEventPosition = -1L;
    private long lastWrittenEventPosition = -1L;
    private boolean onErrorHandling;
    private long errorRecordPosition = -1L;

    public ProcessingStateMachine(ProcessingContext context, StreamProcessorMetrics metrics, BooleanSupplier shouldProcessNext) {
        this.actor = context.getActor();
        this.producerId = context.getProducerId();
        this.streamProcessorName = context.getStreamProcessorName();
        this.eventFilter = context.getEventFilter();
        this.recordProcessorMap = context.getRecordProcessorMap();
        this.eventCache = context.getEventCache();
        this.logStreamReader = context.getLogStreamReader();
        this.logStreamWriter = context.getLogStreamWriter();
        this.logStream = context.getLogStream();
        this.zeebeState = context.getZeebeState();
        this.dbContext = context.getDbContext();
        this.abortCondition = context.getAbortCondition();
        this.metrics = metrics;
        this.writeRetryStrategy = new AbortableRetryStrategy(this.actor);
        this.sideEffectsRetryStrategy = new AbortableRetryStrategy(this.actor);
        this.updateStateRetryStrategy = new RecoverableRetryStrategy(this.actor);
        this.shouldProcessNext = shouldProcessNext;
        this.responseWriter = new TypedResponseWriterImpl(context.getCommandResponseWriter(), this.logStream.getPartitionId());
    }

    private void skipRecord() {
        this.actor.submit(this::readNextEvent);
        this.metrics.incrementEventsSkippedCount();
    }

    void readNextEvent() {
        if (this.shouldProcessNext.getAsBoolean() && this.logStreamReader.hasNext() && this.currentProcessor == null && this.logStream.getCommitPosition() >= this.errorRecordPosition) {
            if (this.onErrorHandling) {
                LOG.info(LOG_ERROR_EVENT_COMMITTED);
                this.onErrorHandling = false;
            }
            this.currentEvent = (LoggedEvent)this.logStreamReader.next();
            if (this.eventFilter == null || this.eventFilter.applies(this.currentEvent)) {
                this.processEvent(this.currentEvent);
            } else {
                this.skipRecord();
            }
        }
    }

    private void processEvent(LoggedEvent event) {
        this.metadata.reset();
        event.readMetadata((BufferReader)this.metadata);
        this.currentProcessor = this.chooseNextProcessor(event);
        if (this.currentProcessor == null) {
            this.skipRecord();
            return;
        }
        try {
            UnifiedRecordValue value = this.eventCache.get(this.metadata.getValueType());
            value.reset();
            event.readValue((BufferReader)value);
            this.typedEvent.wrap(event, this.metadata, value);
            this.processInTransaction(this.typedEvent);
            this.metrics.incrementEventsProcessedCount();
            this.writeEvent();
        }
        catch (RecoverableException recoverableException) {
            LOG.error(ERROR_MESSAGE_PROCESSING_FAILED_RETRY_PROCESSING, new Object[]{event, this.streamProcessorName, recoverableException});
            this.actor.runDelayed(PROCESSING_RETRY_DELAY, () -> this.processEvent(this.currentEvent));
        }
        catch (Exception e) {
            LOG.error(ERROR_MESSAGE_PROCESSING_FAILED_SKIP_EVENT, new Object[]{event, this.streamProcessorName, e});
            this.onError(e, this::writeEvent);
        }
    }

    private TypedRecordProcessor<?> chooseNextProcessor(LoggedEvent event) {
        TypedRecordProcessor typedRecordProcessor = null;
        try {
            typedRecordProcessor = this.recordProcessorMap.get(this.metadata.getRecordType(), this.metadata.getValueType(), this.metadata.getIntent().value());
        }
        catch (Exception e) {
            LOG.error(ERROR_MESSAGE_ON_EVENT_FAILED_SKIP_EVENT, new Object[]{event, this.streamProcessorName, e});
        }
        return typedRecordProcessor;
    }

    private void processInTransaction(TypedEventImpl typedRecord) throws Exception {
        this.zeebeDbTransaction = this.dbContext.getCurrentTransaction();
        this.zeebeDbTransaction.run(() -> {
            boolean isNotOnBlacklist;
            long position = typedRecord.getPosition();
            this.resetOutput(position);
            this.sideEffectProducer = this.responseWriter;
            boolean bl = isNotOnBlacklist = !this.zeebeState.isOnBlacklist(typedRecord);
            if (isNotOnBlacklist) {
                this.currentProcessor.processRecord(position, typedRecord, this.responseWriter, this.logStreamWriter, this::setSideEffectProducer);
            }
            this.zeebeState.markAsProcessed(position);
        });
    }

    private void resetOutput(long sourceRecordPosition) {
        this.responseWriter.reset();
        this.logStreamWriter.reset();
        this.logStreamWriter.configureSourceContext(this.producerId, sourceRecordPosition);
    }

    public void setSideEffectProducer(SideEffectProducer sideEffectProducer) {
        this.sideEffectProducer = sideEffectProducer;
    }

    private void onError(Throwable processingException, Runnable nextStep) {
        ActorFuture retryFuture = this.updateStateRetryStrategy.runWithRetry(() -> {
            this.zeebeDbTransaction.rollback();
            return true;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_ROLLBACK_ABORTED, (Object)this.currentEvent, throwable);
            }
            try {
                this.errorHandlingInTransaction(processingException);
                this.onErrorHandling = true;
                nextStep.run();
            }
            catch (Exception ex) {
                this.onError(ex, nextStep);
            }
        });
    }

    private void errorHandlingInTransaction(Throwable processingException) throws Exception {
        this.zeebeDbTransaction = this.dbContext.getCurrentTransaction();
        this.zeebeDbTransaction.run(() -> {
            long position = this.typedEvent.getPosition();
            this.resetOutput(position);
            this.writeRejectionOnCommand(processingException);
            this.errorRecord.initErrorRecord(processingException, position);
            this.zeebeState.tryToBlacklist(this.typedEvent, arg_0 -> ((ErrorRecord)this.errorRecord).setWorkflowInstanceKey(arg_0));
            this.logStreamWriter.appendFollowUpEvent(this.typedEvent.getKey(), (Intent)ErrorIntent.CREATED, (UnpackedObject)this.errorRecord);
        });
    }

    private void writeRejectionOnCommand(Throwable exception) {
        String errorMessage = String.format(PROCESSING_ERROR_MESSAGE, this.typedEvent, exception.getMessage());
        LOG.error(errorMessage, exception);
        if (this.typedEvent.getMetadata().getRecordType() == RecordType.COMMAND) {
            this.logStreamWriter.appendRejection(this.typedEvent, RejectionType.PROCESSING_ERROR, errorMessage);
            this.responseWriter.writeRejectionOnCommand(this.typedEvent, RejectionType.PROCESSING_ERROR, errorMessage);
        }
    }

    private void writeEvent() {
        ActorFuture retryFuture = this.writeRetryStrategy.runWithRetry(() -> {
            this.writtenEventPosition = this.logStreamWriter.flush();
            return this.writtenEventPosition >= 0L;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, t) -> {
            if (t != null) {
                LOG.error(ERROR_MESSAGE_WRITE_EVENT_ABORTED, (Object)this.currentEvent, t);
                this.onError((Throwable)t, this::writeEvent);
            } else {
                this.metrics.incrementEventsWrittenCount();
                this.updateState();
            }
        });
    }

    private void updateState() {
        ActorFuture retryFuture = this.updateStateRetryStrategy.runWithRetry(() -> {
            this.zeebeDbTransaction.commit();
            if (this.onErrorHandling) {
                this.errorRecordPosition = this.writtenEventPosition;
                LOG.info(LOG_ERROR_EVENT_WRITTEN, (Object)this.errorRecordPosition, (Object)this.logStream.getCommitPosition());
            }
            this.lastSuccessfulProcessedEventPosition = this.currentEvent.getPosition();
            this.lastWrittenEventPosition = this.writtenEventPosition;
            return true;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_UPDATE_STATE_FAILED, new Object[]{this.currentEvent, this.streamProcessorName, throwable});
                this.onError((Throwable)throwable, this::updateState);
            } else {
                this.executeSideEffects();
            }
        });
    }

    private void executeSideEffects() {
        ActorFuture retryFuture = this.sideEffectsRetryStrategy.runWithRetry(this.sideEffectProducer::flush, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_EXECUTE_SIDE_EFFECT_ABORTED, (Object)this.currentEvent, throwable);
            }
            this.currentProcessor = null;
            this.actor.submit(this::readNextEvent);
        });
    }

    public long getLastSuccessfulProcessedEventPosition() {
        return this.lastSuccessfulProcessedEventPosition;
    }

    public long getLastWrittenEventPosition() {
        return this.lastWrittenEventPosition;
    }
}

