/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.engine.processor;

import io.zeebe.db.DbContext;
import io.zeebe.db.TransactionOperation;
import io.zeebe.db.ZeebeDbTransaction;
import io.zeebe.engine.processor.EventFilter;
import io.zeebe.engine.processor.ProcessingContext;
import io.zeebe.engine.processor.RecordProcessorMap;
import io.zeebe.engine.processor.TypedEventImpl;
import io.zeebe.engine.processor.TypedRecord;
import io.zeebe.engine.processor.TypedRecordProcessor;
import io.zeebe.engine.processor.TypedResponseWriter;
import io.zeebe.engine.processor.TypedStreamWriter;
import io.zeebe.engine.state.ZeebeState;
import io.zeebe.logstreams.impl.Loggers;
import io.zeebe.logstreams.log.LogStreamReader;
import io.zeebe.logstreams.log.LoggedEvent;
import io.zeebe.msgpack.UnpackedObject;
import io.zeebe.protocol.impl.record.RecordMetadata;
import io.zeebe.protocol.impl.record.UnifiedRecordValue;
import io.zeebe.protocol.impl.record.value.error.ErrorRecord;
import io.zeebe.protocol.record.RejectionType;
import io.zeebe.protocol.record.ValueType;
import io.zeebe.protocol.record.intent.Intent;
import io.zeebe.util.buffer.BufferReader;
import io.zeebe.util.retry.EndlessRetryStrategy;
import io.zeebe.util.retry.RetryStrategy;
import io.zeebe.util.sched.ActorControl;
import io.zeebe.util.sched.future.ActorFuture;
import io.zeebe.util.sched.future.CompletableActorFuture;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import org.slf4j.Logger;

public final class ReProcessingStateMachine {
    private static final Logger LOG = Loggers.PROCESSOR_LOGGER;
    private static final String ERROR_MESSAGE_ON_EVENT_FAILED_SKIP_EVENT = "Expected to find event processor for event '{}' with processor '{}', but caught an exception. Skip this event.";
    private static final String ERROR_MESSAGE_REPROCESSING_NO_SOURCE_EVENT = "Expected to find last source event position '%d', but last position was '%d'. Failed to reprocess on processor '%s'";
    private static final String ERROR_MESSAGE_REPROCESSING_NO_NEXT_EVENT = "Expected to find last source event position '%d', but found no next event. Failed to reprocess on processor '%s'";
    private static final String LOG_STMT_REPROCESSING_FINISHED = "Processor {} finished reprocessing at event position {}";
    private static final String LOG_STMT_FAILED_ON_PROCESSING = "Event {} failed on processing last time, will call #onError to update workflow instance blacklist.";
    private static final Consumer<Long> NOOP_LONG_CONSUMER = instanceKey -> {};
    public static final Consumer NOOP_SIDE_EFFECT_CONSUMER = sideEffect -> {};
    private final int producerId;
    private final ZeebeState zeebeState;
    private final ActorControl actor;
    private final String streamProcessorName;
    private final ErrorRecord errorRecord = new ErrorRecord();
    protected final RecordMetadata metadata = new RecordMetadata();
    private final TypedEventImpl typedEvent = new TypedEventImpl();
    private final Map<ValueType, UnifiedRecordValue> eventCache;
    private final RecordProcessorMap recordProcessorMap;
    private final EventFilter eventFilter;
    private final LogStreamReader logStreamReader;
    private final TypedStreamWriter noopstreamWriter = new NoopStreamWriter();
    private final TypedResponseWriter noopResponseWriter = new NoopResponseWriter();
    private final DbContext dbContext;
    private final RetryStrategy updateStateRetryStrategy;
    private final RetryStrategy processRetryStrategy;
    private final BooleanSupplier abortCondition;
    private final Set<Long> failedEventPositions = new HashSet<Long>();
    private long lastSourceEventPosition;
    private ActorFuture<Void> recoveryFuture;
    private LoggedEvent currentEvent;
    private TypedRecordProcessor eventProcessor;
    private ZeebeDbTransaction zeebeDbTransaction;

    public ReProcessingStateMachine(ProcessingContext context) {
        this.actor = context.getActor();
        this.streamProcessorName = context.getStreamProcessorName();
        this.eventFilter = context.getEventFilter();
        this.logStreamReader = context.getLogStreamReader();
        this.producerId = context.getProducerId();
        this.eventCache = context.getEventCache();
        this.recordProcessorMap = context.getRecordProcessorMap();
        this.dbContext = context.getDbContext();
        this.zeebeState = context.getZeebeState();
        this.abortCondition = context.getAbortCondition();
        this.updateStateRetryStrategy = new EndlessRetryStrategy(this.actor);
        this.processRetryStrategy = new EndlessRetryStrategy(this.actor);
    }

    ActorFuture<Void> startRecover(long snapshotPosition) {
        this.recoveryFuture = new CompletableActorFuture();
        long startPosition = this.logStreamReader.getPosition();
        LOG.info("Start scanning the log for error events.");
        this.lastSourceEventPosition = this.scanLog(snapshotPosition);
        LOG.info("Finished scanning the log for error events.");
        if (this.lastSourceEventPosition > snapshotPosition) {
            LOG.info("Processor {} starts reprocessing, until last source event position {}", (Object)this.streamProcessorName, (Object)this.lastSourceEventPosition);
            this.logStreamReader.seek(startPosition);
            this.reprocessNextEvent();
        } else {
            this.recoveryFuture.complete(null);
        }
        return this.recoveryFuture;
    }

    private long scanLog(long snapshotPosition) {
        long lastSourceEventPosition = -1L;
        if (this.logStreamReader.hasNext()) {
            lastSourceEventPosition = snapshotPosition;
            while (this.logStreamReader.hasNext()) {
                long sourceEventPosition;
                LoggedEvent newEvent = (LoggedEvent)this.logStreamReader.next();
                this.metadata.reset();
                newEvent.readMetadata((BufferReader)this.metadata);
                long errorPosition = -1L;
                if (this.metadata.getValueType() == ValueType.ERROR) {
                    newEvent.readValue((BufferReader)this.errorRecord);
                    errorPosition = this.errorRecord.getErrorEventPosition();
                }
                if (errorPosition >= 0L) {
                    LOG.debug("Found error-prone event {} on reprocessing, will add position {} to the blacklist.", (Object)newEvent, (Object)errorPosition);
                    this.failedEventPositions.add(errorPosition);
                }
                if (newEvent.getProducerId() != this.producerId || (sourceEventPosition = newEvent.getSourceEventPosition()) <= 0L || sourceEventPosition <= lastSourceEventPosition) continue;
                lastSourceEventPosition = sourceEventPosition;
            }
            this.logStreamReader.seek(snapshotPosition + 1L);
        }
        return lastSourceEventPosition;
    }

    private void readNextEvent() {
        if (!this.logStreamReader.hasNext()) {
            throw new IllegalStateException(String.format(ERROR_MESSAGE_REPROCESSING_NO_NEXT_EVENT, this.lastSourceEventPosition, this.streamProcessorName));
        }
        this.currentEvent = (LoggedEvent)this.logStreamReader.next();
        if (this.currentEvent.getPosition() > this.lastSourceEventPosition) {
            throw new IllegalStateException(String.format(ERROR_MESSAGE_REPROCESSING_NO_SOURCE_EVENT, this.lastSourceEventPosition, this.currentEvent.getPosition(), this.streamProcessorName));
        }
    }

    private void reprocessNextEvent() {
        try {
            this.readNextEvent();
            if (this.eventFilter == null || this.eventFilter.applies(this.currentEvent)) {
                this.reprocessEvent(this.currentEvent);
            } else {
                this.onRecordReprocessed(this.currentEvent);
            }
        }
        catch (RuntimeException e) {
            this.recoveryFuture.completeExceptionally((Throwable)e);
        }
    }

    private void reprocessEvent(LoggedEvent currentEvent) {
        try {
            this.metadata.reset();
            currentEvent.readMetadata((BufferReader)this.metadata);
            this.eventProcessor = this.recordProcessorMap.get(this.metadata.getRecordType(), this.metadata.getValueType(), this.metadata.getIntent().value());
        }
        catch (Exception e) {
            LOG.error(ERROR_MESSAGE_ON_EVENT_FAILED_SKIP_EVENT, new Object[]{currentEvent, this.streamProcessorName, e});
        }
        if (this.eventProcessor == null) {
            this.onRecordReprocessed(currentEvent);
            return;
        }
        UnifiedRecordValue value = this.eventCache.get(this.metadata.getValueType());
        value.reset();
        currentEvent.readValue((BufferReader)value);
        this.typedEvent.wrap(currentEvent, this.metadata, value);
        this.processUntilDone(currentEvent.getPosition(), this.typedEvent);
    }

    private void processUntilDone(long position, TypedRecord<?> currentEvent) {
        TransactionOperation operationOnProcessing = this.chooseOperationForEvent(position, currentEvent);
        ActorFuture resultFuture = this.processRetryStrategy.runWithRetry(() -> {
            boolean onRetry;
            boolean bl = onRetry = this.zeebeDbTransaction != null;
            if (onRetry) {
                this.zeebeDbTransaction.rollback();
            }
            this.zeebeDbTransaction = this.dbContext.getCurrentTransaction();
            this.zeebeDbTransaction.run(operationOnProcessing);
            return true;
        }, this.abortCondition);
        this.actor.runOnCompletion(resultFuture, (v, t) -> {
            assert (t == null) : "On reprocessing there shouldn't be any exception thrown.";
            this.updateStateUntilDone();
        });
    }

    private TransactionOperation chooseOperationForEvent(long position, TypedRecord<?> currentEvent) {
        TransactionOperation operationOnProcessing;
        if (this.failedEventPositions.contains(position)) {
            LOG.info(LOG_STMT_FAILED_ON_PROCESSING, currentEvent);
            operationOnProcessing = () -> this.zeebeState.tryToBlacklist(currentEvent, NOOP_LONG_CONSUMER);
        } else {
            operationOnProcessing = () -> {
                boolean isNotOnBlacklist;
                boolean bl = isNotOnBlacklist = !this.zeebeState.isOnBlacklist(this.typedEvent);
                if (isNotOnBlacklist) {
                    this.eventProcessor.processRecord(position, this.typedEvent, this.noopResponseWriter, this.noopstreamWriter, NOOP_SIDE_EFFECT_CONSUMER);
                }
                this.zeebeState.markAsProcessed(position);
            };
        }
        return operationOnProcessing;
    }

    private void updateStateUntilDone() {
        ActorFuture retryFuture = this.updateStateRetryStrategy.runWithRetry(() -> {
            this.zeebeDbTransaction.commit();
            this.zeebeDbTransaction = null;
            return true;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            assert (throwable == null) : "On reprocessing there shouldn't be any exception thrown.";
            this.onRecordReprocessed(this.currentEvent);
        });
    }

    private void onRecordReprocessed(LoggedEvent currentEvent) {
        if (currentEvent.getPosition() == this.lastSourceEventPosition) {
            LOG.info(LOG_STMT_REPROCESSING_FINISHED, (Object)this.streamProcessorName, (Object)currentEvent.getPosition());
            this.onRecovered();
        } else {
            this.actor.submit(this::reprocessNextEvent);
        }
    }

    private void onRecovered() {
        this.recoveryFuture.complete(null);
        this.failedEventPositions.clear();
    }

    private static final class NoopResponseWriter
    implements TypedResponseWriter {
        private NoopResponseWriter() {
        }

        @Override
        public void writeRejectionOnCommand(TypedRecord<?> command, RejectionType type, String reason) {
        }

        @Override
        public void writeEvent(TypedRecord<?> event) {
        }

        @Override
        public void writeEventOnCommand(long eventKey, Intent eventState, UnpackedObject eventValue, TypedRecord<?> command) {
        }

        @Override
        public boolean flush() {
            return false;
        }
    }

    private static final class NoopStreamWriter
    implements TypedStreamWriter {
        private NoopStreamWriter() {
        }

        @Override
        public void appendRejection(TypedRecord<? extends UnpackedObject> command, RejectionType type, String reason) {
        }

        @Override
        public void appendRejection(TypedRecord<? extends UnpackedObject> command, RejectionType type, String reason, Consumer<RecordMetadata> metadata) {
        }

        @Override
        public void appendNewEvent(long key, Intent intent, UnpackedObject value) {
        }

        @Override
        public void appendFollowUpEvent(long key, Intent intent, UnpackedObject value) {
        }

        @Override
        public void appendFollowUpEvent(long key, Intent intent, UnpackedObject value, Consumer<RecordMetadata> metadata) {
        }

        @Override
        public void appendNewCommand(Intent intent, UnpackedObject value) {
        }

        @Override
        public void appendFollowUpCommand(long key, Intent intent, UnpackedObject value) {
        }

        @Override
        public void appendFollowUpCommand(long key, Intent intent, UnpackedObject value, Consumer<RecordMetadata> metadata) {
        }

        @Override
        public void reset() {
        }

        @Override
        public void configureSourceContext(int producerId, long sourceRecordPosition) {
        }

        @Override
        public long flush() {
            return 0L;
        }
    }
}

