/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.broker.system;

import io.zeebe.broker.Broker;
import io.zeebe.broker.Loggers;
import io.zeebe.broker.system.Component;
import io.zeebe.broker.system.configuration.BrokerCfg;
import io.zeebe.broker.system.configuration.ClusterCfg;
import io.zeebe.broker.system.configuration.SocketBindingCommandApiCfg;
import io.zeebe.broker.system.configuration.ThreadsCfg;
import io.zeebe.servicecontainer.ServiceContainer;
import io.zeebe.servicecontainer.impl.ServiceContainerImpl;
import io.zeebe.util.TomlConfigurationReader;
import io.zeebe.util.metrics.MetricsManager;
import io.zeebe.util.sched.ActorScheduler;
import io.zeebe.util.sched.clock.ActorClock;
import io.zeebe.util.sched.future.ActorFuture;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.slf4j.Logger;

public class SystemContext
implements AutoCloseable {
    public static final Logger LOG = Loggers.SYSTEM_LOGGER;
    public static final String BROKER_ID_LOG_PROPERTY = "broker-id";
    public static final Duration CLOSE_TIMEOUT = Duration.ofSeconds(20L);
    public static final String NODE_ID_ERROR_MSG = "Node id %s needs to be non negative and smaller then cluster size %s.";
    public static final String REPLICATION_FACTOR_ERROR_MSG = "Replication factor %s needs to be larger then zero and not larger then cluster size %s.";
    protected ServiceContainer serviceContainer;
    protected final List<Component> components = new ArrayList<Component>();
    protected final BrokerCfg brokerCfg;
    protected final List<ActorFuture<?>> requiredStartActions = new ArrayList();
    private final List<Closeable> closeablesToReleaseResources = new ArrayList<Closeable>();
    protected Map<String, String> diagnosticContext;
    protected ActorScheduler scheduler;
    private MetricsManager metricsManager;
    private Duration closeTimeout;

    public SystemContext(String configFileLocation, String basePath, ActorClock clock) {
        if (!Paths.get(configFileLocation, new String[0]).isAbsolute()) {
            configFileLocation = Paths.get(basePath, configFileLocation).normalize().toAbsolutePath().toString();
        }
        this.brokerCfg = (BrokerCfg)TomlConfigurationReader.read((String)configFileLocation, BrokerCfg.class);
        this.initSystemContext(clock, basePath);
    }

    public SystemContext(InputStream configStream, String basePath, ActorClock clock) {
        this.brokerCfg = (BrokerCfg)TomlConfigurationReader.read((InputStream)configStream, BrokerCfg.class);
        this.initSystemContext(clock, basePath);
    }

    public SystemContext(BrokerCfg brokerCfg, String basePath, ActorClock clock) {
        this.brokerCfg = brokerCfg;
        this.initSystemContext(clock, basePath);
    }

    private void initSystemContext(ActorClock clock, String basePath) {
        LOG.debug("Initializing configuration with base path {}", (Object)basePath);
        this.brokerCfg.init(basePath);
        this.validateConfiguration();
        SocketBindingCommandApiCfg commandApiCfg = this.brokerCfg.getNetwork().getCommandApi();
        String brokerId = String.format("%s:%d", commandApiCfg.getHost(), commandApiCfg.getPort());
        this.diagnosticContext = Collections.singletonMap(BROKER_ID_LOG_PROPERTY, brokerId);
        this.metricsManager = this.initMetricsManager(brokerId);
        this.scheduler = this.initScheduler(clock, brokerId);
        this.serviceContainer = new ServiceContainerImpl(this.scheduler);
        this.scheduler.start();
        this.initBrokerInfoMetric();
        this.setCloseTimeout(CLOSE_TIMEOUT);
    }

    private void validateConfiguration() {
        ClusterCfg cluster = this.brokerCfg.getCluster();
        int partitionCount = cluster.getPartitionsCount();
        if (partitionCount < 1) {
            throw new IllegalArgumentException("Partition count must not be smaller then 1.");
        }
        int clusterSize = cluster.getClusterSize();
        int nodeId = cluster.getNodeId();
        if (nodeId < 0 || nodeId >= clusterSize) {
            throw new IllegalArgumentException(String.format(NODE_ID_ERROR_MSG, nodeId, clusterSize));
        }
        int replicationFactor = cluster.getReplicationFactor();
        if (replicationFactor < 1 || replicationFactor > clusterSize) {
            throw new IllegalArgumentException(String.format(REPLICATION_FACTOR_ERROR_MSG, replicationFactor, clusterSize));
        }
    }

    private MetricsManager initMetricsManager(String brokerId) {
        HashMap<String, String> globalLabels = new HashMap<String, String>();
        globalLabels.put("cluster", "zeebe");
        globalLabels.put("node", brokerId);
        return new MetricsManager("zb_", globalLabels);
    }

    private void initBrokerInfoMetric() {
        this.metricsManager.newMetric("broker_info").type("counter").label("version", Broker.VERSION).create().incrementOrdered();
    }

    private ActorScheduler initScheduler(ActorClock clock, String brokerId) {
        ThreadsCfg cfg = this.brokerCfg.getThreads();
        int cpuThreads = cfg.getCpuThreadCount();
        int ioThreads = cfg.getIoThreadCount();
        Loggers.SYSTEM_LOGGER.info("Scheduler configuration: Threads{cpu-bound: {}, io-bound: {}}.", (Object)cpuThreads, (Object)ioThreads);
        return ActorScheduler.newActorScheduler().setActorClock(clock).setMetricsManager(this.metricsManager).setCpuBoundActorThreadCount(cpuThreads).setIoBoundActorThreadCount(ioThreads).setSchedulerName(brokerId).build();
    }

    public ActorScheduler getScheduler() {
        return this.scheduler;
    }

    public ServiceContainer getServiceContainer() {
        return this.serviceContainer;
    }

    public void addComponent(Component component) {
        this.components.add(component);
    }

    public List<Component> getComponents() {
        return this.components;
    }

    public void init() {
        this.serviceContainer.start();
        for (Component component : this.components) {
            try {
                component.init(this);
            }
            catch (RuntimeException e) {
                this.close();
                throw e;
            }
        }
        try {
            for (ActorFuture actorFuture : this.requiredStartActions) {
                actorFuture.get(40L, TimeUnit.SECONDS);
            }
        }
        catch (Exception e) {
            LOG.error("Could not start broker", (Throwable)e);
            this.close();
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        LOG.info("Closing...");
        try {
            this.serviceContainer.close(this.getCloseTimeout().toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            LOG.error("Failed to close broker within {} seconds.", (Object)CLOSE_TIMEOUT, (Object)e);
        }
        catch (InterruptedException | ExecutionException e) {
            LOG.error("Exception while closing broker", (Throwable)e);
        }
        finally {
            for (Closeable delegate : this.closeablesToReleaseResources) {
                try {
                    delegate.close();
                }
                catch (IOException ioe) {
                    LOG.error("Exception while releasing resources", (Throwable)ioe);
                }
            }
            try {
                this.scheduler.stop().get(this.getCloseTimeout().toMillis(), TimeUnit.MILLISECONDS);
            }
            catch (TimeoutException e) {
                LOG.error("Failed to close scheduler within {} seconds", (Object)CLOSE_TIMEOUT, (Object)e);
            }
            catch (InterruptedException | ExecutionException e) {
                LOG.error("Exception while closing scheduler", (Throwable)e);
            }
        }
    }

    public BrokerCfg getBrokerConfiguration() {
        return this.brokerCfg;
    }

    public void addRequiredStartAction(ActorFuture<?> future) {
        this.requiredStartActions.add(future);
    }

    public void addResourceReleasingDelegate(Closeable delegate) {
        this.closeablesToReleaseResources.add(delegate);
    }

    public Map<String, String> getDiagnosticContext() {
        return this.diagnosticContext;
    }

    public Duration getCloseTimeout() {
        return this.closeTimeout;
    }

    public void setCloseTimeout(Duration closeTimeout) {
        this.closeTimeout = closeTimeout;
        this.scheduler.setBlockingTasksShutdownTime(closeTimeout);
    }
}

