/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.broker.exporter.stream;

import io.zeebe.broker.Loggers;
import io.zeebe.broker.exporter.stream.ExporterContainer;
import io.zeebe.broker.exporter.stream.ExporterDirectorContext;
import io.zeebe.broker.exporter.stream.ExporterMetrics;
import io.zeebe.broker.exporter.stream.ExporterPhase;
import io.zeebe.broker.exporter.stream.ExportersState;
import io.zeebe.db.ZeebeDb;
import io.zeebe.engine.processing.streamprocessor.EventFilter;
import io.zeebe.engine.processing.streamprocessor.RecordValues;
import io.zeebe.engine.processing.streamprocessor.TypedEventImpl;
import io.zeebe.engine.processing.streamprocessor.TypedRecord;
import io.zeebe.engine.state.ZbColumnFamilies;
import io.zeebe.logstreams.log.LogStream;
import io.zeebe.logstreams.log.LogStreamReader;
import io.zeebe.logstreams.log.LoggedEvent;
import io.zeebe.protocol.impl.record.RecordMetadata;
import io.zeebe.protocol.impl.record.UnifiedRecordValue;
import io.zeebe.protocol.record.RecordType;
import io.zeebe.protocol.record.ValueType;
import io.zeebe.util.buffer.BufferReader;
import io.zeebe.util.retry.BackOffRetryStrategy;
import io.zeebe.util.retry.EndlessRetryStrategy;
import io.zeebe.util.retry.RetryStrategy;
import io.zeebe.util.sched.Actor;
import io.zeebe.util.sched.ActorCondition;
import io.zeebe.util.sched.ActorScheduler;
import io.zeebe.util.sched.SchedulingHints;
import io.zeebe.util.sched.future.ActorFuture;
import io.zeebe.util.sched.future.CompletableActorFuture;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.agrona.LangUtil;
import org.slf4j.Logger;

public final class ExporterDirector
extends Actor {
    private static final String ERROR_MESSAGE_EXPORTING_ABORTED = "Expected to export record '{}' successfully, but exception was thrown.";
    private static final String ERROR_MESSAGE_RECOVER_FROM_SNAPSHOT_FAILED = "Expected to find event with the snapshot position %s in log stream, but nothing was found. Failed to recover '%s'.";
    private static final Logger LOG = Loggers.EXPORTER_LOGGER;
    private final AtomicBoolean isOpened = new AtomicBoolean(false);
    private final List<ExporterContainer> containers;
    private final LogStream logStream;
    private final RecordExporter recordExporter;
    private final ZeebeDb zeebeDb;
    private final ExporterMetrics metrics;
    private final String name;
    private final RetryStrategy exportingRetryStrategy;
    private final RetryStrategy recordWrapStrategy;
    private LogStreamReader logStreamReader;
    private EventFilter eventFilter;
    private ExportersState state;
    private ActorCondition onCommitPositionUpdatedCondition;
    private boolean inExportingPhase;
    private boolean isPaused;
    private ExporterPhase exporterPhase;

    public ExporterDirector(ExporterDirectorContext context, boolean shouldPauseOnStart) {
        this.name = context.getName();
        this.containers = context.getDescriptors().stream().map(ExporterContainer::new).collect(Collectors.toList());
        this.logStream = Objects.requireNonNull(context.getLogStream());
        int partitionId = this.logStream.getPartitionId();
        this.metrics = new ExporterMetrics(partitionId);
        this.recordExporter = new RecordExporter(this.metrics, this.containers, partitionId);
        this.exportingRetryStrategy = new BackOffRetryStrategy(this.actor, Duration.ofSeconds(10L));
        this.recordWrapStrategy = new EndlessRetryStrategy(this.actor);
        this.zeebeDb = context.getZeebeDb();
        this.isPaused = shouldPauseOnStart;
    }

    public ActorFuture<Void> startAsync(ActorScheduler actorScheduler) {
        return actorScheduler.submitActor((Actor)this, SchedulingHints.ioBound());
    }

    public ActorFuture<Void> stopAsync() {
        return this.actor.close();
    }

    public ActorFuture<Void> pauseExporting() {
        return this.actor.call(() -> {
            this.isPaused = true;
            this.exporterPhase = ExporterPhase.PAUSED;
        });
    }

    public ActorFuture<Void> resumeExporting() {
        return this.actor.call(() -> {
            this.isPaused = false;
            this.exporterPhase = ExporterPhase.EXPORTING;
            this.actor.submit(this::readNextEvent);
        });
    }

    public ActorFuture<ExporterPhase> getPhase() {
        if (this.actor.isClosed()) {
            return CompletableActorFuture.completed((Object)((Object)ExporterPhase.CLOSED));
        }
        return this.actor.call(() -> this.exporterPhase);
    }

    public String getName() {
        return this.name;
    }

    protected void onActorStarting() {
        ActorFuture newReaderFuture = this.logStream.newLogStreamReader();
        this.actor.runOnCompletionBlockingCurrentPhase(newReaderFuture, (reader, errorOnReceivingReader) -> {
            if (errorOnReceivingReader == null) {
                this.logStreamReader = reader;
            } else {
                LOG.error("Unexpected error on retrieving reader from log {}", (Object)this.logStream.getLogName(), errorOnReceivingReader);
                this.actor.close();
            }
        });
    }

    protected void onActorStarted() {
        try {
            LOG.debug("Recovering exporter from snapshot");
            this.recoverFromSnapshot();
            for (ExporterContainer container : this.containers) {
                container.initContainer(this.actor, this.metrics, this.state);
                container.configureExporter();
            }
            this.eventFilter = this.createEventFilter(this.containers);
            LOG.debug("Set event filter for exporters: {}", (Object)this.eventFilter);
        }
        catch (Exception e) {
            this.onFailure();
            LangUtil.rethrowUnchecked((Throwable)e);
        }
        this.isOpened.set(true);
        this.onSnapshotRecovered();
    }

    protected void onActorClosing() {
        this.logStreamReader.close();
        if (this.onCommitPositionUpdatedCondition != null) {
            this.logStream.removeOnCommitPositionUpdatedCondition(this.onCommitPositionUpdatedCondition);
            this.onCommitPositionUpdatedCondition = null;
        }
    }

    protected void onActorClosed() {
        LOG.debug("Closed exporter director '{}'.", (Object)this.getName());
        this.exporterPhase = ExporterPhase.CLOSED;
    }

    protected void onActorCloseRequested() {
        this.isOpened.set(false);
        this.containers.forEach(ExporterContainer::close);
    }

    private void recoverFromSnapshot() {
        boolean failedToRecoverReader;
        this.state = new ExportersState((ZeebeDb<ZbColumnFamilies>)this.zeebeDb, this.zeebeDb.createContext());
        long snapshotPosition = this.state.getLowestPosition();
        boolean bl = failedToRecoverReader = !this.logStreamReader.seekToNextEvent(snapshotPosition);
        if (failedToRecoverReader) {
            throw new IllegalStateException(String.format(ERROR_MESSAGE_RECOVER_FROM_SNAPSHOT_FAILED, snapshotPosition, this.getName()));
        }
        LOG.debug("Recovered exporter '{}' from snapshot at lastExportedPosition {}", (Object)this.getName(), (Object)snapshotPosition);
    }

    private ExporterEventFilter createEventFilter(List<ExporterContainer> containers) {
        List recordFilters = containers.stream().map(c -> c.getContext().getFilter()).collect(Collectors.toList());
        Map<RecordType, Boolean> acceptRecordTypes = Arrays.stream(RecordType.values()).collect(Collectors.toMap(Function.identity(), type -> recordFilters.stream().anyMatch(f -> f.acceptType(type))));
        Map<ValueType, Boolean> acceptValueTypes = Arrays.stream(ValueType.values()).collect(Collectors.toMap(Function.identity(), type -> recordFilters.stream().anyMatch(f -> f.acceptValue(type))));
        return new ExporterEventFilter(acceptRecordTypes, acceptValueTypes);
    }

    private void onFailure() {
        this.isOpened.set(false);
        this.actor.close();
    }

    private void onSnapshotRecovered() {
        this.onCommitPositionUpdatedCondition = this.actor.onCondition(this.getName() + "-on-commit-lastExportedPosition-updated", this::readNextEvent);
        this.logStream.registerOnCommitPositionUpdatedCondition(this.onCommitPositionUpdatedCondition);
        for (ExporterContainer container : this.containers) {
            container.initPosition();
            container.openExporter();
        }
        this.clearExporterState();
        if (this.state.hasExporters()) {
            if (!this.isPaused) {
                this.exporterPhase = ExporterPhase.EXPORTING;
                this.actor.submit(this::readNextEvent);
            } else {
                this.exporterPhase = ExporterPhase.PAUSED;
            }
        } else {
            this.actor.close();
        }
    }

    private void skipRecord(LoggedEvent currentEvent) {
        RecordMetadata metadata = new RecordMetadata();
        long eventPosition = currentEvent.getPosition();
        currentEvent.readMetadata((BufferReader)metadata);
        this.metrics.eventSkipped(metadata.getValueType());
        for (ExporterContainer container : this.containers) {
            container.updatePositionOnSkipIfUpToDate(eventPosition);
        }
        this.actor.submit(this::readNextEvent);
    }

    private void readNextEvent() {
        if (this.shouldExport()) {
            LoggedEvent currentEvent = (LoggedEvent)this.logStreamReader.next();
            if (this.eventFilter == null || this.eventFilter.applies(currentEvent)) {
                this.inExportingPhase = true;
                this.exportEvent(currentEvent);
            } else {
                this.skipRecord(currentEvent);
            }
        }
    }

    private boolean shouldExport() {
        return this.isOpened.get() && this.logStreamReader.hasNext() && !this.inExportingPhase && !this.isPaused;
    }

    private void exportEvent(LoggedEvent event) {
        ActorFuture wrapRetryFuture = this.recordWrapStrategy.runWithRetry(() -> {
            this.recordExporter.wrap(event);
            return true;
        }, this::isClosed);
        this.actor.runOnCompletion(wrapRetryFuture, (b, t) -> {
            assert (t == null) : "Throwable must be null";
            ActorFuture retryFuture = this.exportingRetryStrategy.runWithRetry(this.recordExporter::export, this::isClosed);
            this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
                if (throwable != null) {
                    LOG.error(ERROR_MESSAGE_EXPORTING_ABORTED, (Object)event, throwable);
                    this.onFailure();
                } else {
                    this.metrics.eventExported(this.recordExporter.getTypedEvent().getValueType());
                    this.inExportingPhase = false;
                    this.actor.submit(this::readNextEvent);
                }
            });
        });
    }

    public ExportersState getState() {
        return this.state;
    }

    private void clearExporterState() {
        List exporterIds = this.containers.stream().map(ExporterContainer::getId).collect(Collectors.toList());
        this.state.visitPositions((exporterId, position) -> {
            if (!exporterIds.contains(exporterId)) {
                this.state.removePosition((String)exporterId);
                LOG.info("The exporter '{}' is not configured anymore. Its lastExportedPosition is removed from the state.", exporterId);
            }
        });
    }

    private boolean isClosed() {
        return !this.isOpened.get();
    }

    private static class ExporterEventFilter
    implements EventFilter {
        private final RecordMetadata metadata = new RecordMetadata();
        private final Map<RecordType, Boolean> acceptRecordTypes;
        private final Map<ValueType, Boolean> acceptValueTypes;

        ExporterEventFilter(Map<RecordType, Boolean> acceptRecordTypes, Map<ValueType, Boolean> acceptValueTypes) {
            this.acceptRecordTypes = acceptRecordTypes;
            this.acceptValueTypes = acceptValueTypes;
        }

        public boolean applies(LoggedEvent event) {
            event.readMetadata((BufferReader)this.metadata);
            RecordType recordType = this.metadata.getRecordType();
            ValueType valueType = this.metadata.getValueType();
            return this.acceptRecordTypes.get(recordType) != false && this.acceptValueTypes.get(valueType) != false;
        }

        public String toString() {
            return "ExporterEventFilter{acceptRecordTypes=" + this.acceptRecordTypes + ", acceptValueTypes=" + this.acceptValueTypes + "}";
        }
    }

    private static class RecordExporter {
        private final RecordValues recordValues = new RecordValues();
        private final RecordMetadata rawMetadata = new RecordMetadata();
        private final List<ExporterContainer> containers;
        private final TypedEventImpl typedEvent;
        private final ExporterMetrics exporterMetrics;
        private boolean shouldExport;
        private int exporterIndex;

        RecordExporter(ExporterMetrics exporterMetrics, List<ExporterContainer> containers, int partitionId) {
            this.containers = containers;
            this.typedEvent = new TypedEventImpl(partitionId);
            this.exporterMetrics = exporterMetrics;
        }

        void wrap(LoggedEvent rawEvent) {
            rawEvent.readMetadata((BufferReader)this.rawMetadata);
            UnifiedRecordValue recordValue = this.recordValues.readRecordValue(rawEvent, this.rawMetadata.getValueType());
            boolean bl = this.shouldExport = recordValue != null;
            if (this.shouldExport) {
                this.typedEvent.wrap(rawEvent, this.rawMetadata, recordValue);
                this.exporterIndex = 0;
            }
        }

        public boolean export() {
            if (!this.shouldExport) {
                return true;
            }
            int exportersCount = this.containers.size();
            while (this.exporterIndex < exportersCount) {
                ExporterContainer container = this.containers.get(this.exporterIndex);
                if (container.exportRecord(this.rawMetadata, (TypedRecord)this.typedEvent)) {
                    ++this.exporterIndex;
                    this.exporterMetrics.setLastExportedPosition(container.getId(), this.typedEvent.getPosition());
                    continue;
                }
                return false;
            }
            return true;
        }

        TypedEventImpl getTypedEvent() {
            return this.typedEvent;
        }
    }
}

