/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.broker.system;

import io.zeebe.broker.Loggers;
import io.zeebe.broker.system.configuration.BrokerCfg;
import io.zeebe.broker.system.configuration.ClusterCfg;
import io.zeebe.broker.system.configuration.DataCfg;
import io.zeebe.broker.system.configuration.ExperimentalCfg;
import io.zeebe.broker.system.configuration.ThreadsCfg;
import io.zeebe.broker.system.partitions.impl.AsyncSnapshotDirector;
import io.zeebe.util.sched.ActorScheduler;
import io.zeebe.util.sched.clock.ActorClock;
import java.time.Duration;
import java.util.Collections;
import java.util.Map;
import org.slf4j.Logger;
import org.springframework.util.unit.DataSize;

public final class SystemContext {
    public static final Logger LOG = Loggers.SYSTEM_LOGGER;
    private static final String BROKER_ID_LOG_PROPERTY = "broker-id";
    private static final String NODE_ID_ERROR_MSG = "Node id %s needs to be non negative and smaller then cluster size %s.";
    private static final String REPLICATION_FACTOR_ERROR_MSG = "Replication factor %s needs to be larger then zero and not larger then cluster size %s.";
    private static final String SNAPSHOT_PERIOD_ERROR_MSG = "Snapshot period %s needs to be larger then or equals to one minute.";
    private static final String MAX_BATCH_SIZE_ERROR_MSG = "Expected to have an append batch size maximum which is non negative and smaller then '%d', but was '%s'.";
    private static final String REPLICATION_WITH_DISABLED_FLUSH_WARNING = "Disabling explicit flushing is an experimental feature and can lead to inconsistencies and/or data loss! Please refer to the documentation whether or not you should use this!";
    protected final BrokerCfg brokerCfg;
    private Map<String, String> diagnosticContext;
    private ActorScheduler scheduler;
    private Duration stepTimeout;

    public SystemContext(BrokerCfg brokerCfg, String basePath, ActorClock clock) {
        this.brokerCfg = brokerCfg;
        this.initSystemContext(clock, basePath);
    }

    private void initSystemContext(ActorClock clock, String basePath) {
        LOG.debug("Initializing system with base path {}", (Object)basePath);
        this.brokerCfg.init(basePath);
        this.validateConfiguration();
        this.stepTimeout = this.brokerCfg.getStepTimeout();
        ClusterCfg cluster = this.brokerCfg.getCluster();
        String brokerId = String.format("Broker-%d", cluster.getNodeId());
        this.diagnosticContext = Collections.singletonMap(BROKER_ID_LOG_PROPERTY, brokerId);
        this.scheduler = this.initScheduler(clock, brokerId);
        this.setStepTimeout(this.stepTimeout);
    }

    private void validateConfiguration() {
        ClusterCfg cluster = this.brokerCfg.getCluster();
        DataCfg data = this.brokerCfg.getData();
        ExperimentalCfg experimental = this.brokerCfg.getExperimental();
        int partitionCount = cluster.getPartitionsCount();
        if (partitionCount < 1) {
            throw new IllegalArgumentException("Partition count must not be smaller then 1.");
        }
        int clusterSize = cluster.getClusterSize();
        int nodeId = cluster.getNodeId();
        if (nodeId < 0 || nodeId >= clusterSize) {
            throw new IllegalArgumentException(String.format(NODE_ID_ERROR_MSG, nodeId, clusterSize));
        }
        DataSize maxAppendBatchSize = experimental.getMaxAppendBatchSize();
        if (maxAppendBatchSize.isNegative() || maxAppendBatchSize.toBytes() >= Integer.MAX_VALUE) {
            throw new IllegalArgumentException(String.format(MAX_BATCH_SIZE_ERROR_MSG, Integer.MAX_VALUE, maxAppendBatchSize));
        }
        int replicationFactor = cluster.getReplicationFactor();
        if (replicationFactor < 1 || replicationFactor > clusterSize) {
            throw new IllegalArgumentException(String.format(REPLICATION_FACTOR_ERROR_MSG, replicationFactor, clusterSize));
        }
        DataCfg dataCfg = this.brokerCfg.getData();
        Duration snapshotPeriod = dataCfg.getSnapshotPeriod();
        if (snapshotPeriod.isNegative() || snapshotPeriod.minus(AsyncSnapshotDirector.MINIMUM_SNAPSHOT_PERIOD).isNegative()) {
            throw new IllegalArgumentException(String.format(SNAPSHOT_PERIOD_ERROR_MSG, snapshotPeriod));
        }
        double diskUsageCommandWatermark = dataCfg.getDiskUsageCommandWatermark();
        if (!(diskUsageCommandWatermark > 0.0) || !(diskUsageCommandWatermark <= 1.0)) {
            throw new IllegalArgumentException(String.format("Expected diskUsageCommandWatermark to be in the range (0,1], but found %f", diskUsageCommandWatermark));
        }
        double diskUsageReplicationWatermark = dataCfg.getDiskUsageReplicationWatermark();
        if (!(diskUsageReplicationWatermark > 0.0) || !(diskUsageReplicationWatermark <= 1.0)) {
            throw new IllegalArgumentException(String.format("Expected diskUsageReplicationWatermark to be in the range (0,1], but found %f", diskUsageReplicationWatermark));
        }
        if (data.isDiskUsageMonitoringEnabled() && diskUsageCommandWatermark >= diskUsageReplicationWatermark) {
            throw new IllegalArgumentException(String.format("diskUsageCommandWatermark (%f) must be less than diskUsageReplicationWatermark (%f)", diskUsageCommandWatermark, diskUsageReplicationWatermark));
        }
        if (experimental.isDisableExplicitRaftFlush()) {
            LOG.warn(REPLICATION_WITH_DISABLED_FLUSH_WARNING);
        }
    }

    private ActorScheduler initScheduler(ActorClock clock, String brokerId) {
        ThreadsCfg cfg = this.brokerCfg.getThreads();
        int cpuThreads = cfg.getCpuThreadCount();
        int ioThreads = cfg.getIoThreadCount();
        return ActorScheduler.newActorScheduler().setActorClock(clock).setCpuBoundActorThreadCount(cpuThreads).setIoBoundActorThreadCount(ioThreads).setSchedulerName(brokerId).build();
    }

    public ActorScheduler getScheduler() {
        return this.scheduler;
    }

    public BrokerCfg getBrokerConfiguration() {
        return this.brokerCfg;
    }

    public Map<String, String> getDiagnosticContext() {
        return this.diagnosticContext;
    }

    public Duration getStepTimeout() {
        return this.stepTimeout;
    }

    private void setStepTimeout(Duration stepTimeout) {
        this.stepTimeout = stepTimeout;
    }
}

