/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.broker.system.partitions.impl;

import io.atomix.raft.storage.log.IndexedRaftRecord;
import io.zeebe.broker.system.partitions.AtomixRecordEntrySupplier;
import io.zeebe.broker.system.partitions.SnapshotReplication;
import io.zeebe.broker.system.partitions.StateController;
import io.zeebe.broker.system.partitions.impl.SnapshotReplicationMetrics;
import io.zeebe.db.ZeebeDb;
import io.zeebe.db.ZeebeDbFactory;
import io.zeebe.logstreams.impl.Loggers;
import io.zeebe.snapshots.broker.ConstructableSnapshotStore;
import io.zeebe.snapshots.raft.PersistedSnapshot;
import io.zeebe.snapshots.raft.PersistedSnapshotListener;
import io.zeebe.snapshots.raft.ReceivableSnapshotStore;
import io.zeebe.snapshots.raft.ReceivedSnapshot;
import io.zeebe.snapshots.raft.SnapshotChunk;
import io.zeebe.snapshots.raft.SnapshotChunkReader;
import io.zeebe.snapshots.raft.TransientSnapshot;
import io.zeebe.util.FileUtil;
import io.zeebe.util.sched.future.ActorFuture;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Map;
import java.util.Optional;
import java.util.function.ToLongFunction;
import org.agrona.collections.Object2NullableObjectHashMap;
import org.slf4j.Logger;

public class StateControllerImpl
implements StateController,
PersistedSnapshotListener {
    private static final ReplicationContext INVALID_SNAPSHOT = new ReplicationContext(null, -1L, null);
    private static final Logger LOG = Loggers.SNAPSHOT_LOGGER;
    private final SnapshotReplication replication;
    private final Map<String, ReplicationContext> receivedSnapshots = new Object2NullableObjectHashMap();
    private final Path runtimeDirectory;
    private final ZeebeDbFactory zeebeDbFactory;
    private final ToLongFunction<ZeebeDb> exporterPositionSupplier;
    private final AtomixRecordEntrySupplier entrySupplier;
    private final SnapshotReplicationMetrics metrics;
    private ZeebeDb db;
    private final ConstructableSnapshotStore constructableSnapshotStore;
    private final ReceivableSnapshotStore receivableSnapshotStore;

    public StateControllerImpl(int partitionId, ZeebeDbFactory zeebeDbFactory, ConstructableSnapshotStore constructableSnapshotStore, ReceivableSnapshotStore receivableSnapshotStore, Path runtimeDirectory, SnapshotReplication replication, AtomixRecordEntrySupplier entrySupplier, ToLongFunction<ZeebeDb> exporterPositionSupplier) {
        this.constructableSnapshotStore = constructableSnapshotStore;
        this.receivableSnapshotStore = receivableSnapshotStore;
        this.runtimeDirectory = runtimeDirectory;
        this.zeebeDbFactory = zeebeDbFactory;
        this.exporterPositionSupplier = exporterPositionSupplier;
        this.entrySupplier = entrySupplier;
        this.replication = replication;
        this.metrics = new SnapshotReplicationMetrics(Integer.toString(partitionId));
    }

    @Override
    public Optional<TransientSnapshot> takeTransientSnapshot(long lowerBoundSnapshotPosition) {
        if (!this.isDbOpened()) {
            LOG.warn("Expected to take snapshot for last processed position {}, but database was closed.", (Object)lowerBoundSnapshotPosition);
            return Optional.empty();
        }
        long exportedPosition = this.exporterPositionSupplier.applyAsLong(this.openDb());
        long snapshotPosition = this.determineSnapshotPosition(lowerBoundSnapshotPosition, exportedPosition);
        Optional<IndexedRaftRecord> optionalIndexed = this.entrySupplier.getPreviousIndexedEntry(snapshotPosition);
        if (optionalIndexed.isEmpty()) {
            LOG.warn("Failed to take snapshot. Expected to find an indexed entry for determined snapshot position {}, but found no matching indexed entry which contains this position.", (Object)snapshotPosition);
            return Optional.empty();
        }
        IndexedRaftRecord snapshotIndexedEntry = optionalIndexed.get();
        Optional transientSnapshot = this.constructableSnapshotStore.newTransientSnapshot(snapshotIndexedEntry.index(), snapshotIndexedEntry.entry().term(), lowerBoundSnapshotPosition, exportedPosition);
        transientSnapshot.ifPresent(this::takeSnapshot);
        return transientSnapshot;
    }

    @Override
    public void consumeReplicatedSnapshots() {
        this.replication.consume(this::consumeSnapshotChunk);
    }

    @Override
    public void recover() throws Exception {
        Optional optLatestSnapshot;
        if (Files.exists(this.runtimeDirectory, new LinkOption[0])) {
            FileUtil.deleteFolder((Path)this.runtimeDirectory);
        }
        if ((optLatestSnapshot = this.constructableSnapshotStore.getLatestSnapshot()).isPresent()) {
            PersistedSnapshot snapshot = (PersistedSnapshot)optLatestSnapshot.get();
            LOG.debug("Available snapshot: {}", (Object)snapshot);
            FileUtil.copySnapshot((Path)this.runtimeDirectory, (Path)snapshot.getPath());
            try {
                this.openDb();
                LOG.debug("Recovered state from snapshot '{}'", (Object)snapshot);
            }
            catch (Exception exception) {
                LOG.error("Failed to open snapshot '{}'. No snapshots available to recover from. Manual action is required.", (Object)snapshot, (Object)exception);
                FileUtil.deleteFolder((Path)this.runtimeDirectory);
                throw new IllegalStateException("Failed to recover from snapshots", exception);
            }
        }
    }

    @Override
    public ZeebeDb openDb() {
        if (this.db == null) {
            this.db = this.zeebeDbFactory.createDb(this.runtimeDirectory.toFile());
            LOG.debug("Opened database from '{}'.", (Object)this.runtimeDirectory);
        }
        return this.db;
    }

    @Override
    public int getValidSnapshotsCount() {
        return this.constructableSnapshotStore.getLatestSnapshot().isPresent() ? 1 : 0;
    }

    @Override
    public void close() throws Exception {
        if (this.db != null) {
            this.db.close();
            LOG.debug("Closed database from '{}'.", (Object)this.runtimeDirectory);
            this.db = null;
        }
    }

    boolean isDbOpened() {
        return this.db != null;
    }

    private ActorFuture<Boolean> takeSnapshot(TransientSnapshot snapshot) {
        return snapshot.take(snapshotDir -> {
            if (this.db == null) {
                LOG.error("Expected to take a snapshot, but no database was opened");
                return false;
            }
            LOG.debug("Taking temporary snapshot into {}.", snapshotDir);
            try {
                this.db.createSnapshot(snapshotDir.toFile());
            }
            catch (Exception e) {
                LOG.error("Failed to create snapshot of runtime database", (Throwable)e);
                return false;
            }
            return true;
        });
    }

    public void onNewSnapshot(PersistedSnapshot newPersistedSnapshot) {
        LOG.debug("New snapshot {} was persisted. Start replicating.", (Object)newPersistedSnapshot.getId());
        try (SnapshotChunkReader snapshotChunkReader = newPersistedSnapshot.newChunkReader();){
            while (snapshotChunkReader.hasNext()) {
                SnapshotChunk snapshotChunk = (SnapshotChunk)snapshotChunkReader.next();
                this.replication.replicate(snapshotChunk);
            }
        }
    }

    private void consumeSnapshotChunk(SnapshotChunk snapshotChunk) {
        String snapshotId = snapshotChunk.getSnapshotId();
        String chunkName = snapshotChunk.getChunkName();
        ReplicationContext context = this.receivedSnapshots.computeIfAbsent(snapshotId, id -> {
            long startTimestamp = System.currentTimeMillis();
            ReceivedSnapshot transientSnapshot = this.receivableSnapshotStore.newReceivedSnapshot(snapshotChunk.getSnapshotId());
            return this.newReplication(startTimestamp, transientSnapshot);
        });
        if (context == INVALID_SNAPSHOT) {
            LOG.trace("Ignore snapshot chunk {}, because snapshot {} is marked as invalid.", (Object)chunkName, (Object)snapshotId);
            return;
        }
        try {
            if (context.apply(snapshotChunk)) {
                this.validateWhenReceivedAllChunks(snapshotChunk, context);
            } else {
                this.markSnapshotAsInvalid(context, snapshotChunk);
            }
        }
        catch (IOException e) {
            LOG.error("Unexpected error on writing the received snapshot chunk {}", (Object)snapshotChunk, (Object)e);
            this.markSnapshotAsInvalid(context, snapshotChunk);
        }
    }

    private void markSnapshotAsInvalid(ReplicationContext replicationContext, SnapshotChunk chunk) {
        LOG.debug("Abort snapshot {} and mark it as invalid.", (Object)chunk.getSnapshotId());
        replicationContext.abort();
        this.receivedSnapshots.put(chunk.getSnapshotId(), INVALID_SNAPSHOT);
    }

    private void validateWhenReceivedAllChunks(SnapshotChunk snapshotChunk, ReplicationContext context) {
        int totalChunkCount = snapshotChunk.getTotalCount();
        if (context.incrementCount() == (long)totalChunkCount) {
            LOG.debug("Received all snapshot chunks ({}/{}), snapshot {} is valid", new Object[]{context.getChunkCount(), snapshotChunk.getSnapshotId(), totalChunkCount});
            if (!this.tryToMarkSnapshotAsValid(snapshotChunk, context)) {
                LOG.debug("Failed to mark snapshot {} as valid", (Object)snapshotChunk.getSnapshotId());
            }
        } else {
            LOG.debug("Waiting for more snapshot chunks of snapshot {}, currently have {}/{}", new Object[]{snapshotChunk.getSnapshotId(), context.getChunkCount(), totalChunkCount});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean tryToMarkSnapshotAsValid(SnapshotChunk snapshotChunk, ReplicationContext context) {
        try {
            context.persist();
        }
        catch (Exception exception) {
            this.markSnapshotAsInvalid(context, snapshotChunk);
            LOG.warn("Unexpected error on persisting received snapshot.", (Throwable)exception);
            boolean bl = false;
            return bl;
        }
        finally {
            this.receivedSnapshots.remove(snapshotChunk.getSnapshotId());
        }
        return true;
    }

    private ReplicationContext newReplication(long startTimestamp, ReceivedSnapshot transientSnapshot) {
        return new ReplicationContext(this.metrics, startTimestamp, transientSnapshot);
    }

    private long determineSnapshotPosition(long lowerBoundSnapshotPosition, long exportedPosition) {
        long snapshotPosition = Math.min(exportedPosition, lowerBoundSnapshotPosition);
        LOG.debug("Based on lowest exporter position '{}' and last processed position '{}', determined '{}' as snapshot position.", new Object[]{exportedPosition, lowerBoundSnapshotPosition, snapshotPosition});
        return snapshotPosition;
    }

    private static final class ReplicationContext {
        private final long startTimestamp;
        private final ReceivedSnapshot receivedSnapshot;
        private final SnapshotReplicationMetrics metrics;
        private long chunkCount;

        ReplicationContext(SnapshotReplicationMetrics metrics, long startTimestamp, ReceivedSnapshot receivedSnapshot) {
            this.metrics = metrics;
            if (metrics != null) {
                metrics.incrementCount();
            }
            this.startTimestamp = startTimestamp;
            this.chunkCount = 0L;
            this.receivedSnapshot = receivedSnapshot;
        }

        long incrementCount() {
            return ++this.chunkCount;
        }

        long getChunkCount() {
            return this.chunkCount;
        }

        void abort() {
            try {
                this.receivedSnapshot.abort();
            }
            finally {
                this.metrics.decrementCount();
            }
        }

        void persist() {
            try {
                this.receivedSnapshot.persist();
            }
            finally {
                long end = System.currentTimeMillis();
                this.metrics.decrementCount();
                this.metrics.observeDuration(end - this.startTimestamp);
            }
        }

        public boolean apply(SnapshotChunk snapshotChunk) throws IOException {
            return (Boolean)this.receivedSnapshot.apply(snapshotChunk).join();
        }
    }
}

