/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.exporter;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.prometheus.client.Histogram;
import io.zeebe.exporter.ElasticsearchExporter;
import io.zeebe.exporter.ElasticsearchExporterConfiguration;
import io.zeebe.exporter.ElasticsearchExporterException;
import io.zeebe.exporter.ElasticsearchMetrics;
import io.zeebe.exporter.dto.BulkItemError;
import io.zeebe.exporter.dto.BulkResponse;
import io.zeebe.exporter.dto.PutIndexTemplateResponse;
import io.zeebe.protocol.record.Record;
import io.zeebe.protocol.record.ValueType;
import io.zeebe.protocol.record.value.VariableRecordValue;
import io.zeebe.util.VersionUtil;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.elasticsearch.client.Request;
import org.elasticsearch.client.Response;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.common.xcontent.DeprecationHandler;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.common.xcontent.XContent;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.common.xcontent.XContentType;
import org.slf4j.Logger;

public class ElasticsearchClient {
    public static final String INDEX_TEMPLATE_FILENAME_PATTERN = "/zeebe-record-%s-template.json";
    public static final String INDEX_DELIMITER = "_";
    public static final String ALIAS_DELIMITER = "-";
    private static final ObjectMapper MAPPER = new ObjectMapper();
    protected final RestClient client;
    private final ElasticsearchExporterConfiguration configuration;
    private final Logger log;
    private final DateTimeFormatter formatter;
    private List<String> bulkRequest;
    private ElasticsearchMetrics metrics;

    public ElasticsearchClient(ElasticsearchExporterConfiguration configuration, Logger log) {
        this(configuration, log, new ArrayList<String>());
    }

    ElasticsearchClient(ElasticsearchExporterConfiguration configuration, Logger log, List<String> bulkRequest) {
        this.configuration = configuration;
        this.log = log;
        this.client = this.createClient();
        this.bulkRequest = bulkRequest;
        this.formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd").withZone(ZoneOffset.UTC);
    }

    public void close() throws IOException {
        this.client.close();
    }

    public void index(Record<?> record) {
        if (this.metrics == null) {
            this.metrics = new ElasticsearchMetrics(record.getPartitionId());
        }
        this.checkRecord(record);
        this.bulk(this.newIndexCommand(record), record);
    }

    private void checkRecord(Record<?> record) {
        if (record.getValueType() == ValueType.VARIABLE) {
            this.checkVariableRecordValue(record);
        }
    }

    private void checkVariableRecordValue(Record<VariableRecordValue> record) {
        VariableRecordValue value = (VariableRecordValue)record.getValue();
        int size = value.getValue().getBytes().length;
        if (size > this.configuration.index.ignoreVariablesAbove) {
            this.log.warn("Variable {key: {}, name: {}, variableScope: {}, processInstanceKey: {}} exceeded max size of {} bytes with a size of {} bytes. As a consequence this variable is not index by elasticsearch.", new Object[]{record.getKey(), value.getName(), value.getScopeKey(), value.getProcessInstanceKey(), this.configuration.index.ignoreVariablesAbove, size});
        }
    }

    public void bulk(Map<String, Object> command, Record<?> record) {
        String serializedCommand;
        try {
            serializedCommand = MAPPER.writeValueAsString(command);
        }
        catch (IOException e) {
            throw new ElasticsearchExporterException("Failed to serialize bulk request command to JSON", e);
        }
        String jsonCommand = serializedCommand + "\n" + record.toJson();
        if (this.bulkRequest.isEmpty() || !this.bulkRequest.get(this.bulkRequest.size() - 1).equals(jsonCommand)) {
            this.bulkRequest.add(jsonCommand);
        }
    }

    public void flush() {
        BulkResponse bulkResponse;
        if (this.bulkRequest.isEmpty()) {
            return;
        }
        int bulkSize = this.bulkRequest.size();
        this.metrics.recordBulkSize(bulkSize);
        int bulkMemorySize = this.getBulkMemorySize();
        this.metrics.recordBulkMemorySize(bulkMemorySize);
        try {
            bulkResponse = this.exportBulk();
        }
        catch (IOException e) {
            throw new ElasticsearchExporterException("Failed to flush bulk", e);
        }
        boolean success = this.checkBulkResponse(bulkResponse);
        if (!success) {
            throw new ElasticsearchExporterException("Failed to flush all items of the bulk");
        }
        this.bulkRequest = new ArrayList<String>();
    }

    private boolean checkBulkResponse(BulkResponse bulkResponse) {
        boolean hasErrors = bulkResponse.hasErrors();
        if (hasErrors) {
            bulkResponse.getItems().stream().flatMap(item -> Optional.ofNullable(item.getIndex()).stream()).flatMap(index -> Optional.ofNullable(index.getError()).stream()).collect(Collectors.groupingBy(BulkItemError::getType)).forEach((errorType, errors) -> this.log.warn("Failed to flush {} item(s) of bulk request [type: {}, reason: {}]", new Object[]{errors.size(), errorType, ((BulkItemError)errors.get(0)).getReason()}));
        }
        return !hasErrors;
    }

    private BulkResponse exportBulk() throws IOException {
        try (Histogram.Timer timer = this.metrics.measureFlushDuration();){
            Request request = new Request("POST", "/_bulk");
            request.setJsonEntity(String.join((CharSequence)"\n", this.bulkRequest) + "\n");
            Response response = this.client.performRequest(request);
            BulkResponse bulkResponse = (BulkResponse)MAPPER.readValue(response.getEntity().getContent(), BulkResponse.class);
            return bulkResponse;
        }
    }

    public boolean shouldFlush() {
        return this.bulkRequest.size() >= this.configuration.bulk.size || this.getBulkMemorySize() >= this.configuration.bulk.memoryLimit;
    }

    private int getBulkMemorySize() {
        return this.bulkRequest.stream().mapToInt(String::length).sum();
    }

    public boolean putIndexTemplate(ValueType valueType) {
        String templateName = this.indexPrefixForValueType(valueType);
        String aliasName = this.aliasNameForValueType(valueType);
        String filename = ElasticsearchClient.indexTemplateForValueType(valueType);
        return this.putIndexTemplate(templateName, aliasName, filename);
    }

    public boolean putIndexTemplate(String templateName, String aliasName, String filename) {
        Map<String, Object> template;
        block9: {
            try (InputStream inputStream = ElasticsearchExporter.class.getResourceAsStream(filename);){
                if (inputStream != null) {
                    template = this.convertToMap(XContentType.JSON.xContent(), inputStream);
                    break block9;
                }
                throw new ElasticsearchExporterException("Failed to find index template in classpath " + filename);
            }
            catch (IOException e) {
                throw new ElasticsearchExporterException("Failed to load index template from classpath " + filename, e);
            }
        }
        template.put("index_patterns", Collections.singletonList(templateName + "_*"));
        template.put("aliases", Collections.singletonMap(aliasName, Collections.emptyMap()));
        return this.putIndexTemplate(templateName, template);
    }

    private boolean putIndexTemplate(String templateName, Object body) {
        try {
            Request request = new Request("PUT", "/_template/" + templateName);
            request.addParameter("include_type_name", "true");
            request.setJsonEntity(MAPPER.writeValueAsString(body));
            Response response = this.client.performRequest(request);
            PutIndexTemplateResponse putIndexTemplateResponse = (PutIndexTemplateResponse)MAPPER.readValue(response.getEntity().getContent(), PutIndexTemplateResponse.class);
            return putIndexTemplateResponse.isAcknowledged();
        }
        catch (IOException e) {
            throw new ElasticsearchExporterException("Failed to put index template", e);
        }
    }

    private RestClient createClient() {
        HttpHost[] httpHosts = ElasticsearchClient.urlsToHttpHosts(this.configuration.url);
        RestClientBuilder builder = RestClient.builder((HttpHost[])httpHosts).setHttpClientConfigCallback(this::setHttpClientConfigCallback);
        return builder.build();
    }

    private HttpAsyncClientBuilder setHttpClientConfigCallback(HttpAsyncClientBuilder builder) {
        builder.setDefaultIOReactorConfig(IOReactorConfig.custom().setIoThreadCount(1).build());
        if (this.configuration.hasAuthenticationPresent()) {
            this.setupBasicAuthentication(builder);
        }
        return builder;
    }

    private void setupBasicAuthentication(HttpAsyncClientBuilder builder) {
        BasicCredentialsProvider credentialsProvider = new BasicCredentialsProvider();
        credentialsProvider.setCredentials(AuthScope.ANY, (Credentials)new UsernamePasswordCredentials(this.configuration.getAuthentication().getUsername(), this.configuration.getAuthentication().getPassword()));
        builder.setDefaultCredentialsProvider((CredentialsProvider)credentialsProvider);
    }

    private static HttpHost[] urlsToHttpHosts(String urls) {
        return (HttpHost[])Arrays.stream(urls.split(",")).map(String::trim).map(ElasticsearchClient::urlToHttpHost).toArray(HttpHost[]::new);
    }

    private static HttpHost urlToHttpHost(String url) {
        URI uri;
        try {
            uri = new URI(url);
        }
        catch (URISyntaxException e) {
            throw new ElasticsearchExporterException("Failed to parse url " + url, e);
        }
        return new HttpHost(uri.getHost(), uri.getPort(), uri.getScheme());
    }

    protected String indexFor(Record<?> record) {
        Instant timestamp = Instant.ofEpochMilli(record.getTimestamp());
        return this.indexPrefixForValueTypeWithDelimiter(record.getValueType()) + this.formatter.format(timestamp);
    }

    protected String idFor(Record<?> record) {
        return record.getPartitionId() + ALIAS_DELIMITER + record.getPosition();
    }

    protected String typeFor(Record<?> record) {
        return "_doc";
    }

    protected String indexPrefixForValueTypeWithDelimiter(ValueType valueType) {
        return this.indexPrefixForValueType(valueType) + INDEX_DELIMITER;
    }

    private String aliasNameForValueType(ValueType valueType) {
        return this.configuration.index.prefix + ALIAS_DELIMITER + ElasticsearchClient.valueTypeToString(valueType);
    }

    private String indexPrefixForValueType(ValueType valueType) {
        String version = VersionUtil.getVersionLowerCase();
        return this.configuration.index.prefix + INDEX_DELIMITER + ElasticsearchClient.valueTypeToString(valueType) + INDEX_DELIMITER + version;
    }

    private static String valueTypeToString(ValueType valueType) {
        return valueType.name().toLowerCase().replaceAll(INDEX_DELIMITER, ALIAS_DELIMITER);
    }

    private static String indexTemplateForValueType(ValueType valueType) {
        return String.format(INDEX_TEMPLATE_FILENAME_PATTERN, ElasticsearchClient.valueTypeToString(valueType));
    }

    private Map<String, Object> convertToMap(XContent content, InputStream input) {
        Map map;
        block8: {
            XContentParser parser = content.createParser(NamedXContentRegistry.EMPTY, DeprecationHandler.THROW_UNSUPPORTED_OPERATION, input);
            try {
                map = parser.mapOrdered();
                if (parser == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (parser != null) {
                        try {
                            parser.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new ElasticsearchExporterException("Failed to parse content to map", e);
                }
            }
            parser.close();
        }
        return map;
    }

    private Map<String, Object> newIndexCommand(Record<?> record) {
        HashMap<String, Object> command = new HashMap<String, Object>();
        HashMap<String, String> contents = new HashMap<String, String>();
        contents.put("_index", this.indexFor(record));
        contents.put("_type", this.typeFor(record));
        contents.put("_id", this.idFor(record));
        contents.put("routing", String.valueOf(record.getPartitionId()));
        command.put("index", contents);
        return command;
    }
}

