/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.cluster.messaging.impl;

import com.google.common.collect.Maps;
import io.atomix.cluster.messaging.MessagingException;
import io.atomix.cluster.messaging.impl.ClientConnection;
import io.atomix.cluster.messaging.impl.ProtocolReply;
import java.net.ConnectException;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.math3.stat.descriptive.DescriptiveStatistics;
import org.apache.commons.math3.stat.descriptive.SynchronizedDescriptiveStatistics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class AbstractClientConnection
implements ClientConnection {
    private static final int WINDOW_SIZE = 10;
    private static final int MIN_SAMPLES = 50;
    private static final int TIMEOUT_FACTOR = 5;
    private static final long MIN_TIMEOUT_MILLIS = 100L;
    private static final long MAX_TIMEOUT_MILLIS = 5000L;
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final ScheduledExecutorService executorService;
    private final Map<Long, Callback> callbacks = Maps.newConcurrentMap();
    private final Map<String, DescriptiveStatistics> replySamples = new ConcurrentHashMap<String, DescriptiveStatistics>();
    private final AtomicBoolean closed = new AtomicBoolean(false);

    AbstractClientConnection(ScheduledExecutorService executorService) {
        this.executorService = executorService;
    }

    @Override
    public void dispatch(ProtocolReply message) {
        Callback callback = this.callbacks.remove(message.id());
        if (callback != null) {
            if (message.status() == ProtocolReply.Status.OK) {
                callback.complete(message.payload());
            } else if (message.status() == ProtocolReply.Status.ERROR_NO_HANDLER) {
                callback.completeExceptionally(new MessagingException.NoRemoteHandler());
            } else if (message.status() == ProtocolReply.Status.ERROR_HANDLER_EXCEPTION) {
                callback.completeExceptionally(new MessagingException.RemoteHandlerFailure());
            } else if (message.status() == ProtocolReply.Status.PROTOCOL_EXCEPTION) {
                callback.completeExceptionally(new MessagingException.ProtocolException());
            }
        } else {
            this.log.debug("Received a reply for message id:[{}] but was unable to locate the request handle", (Object)message.id());
        }
    }

    private void addReplyTime(String type, long replyTime) {
        DescriptiveStatistics samples = this.replySamples.get(type);
        if (samples == null) {
            samples = this.replySamples.computeIfAbsent(type, t -> new SynchronizedDescriptiveStatistics(10));
        }
        samples.addValue((double)replyTime);
    }

    private long getTimeoutMillis(String type, Duration timeout) {
        return timeout != null ? timeout.toMillis() : this.computeTimeoutMillis(type);
    }

    private long computeTimeoutMillis(String type) {
        DescriptiveStatistics samples = this.replySamples.get(type);
        if (samples == null || samples.getN() < 50L) {
            return 5000L;
        }
        return Math.min(Math.max((long)samples.getMax() * 5L, 100L), 5000L);
    }

    @Override
    public void close() {
        if (this.closed.compareAndSet(false, true)) {
            for (Callback callback : this.callbacks.values()) {
                callback.completeExceptionally(new ConnectException());
            }
        }
    }

    final class Callback {
        private final long id;
        private final String type;
        private final long time = System.currentTimeMillis();
        private final long timeout;
        private final ScheduledFuture<?> scheduledFuture;
        private final CompletableFuture<byte[]> replyFuture;

        Callback(long id, String type, Duration timeout, CompletableFuture<byte[]> future) {
            this.id = id;
            this.type = type;
            this.timeout = AbstractClientConnection.this.getTimeoutMillis(type, timeout);
            this.scheduledFuture = AbstractClientConnection.this.executorService.schedule(this::timeout, this.timeout, TimeUnit.MILLISECONDS);
            this.replyFuture = future;
            future.thenRun(() -> AbstractClientConnection.this.addReplyTime(type, System.currentTimeMillis() - this.time));
            AbstractClientConnection.this.callbacks.put(id, this);
        }

        String type() {
            return this.type;
        }

        private void timeout() {
            this.replyFuture.completeExceptionally(new TimeoutException("Request type " + this.type + " timed out in " + this.timeout + " milliseconds"));
            AbstractClientConnection.this.callbacks.remove(this.id);
        }

        void complete(byte[] value) {
            this.scheduledFuture.cancel(false);
            this.replyFuture.complete(value);
        }

        void completeExceptionally(Throwable error) {
            this.scheduledFuture.cancel(false);
            this.replyFuture.completeExceptionally(error);
            AbstractClientConnection.this.callbacks.remove(this.id);
        }
    }
}

