/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.raft.impl;

import com.google.common.base.Preconditions;
import io.atomix.cluster.ClusterMembershipService;
import io.atomix.cluster.MemberId;
import io.atomix.raft.RaftCommitListener;
import io.atomix.raft.RaftException;
import io.atomix.raft.RaftRoleChangeListener;
import io.atomix.raft.RaftServer;
import io.atomix.raft.RaftThreadContextFactory;
import io.atomix.raft.cluster.RaftMember;
import io.atomix.raft.cluster.impl.DefaultRaftMember;
import io.atomix.raft.cluster.impl.RaftClusterContext;
import io.atomix.raft.impl.zeebe.LogCompactor;
import io.atomix.raft.metrics.RaftReplicationMetrics;
import io.atomix.raft.metrics.RaftRoleMetrics;
import io.atomix.raft.protocol.AppendRequest;
import io.atomix.raft.protocol.ConfigureRequest;
import io.atomix.raft.protocol.InstallRequest;
import io.atomix.raft.protocol.PollRequest;
import io.atomix.raft.protocol.RaftResponse;
import io.atomix.raft.protocol.RaftServerProtocol;
import io.atomix.raft.protocol.ReconfigureRequest;
import io.atomix.raft.protocol.TransferRequest;
import io.atomix.raft.protocol.VoteRequest;
import io.atomix.raft.roles.ActiveRole;
import io.atomix.raft.roles.CandidateRole;
import io.atomix.raft.roles.FollowerRole;
import io.atomix.raft.roles.InactiveRole;
import io.atomix.raft.roles.LeaderRole;
import io.atomix.raft.roles.PassiveRole;
import io.atomix.raft.roles.PromotableRole;
import io.atomix.raft.roles.RaftRole;
import io.atomix.raft.storage.RaftStorage;
import io.atomix.raft.storage.log.RaftLog;
import io.atomix.raft.storage.log.RaftLogReader;
import io.atomix.raft.storage.system.MetaStore;
import io.atomix.raft.zeebe.EntryValidator;
import io.atomix.storage.StorageException;
import io.atomix.utils.concurrent.ComposableFuture;
import io.atomix.utils.concurrent.ThreadContext;
import io.atomix.utils.concurrent.Threads;
import io.atomix.utils.logging.ContextualLoggerFactory;
import io.atomix.utils.logging.LoggerContext;
import io.zeebe.snapshots.raft.PersistedSnapshot;
import io.zeebe.snapshots.raft.ReceivableSnapshotStore;
import java.time.Duration;
import java.util.Objects;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.slf4j.Logger;

public class RaftContext
implements AutoCloseable {
    protected final String name;
    protected final ThreadContext threadContext;
    protected final ClusterMembershipService membershipService;
    protected final RaftClusterContext cluster;
    protected final RaftServerProtocol protocol;
    protected final RaftStorage storage;
    private final Logger log;
    private final Set<RaftRoleChangeListener> roleChangeListeners = new CopyOnWriteArraySet<RaftRoleChangeListener>();
    private final Set<Consumer<State>> stateChangeListeners = new CopyOnWriteArraySet<Consumer<State>>();
    private final Set<Consumer<RaftMember>> electionListeners = new CopyOnWriteArraySet<Consumer<RaftMember>>();
    private final Set<RaftCommitListener> commitListeners = new CopyOnWriteArraySet<RaftCommitListener>();
    private final Set<Runnable> failureListeners = new CopyOnWriteArraySet<Runnable>();
    private final RaftRoleMetrics raftRoleMetrics;
    private final RaftReplicationMetrics replicationMetrics;
    private final MetaStore meta;
    private final RaftLog raftLog;
    private final RaftLogReader logReader;
    private final ReceivableSnapshotStore persistedSnapshotStore;
    private final LogCompactor logCompactor;
    private volatile State state = State.ACTIVE;
    private RaftRole role = new InactiveRole(this);
    private Duration electionTimeout = Duration.ofMillis(500L);
    private Duration heartbeatInterval = Duration.ofMillis(150L);
    private volatile MemberId leader;
    private volatile long term;
    private MemberId lastVotedFor;
    private long commitIndex;
    private volatile long firstCommitIndex;
    private volatile boolean started;
    private EntryValidator entryValidator;
    private final int maxAppendBatchSize;
    private final int maxAppendsPerFollower;
    private final Random random;
    private PersistedSnapshot currentSnapshot;

    public RaftContext(String name, MemberId localMemberId, ClusterMembershipService membershipService, RaftServerProtocol protocol, RaftStorage storage, RaftThreadContextFactory threadContextFactory, int maxAppendBatchSize, int maxAppendsPerFollower, Supplier<Random> randomFactory) {
        this.name = (String)Preconditions.checkNotNull((Object)name, (Object)"name cannot be null");
        this.membershipService = (ClusterMembershipService)Preconditions.checkNotNull((Object)membershipService, (Object)"membershipService cannot be null");
        this.protocol = (RaftServerProtocol)Preconditions.checkNotNull((Object)protocol, (Object)"protocol cannot be null");
        this.storage = (RaftStorage)Preconditions.checkNotNull((Object)storage, (Object)"storage cannot be null");
        this.random = randomFactory.get();
        this.log = ContextualLoggerFactory.getLogger(this.getClass(), (LoggerContext)LoggerContext.builder(RaftServer.class).addValue((Object)name).build());
        if (!storage.lock((String)((Object)localMemberId.id()))) {
            throw new StorageException("Failed to acquire storage lock; ensure each Raft server is configured with a distinct storage directory");
        }
        String baseThreadName = String.format("raft-server-%s-%s", localMemberId.id(), name);
        this.threadContext = threadContextFactory.createContext(Threads.namedThreads((String)baseThreadName, (Logger)this.log), this::onUncaughtException);
        this.meta = storage.openMetaStore();
        this.term = this.meta.loadTerm();
        this.lastVotedFor = this.meta.loadVote();
        this.raftLog = storage.openLog();
        this.logReader = this.raftLog.openReader(1L, RaftLogReader.Mode.ALL);
        this.persistedSnapshotStore = storage.getPersistedSnapshotStore();
        this.persistedSnapshotStore.addSnapshotListener(this::setSnapshot);
        this.persistedSnapshotStore.getLatestSnapshot().ifPresent(persistedSnapshot -> {
            this.currentSnapshot = persistedSnapshot;
        });
        this.verifySnapshotLogConsistent();
        this.logCompactor = new LogCompactor(this);
        this.maxAppendBatchSize = maxAppendBatchSize;
        this.maxAppendsPerFollower = maxAppendsPerFollower;
        this.cluster = new RaftClusterContext(localMemberId, this);
        this.registerHandlers(protocol);
        this.raftRoleMetrics = new RaftRoleMetrics(name);
        this.replicationMetrics = new RaftReplicationMetrics(name);
        this.replicationMetrics.setAppendIndex(this.raftLog.getLastIndex());
        this.started = true;
    }

    private void verifySnapshotLogConsistent() {
        long currentSnapshotIndex = this.getCurrentSnapshotIndex();
        if (currentSnapshotIndex <= 0L && this.raftLog.getFirstIndex() != 1L || currentSnapshotIndex > 0L && this.currentSnapshot.getIndex() + 1L < this.raftLog.getFirstIndex()) {
            throw new IllegalStateException(String.format("Expected to find a snapshot at index >= log's first index %d, but found snapshot %d", this.raftLog.getFirstIndex(), currentSnapshotIndex));
        }
    }

    private void setSnapshot(PersistedSnapshot persistedSnapshot) {
        this.threadContext.execute(() -> {
            this.currentSnapshot = persistedSnapshot;
        });
    }

    private void onUncaughtException(Throwable error) {
        this.log.error("An uncaught exception occurred, transition to inactive role", error);
        try {
            this.transition(RaftServer.Role.INACTIVE);
        }
        catch (Throwable e) {
            this.log.error("An error occurred when transitioning to {}, closing the raft context", (Object)RaftServer.Role.INACTIVE, (Object)error);
            this.close();
        }
        this.notifyFailureListeners();
    }

    private void notifyFailureListeners() {
        try {
            this.failureListeners.forEach(Runnable::run);
        }
        catch (Exception exception) {
            this.log.error("Could not notify failure listeners", (Throwable)exception);
        }
    }

    private void registerHandlers(RaftServerProtocol protocol) {
        protocol.registerConfigureHandler(request -> this.runOnContext(() -> this.role.onConfigure((ConfigureRequest)request)));
        protocol.registerInstallHandler(request -> this.runOnContext(() -> this.role.onInstall((InstallRequest)request)));
        protocol.registerReconfigureHandler(request -> this.runOnContext(() -> this.role.onReconfigure((ReconfigureRequest)request)));
        protocol.registerTransferHandler(request -> this.runOnContext(() -> this.role.onTransfer((TransferRequest)request)));
        protocol.registerAppendHandler(request -> this.runOnContext(() -> this.role.onAppend((AppendRequest)request)));
        protocol.registerPollHandler(request -> this.runOnContext(() -> this.role.onPoll((PollRequest)request)));
        protocol.registerVoteHandler(request -> this.runOnContext(() -> this.role.onVote((VoteRequest)request)));
    }

    private <R extends RaftResponse> CompletableFuture<R> runOnContext(Supplier<CompletableFuture<R>> function) {
        CompletableFuture future = new CompletableFuture();
        this.threadContext.execute(() -> ((CompletableFuture)function.get()).whenComplete((response, error) -> {
            if (error == null) {
                future.complete(response);
            } else {
                future.completeExceptionally((Throwable)error);
            }
        }));
        return future;
    }

    public MemberId localMemberId() {
        return this.membershipService.getLocalMember().id();
    }

    public int getMaxAppendBatchSize() {
        return this.maxAppendBatchSize;
    }

    public int getMaxAppendsPerFollower() {
        return this.maxAppendsPerFollower;
    }

    public void addRoleChangeListener(RaftRoleChangeListener listener) {
        this.roleChangeListeners.add(listener);
    }

    public void removeRoleChangeListener(RaftRoleChangeListener listener) {
        this.roleChangeListeners.remove(listener);
    }

    public void addFailureListener(Runnable failureListener) {
        this.failureListeners.add(failureListener);
    }

    public void removeFailureListener(Runnable failureListener) {
        this.failureListeners.remove(failureListener);
    }

    public void awaitState(State state, final Consumer<State> listener) {
        if (this.state == state) {
            listener.accept(this.state);
        } else {
            this.addStateChangeListener(new Consumer<State>(){

                @Override
                public void accept(State state) {
                    listener.accept(state);
                    RaftContext.this.removeStateChangeListener(this);
                }
            });
        }
    }

    public void addStateChangeListener(Consumer<State> listener) {
        this.stateChangeListeners.add(listener);
    }

    public void removeStateChangeListener(Consumer<State> listener) {
        this.stateChangeListeners.remove(listener);
    }

    public void addCommitListener(RaftCommitListener commitListener) {
        this.commitListeners.add(commitListener);
    }

    public void removeCommitListener(RaftCommitListener commitListener) {
        this.commitListeners.remove(commitListener);
    }

    public void notifyCommitListeners(long lastCommitIndex) {
        this.commitListeners.forEach(listener -> listener.onCommit(lastCommitIndex));
    }

    public long setCommitIndex(long commitIndex) {
        Preconditions.checkArgument((commitIndex >= 0L ? 1 : 0) != 0, (Object)"commitIndex must be positive");
        long previousCommitIndex = this.commitIndex;
        if (commitIndex > previousCommitIndex) {
            long configurationIndex;
            this.commitIndex = commitIndex;
            this.raftLog.setCommitIndex(Math.min(commitIndex, this.raftLog.getLastIndex()));
            if (this.raftLog.shouldFlushExplicitly() && this.isLeader()) {
                this.raftLog.flush();
            }
            if ((configurationIndex = this.cluster.getConfiguration().index()) > previousCommitIndex && configurationIndex <= commitIndex) {
                this.cluster.commit();
            }
            this.setFirstCommitIndex(commitIndex);
            if (this.state == State.ACTIVE && commitIndex >= this.firstCommitIndex) {
                this.state = State.READY;
                this.stateChangeListeners.forEach(l -> l.accept(this.state));
            }
            this.replicationMetrics.setCommitIndex(commitIndex);
        }
        return previousCommitIndex;
    }

    public CompletableFuture<Void> compact() {
        ComposableFuture future = new ComposableFuture();
        this.threadContext.execute(() -> this.logCompactor.compact().whenComplete((BiConsumer)future));
        return future;
    }

    public CompletableFuture<Void> anoint() {
        if (this.role.role() == RaftServer.Role.LEADER) {
            return CompletableFuture.completedFuture(null);
        }
        final CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.threadContext.execute(() -> {
            Consumer<RaftMember> electionListener = new Consumer<RaftMember>(){

                @Override
                public void accept(RaftMember member) {
                    if (member.memberId().equals(RaftContext.this.cluster.getLocalMember().memberId())) {
                        future.complete(null);
                    } else {
                        future.completeExceptionally(new RaftException.ProtocolException("Failed to transfer leadership", new Object[0]));
                    }
                    RaftContext.this.removeLeaderElectionListener(this);
                }
            };
            this.addLeaderElectionListener(electionListener);
            RaftMember member = this.getCluster().getLocalMember();
            DefaultRaftMember leader = this.getLeader();
            if (leader != null) {
                this.protocol.transfer(leader.memberId(), TransferRequest.builder().withMember(member.memberId()).build()).whenCompleteAsync((response, error) -> {
                    if (error != null) {
                        future.completeExceptionally((Throwable)error);
                    } else if (response.status() == RaftResponse.Status.ERROR) {
                        future.completeExceptionally((Throwable)((Object)response.error().createException()));
                    } else {
                        this.transition(RaftServer.Role.CANDIDATE);
                    }
                }, (Executor)this.threadContext);
            } else {
                this.transition(RaftServer.Role.CANDIDATE);
            }
        });
        return future;
    }

    public void addLeaderElectionListener(Consumer<RaftMember> listener) {
        this.electionListeners.add(listener);
    }

    public void removeLeaderElectionListener(Consumer<RaftMember> listener) {
        this.electionListeners.remove(listener);
    }

    public RaftClusterContext getCluster() {
        return this.cluster;
    }

    public DefaultRaftMember getLeader() {
        MemberId leader = this.leader;
        return leader != null ? this.cluster.getMember(leader) : null;
    }

    public void transition(RaftServer.Role role) {
        this.checkThread();
        Preconditions.checkNotNull((Object)((Object)role));
        if (this.role.role() == role) {
            return;
        }
        this.log.info("Transitioning to {}", (Object)role);
        try {
            this.role.stop().get();
        }
        catch (InterruptedException | ExecutionException e) {
            throw new IllegalStateException("failed to close Raft state", e);
        }
        try {
            this.role = this.createRole(role);
            this.role.start().get();
        }
        catch (InterruptedException | ExecutionException e) {
            throw new IllegalStateException("failed to initialize Raft state", e);
        }
        if (this.role.role() == role) {
            if (this.role.role() == RaftServer.Role.LEADER) {
                LeaderRole leaderRole = (LeaderRole)this.role;
                leaderRole.onInitialEntriesCommitted(() -> {
                    if (this.role == leaderRole) {
                        this.notifyRoleChangeListeners();
                    }
                });
            } else {
                this.notifyRoleChangeListeners();
            }
        }
    }

    private void notifyRoleChangeListeners() {
        try {
            this.roleChangeListeners.forEach(l -> l.onNewRole(this.role.role(), this.getTerm()));
        }
        catch (Exception exception) {
            this.log.error("Unexpected error on calling role change listeners.", (Throwable)exception);
        }
    }

    public void checkThread() {
        this.threadContext.checkThread();
    }

    private RaftRole createRole(RaftServer.Role role) {
        switch (role) {
            case INACTIVE: {
                return new InactiveRole(this);
            }
            case PASSIVE: {
                return new PassiveRole(this);
            }
            case PROMOTABLE: {
                return new PromotableRole(this);
            }
            case FOLLOWER: {
                this.raftRoleMetrics.becomingFollower();
                return new FollowerRole(this);
            }
            case CANDIDATE: {
                this.raftRoleMetrics.becomingCandidate();
                return new CandidateRole(this);
            }
            case LEADER: {
                this.raftRoleMetrics.becomingLeader();
                return new LeaderRole(this);
            }
        }
        throw new AssertionError();
    }

    public void transition(RaftMember.Type type) {
        switch (type) {
            case ACTIVE: {
                if (this.role instanceof ActiveRole) break;
                this.transition(RaftServer.Role.FOLLOWER);
                break;
            }
            case PROMOTABLE: {
                if (this.role.role() == RaftServer.Role.PROMOTABLE) break;
                this.transition(RaftServer.Role.PROMOTABLE);
                break;
            }
            case PASSIVE: {
                if (this.role.role() == RaftServer.Role.PASSIVE) break;
                this.transition(RaftServer.Role.PASSIVE);
                break;
            }
            default: {
                if (this.role.role() == RaftServer.Role.INACTIVE) break;
                this.transition(RaftServer.Role.INACTIVE);
            }
        }
    }

    @Override
    public void close() {
        this.started = false;
        this.unregisterHandlers(this.protocol);
        this.logCompactor.close();
        try {
            this.raftLog.close();
        }
        catch (Exception e) {
            this.log.error("Failed to close raft log", (Throwable)e);
        }
        try {
            this.meta.close();
        }
        catch (Exception e) {
            this.log.error("Failed to close metastore", (Throwable)e);
        }
        try {
            this.persistedSnapshotStore.close();
        }
        catch (Exception e) {
            this.log.error("Failed to close snapshot store", (Throwable)e);
        }
        this.threadContext.close();
    }

    private void unregisterHandlers(RaftServerProtocol protocol) {
        protocol.unregisterConfigureHandler();
        protocol.unregisterInstallHandler();
        protocol.unregisterReconfigureHandler();
        protocol.unregisterTransferHandler();
        protocol.unregisterAppendHandler();
        protocol.unregisterPollHandler();
        protocol.unregisterVoteHandler();
    }

    public String toString() {
        return this.getClass().getCanonicalName();
    }

    public long getCommitIndex() {
        return this.commitIndex;
    }

    public Duration getElectionTimeout() {
        return this.electionTimeout;
    }

    public void setElectionTimeout(Duration electionTimeout) {
        this.electionTimeout = electionTimeout;
    }

    public long getFirstCommitIndex() {
        return this.firstCommitIndex;
    }

    public void setFirstCommitIndex(long firstCommitIndex) {
        if (this.firstCommitIndex == 0L) {
            this.firstCommitIndex = firstCommitIndex;
            this.log.info("Setting firstCommitIndex to {}. RaftServer is ready only after it has committed events upto this index", (Object)firstCommitIndex);
        }
    }

    public Duration getHeartbeatInterval() {
        return this.heartbeatInterval;
    }

    public void setHeartbeatInterval(Duration heartbeatInterval) {
        this.heartbeatInterval = (Duration)Preconditions.checkNotNull((Object)heartbeatInterval, (Object)"heartbeatInterval cannot be null");
    }

    public EntryValidator getEntryValidator() {
        return this.entryValidator;
    }

    public void setEntryValidator(EntryValidator validator) {
        this.entryValidator = validator;
    }

    public MemberId getLastVotedFor() {
        return this.lastVotedFor;
    }

    public void setLastVotedFor(MemberId candidate) {
        Preconditions.checkState((this.lastVotedFor == null || candidate == null ? 1 : 0) != 0, (Object)"Already voted for another candidate");
        DefaultRaftMember member = this.cluster.getMember(candidate);
        Preconditions.checkState((member != null ? 1 : 0) != 0, (String)"Unknown candidate: %d", (Object)candidate);
        this.lastVotedFor = candidate;
        this.meta.storeVote(this.lastVotedFor);
        if (candidate != null) {
            this.log.debug("Voted for {}", (Object)member.memberId());
        } else {
            this.log.trace("Reset last voted for");
        }
    }

    public RaftLog getLog() {
        return this.raftLog;
    }

    public RaftLogReader getLogReader() {
        return this.logReader;
    }

    public ClusterMembershipService getMembershipService() {
        return this.membershipService;
    }

    public MetaStore getMetaStore() {
        return this.meta;
    }

    public String getName() {
        return this.name;
    }

    public RaftServerProtocol getProtocol() {
        return this.protocol;
    }

    public RaftRole getRaftRole() {
        return this.role;
    }

    public RaftRoleMetrics getRaftRoleMetrics() {
        return this.raftRoleMetrics;
    }

    public RaftServer.Role getRole() {
        return this.role.role();
    }

    public LogCompactor getLogCompactor() {
        return this.logCompactor;
    }

    public ReceivableSnapshotStore getPersistedSnapshotStore() {
        return this.persistedSnapshotStore;
    }

    public State getState() {
        return this.state;
    }

    public RaftStorage getStorage() {
        return this.storage;
    }

    public long getTerm() {
        return this.term;
    }

    public void setTerm(long term) {
        if (term > this.term) {
            this.term = term;
            this.leader = null;
            this.lastVotedFor = null;
            this.meta.storeTerm(this.term);
            this.meta.storeVote(this.lastVotedFor);
            this.log.debug("Set term {}", (Object)term);
        }
    }

    public ThreadContext getThreadContext() {
        return this.threadContext;
    }

    public boolean isLeader() {
        MemberId leader = this.leader;
        return leader != null && leader.equals(this.cluster.getLocalMember().memberId());
    }

    public void setLeader(MemberId leader) {
        if (!Objects.equals(this.leader, leader)) {
            if (leader == null) {
                this.leader = null;
            } else {
                DefaultRaftMember member = this.cluster.getMember(leader);
                if (member != null) {
                    this.leader = leader;
                    this.log.info("Found leader {}", (Object)member.memberId());
                    this.electionListeners.forEach(l -> l.accept(member));
                }
            }
            this.log.trace("Set leader {}", (Object)this.leader);
        }
    }

    public PersistedSnapshot getCurrentSnapshot() {
        return this.currentSnapshot;
    }

    public long getCurrentSnapshotIndex() {
        return this.currentSnapshot != null ? this.currentSnapshot.getIndex() : 0L;
    }

    public boolean isRunning() {
        return this.started;
    }

    public RaftReplicationMetrics getReplicationMetrics() {
        return this.replicationMetrics;
    }

    public Random getRandom() {
        return this.random;
    }

    public static enum State {
        ACTIVE,
        READY;

    }
}

