/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.raft.roles;

import io.atomix.cluster.ClusterMembershipEvent;
import io.atomix.cluster.ClusterMembershipEventListener;
import io.atomix.cluster.Member;
import io.atomix.cluster.MemberId;
import io.atomix.raft.RaftServer;
import io.atomix.raft.cluster.impl.DefaultRaftMember;
import io.atomix.raft.cluster.impl.RaftMemberContext;
import io.atomix.raft.impl.RaftContext;
import io.atomix.raft.protocol.AppendRequest;
import io.atomix.raft.protocol.AppendResponse;
import io.atomix.raft.protocol.ConfigureRequest;
import io.atomix.raft.protocol.ConfigureResponse;
import io.atomix.raft.protocol.InstallRequest;
import io.atomix.raft.protocol.InstallResponse;
import io.atomix.raft.protocol.PollRequest;
import io.atomix.raft.protocol.PollResponse;
import io.atomix.raft.protocol.VoteRequest;
import io.atomix.raft.protocol.VoteResponse;
import io.atomix.raft.roles.ActiveRole;
import io.atomix.raft.roles.RaftRole;
import io.atomix.raft.storage.log.IndexedRaftRecord;
import io.atomix.raft.utils.Quorum;
import io.atomix.utils.concurrent.Scheduled;
import java.time.Duration;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

public final class FollowerRole
extends ActiveRole {
    private Scheduled heartbeatTimer;
    private final ClusterMembershipEventListener clusterListener = this::handleClusterEvent;
    private long lastHeartbeat;

    public FollowerRole(RaftContext context) {
        super(context);
    }

    @Override
    public synchronized CompletableFuture<RaftRole> start() {
        if (this.raft.getCluster().getActiveMemberStates().isEmpty()) {
            this.log.info("Single member cluster. Transitioning directly to candidate.");
            this.raft.transition(RaftServer.Role.CANDIDATE);
            return CompletableFuture.completedFuture(this);
        }
        this.raft.getMembershipService().addListener(this.clusterListener);
        return ((CompletableFuture)super.start().thenRun(this::resetHeartbeatTimeout)).thenApply(v -> this);
    }

    @Override
    public synchronized CompletableFuture<Void> stop() {
        this.raft.getMembershipService().removeListener(this.clusterListener);
        this.cancelHeartbeatTimer();
        return super.stop();
    }

    @Override
    public RaftServer.Role role() {
        return RaftServer.Role.FOLLOWER;
    }

    @Override
    public CompletableFuture<InstallResponse> onInstall(InstallRequest request) {
        CompletableFuture<InstallResponse> future = super.onInstall(request);
        if (this.isRequestFromCurrentLeader(request.currentTerm(), request.leader())) {
            this.resetHeartbeatTimeout();
        }
        return future;
    }

    private void cancelHeartbeatTimer() {
        if (this.heartbeatTimer != null) {
            this.log.trace("Cancelling heartbeat timer");
            this.heartbeatTimer.cancel();
            this.heartbeatTimer = null;
        }
    }

    private void handleClusterEvent(ClusterMembershipEvent event) {
        this.raft.getThreadContext().execute(() -> {
            DefaultRaftMember leader = this.raft.getLeader();
            if (leader != null && event.type() == ClusterMembershipEvent.Type.MEMBER_REMOVED && ((Member)event.subject()).id().equals(leader.memberId())) {
                this.log.info("Known leader {} was removed from cluster, sending poll requests", (Object)leader.memberId());
                this.raft.setLeader(null);
                this.sendPollRequests();
            }
        });
    }

    private void sendPollRequests() {
        AtomicBoolean complete = new AtomicBoolean();
        Set votingMembers = this.raft.getCluster().getActiveMemberStates().stream().map(RaftMemberContext::getMember).collect(Collectors.toSet());
        if (votingMembers.isEmpty()) {
            this.raft.transition(RaftServer.Role.CANDIDATE);
            return;
        }
        Quorum quorum = new Quorum(this.raft.getCluster().getQuorum(), elected -> {
            complete.set(true);
            if (this.raft.getLeader() == null && elected.booleanValue()) {
                this.raft.transition(RaftServer.Role.CANDIDATE);
            } else {
                this.resetHeartbeatTimeout();
            }
        });
        IndexedRaftRecord lastEntry = this.raft.getLog().getLastEntry();
        long lastTerm = lastEntry != null ? lastEntry.entry().term() : 0L;
        this.log.debug("Polling members {}", votingMembers);
        for (DefaultRaftMember member : votingMembers) {
            this.log.debug("Polling {} for next term {}", (Object)member, (Object)(this.raft.getTerm() + 1L));
            PollRequest request = PollRequest.builder().withTerm(this.raft.getTerm()).withCandidate(this.raft.getCluster().getLocalMember().memberId()).withLastLogIndex(lastEntry != null ? lastEntry.index() : 0L).withLastLogTerm(lastTerm).build();
            this.raft.getProtocol().poll(member.memberId(), request).whenCompleteAsync((response, error) -> this.handlePollResponse(complete, quorum, member, (PollResponse)response, (Throwable)error), (Executor)this.raft.getThreadContext());
        }
    }

    @Override
    public CompletableFuture<ConfigureResponse> onConfigure(ConfigureRequest request) {
        CompletableFuture<ConfigureResponse> future = super.onConfigure(request);
        if (this.isRequestFromCurrentLeader(request.term(), request.leader())) {
            this.resetHeartbeatTimeout();
        }
        return future;
    }

    @Override
    public CompletableFuture<AppendResponse> onAppend(AppendRequest request) {
        CompletableFuture<AppendResponse> future = super.onAppend(request);
        if (this.isRequestFromCurrentLeader(request.term(), request.leader())) {
            this.resetHeartbeatTimeout();
        }
        return future;
    }

    @Override
    protected VoteResponse handleVote(VoteRequest request) {
        VoteResponse response = super.handleVote(request);
        if (response.voted()) {
            this.resetHeartbeatTimeout();
        }
        return response;
    }

    private boolean isRequestFromCurrentLeader(long term, MemberId leader) {
        long currentTerm = this.raft.getTerm();
        DefaultRaftMember currentLeader = this.raft.getLeader();
        if (term < currentTerm || term == currentTerm && (currentLeader == null || !leader.equals(currentLeader.memberId()))) {
            this.log.debug("Expected heartbeat from {} in term {}, but received one from {} in term {}, ignoring it", new Object[]{currentLeader, currentTerm, leader, leader});
            return false;
        }
        return true;
    }

    private void resetHeartbeatTimeout() {
        this.raft.checkThread();
        if (!this.isRunning()) {
            return;
        }
        this.cancelHeartbeatTimer();
        this.updateHeartbeat(System.currentTimeMillis());
        Duration delay = this.raft.getElectionTimeout().plus(Duration.ofMillis(this.raft.getRandom().nextInt((int)this.raft.getElectionTimeout().toMillis())));
        this.heartbeatTimer = this.raft.getThreadContext().schedule(delay, () -> this.onHeartbeatTimeout(delay));
    }

    private void updateHeartbeat(long currentTimestamp) {
        if (this.lastHeartbeat > 0L) {
            this.raft.getRaftRoleMetrics().observeHeartbeatInterval(currentTimestamp - this.lastHeartbeat);
        }
        this.lastHeartbeat = currentTimestamp;
    }

    private void schedulePollRequests(Duration delay) {
        this.raft.checkThread();
        if (!this.isRunning()) {
            return;
        }
        if (this.raft.getFirstCommitIndex() == 0L || this.raft.getState() == RaftContext.State.READY) {
            long missTime = System.currentTimeMillis() - this.lastHeartbeat;
            this.log.info("No heartbeat from {} in the last {} (calculated from last {} ms), sending poll requests", new Object[]{this.raft.getLeader(), delay, missTime});
            this.raft.getRaftRoleMetrics().countHeartbeatMiss();
            this.raft.setLeader(null);
            this.sendPollRequests();
        }
    }

    private void onHeartbeatTimeout(Duration delay) {
        if (!this.isRunning()) {
            return;
        }
        Duration pollTimeout = this.raft.getElectionTimeout();
        this.heartbeatTimer = this.raft.getThreadContext().schedule(pollTimeout, () -> {
            this.log.debug("Failed to poll a majority of the cluster in {}", (Object)pollTimeout);
            this.resetHeartbeatTimeout();
        });
        this.schedulePollRequests(delay);
    }

    private void handlePollResponse(AtomicBoolean complete, Quorum quorum, DefaultRaftMember member, PollResponse response, Throwable error) {
        this.raft.checkThread();
        if (this.isRunning() && !complete.get()) {
            if (error != null) {
                this.log.warn("Poll request to {} failed: {}", (Object)member.memberId(), (Object)error.getMessage());
                quorum.fail();
            } else {
                if (response.term() > this.raft.getTerm()) {
                    this.raft.setTerm(response.term());
                }
                if (!response.accepted()) {
                    this.log.debug("Received rejected poll from {}", (Object)member);
                    quorum.fail();
                } else if (response.term() != this.raft.getTerm()) {
                    this.log.debug("Received accepted poll for a different term from {}", (Object)member);
                    quorum.fail();
                } else {
                    this.log.debug("Received accepted poll from {}", (Object)member);
                    quorum.succeed();
                }
            }
        }
    }
}

