/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.client.impl.worker;

import io.zeebe.client.api.response.ActivatedJob;
import io.zeebe.client.api.worker.BackoffSupplier;
import io.zeebe.client.api.worker.JobWorker;
import io.zeebe.client.impl.Loggers;
import io.zeebe.client.impl.worker.JobPoller;
import io.zeebe.client.impl.worker.JobRunnableFactory;
import io.zeebe.client.impl.worker.JobWorkerBuilderImpl;
import java.io.Closeable;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;

public final class JobWorkerImpl
implements JobWorker,
Closeable {
    private static final BackoffSupplier DEFAULT_BACKOFF_SUPPLIER = JobWorkerBuilderImpl.DEFAULT_BACKOFF_SUPPLIER;
    private static final Logger LOG = Loggers.JOB_WORKER_LOGGER;
    private static final String SUPPLY_RETRY_DELAY_FAILURE_MESSAGE = "Expected to supply retry delay, but an exception was thrown. Falling back to default backoff supplier";
    private final int maxJobsActive;
    private final int activationThreshold;
    private final AtomicInteger remainingJobs;
    private final ScheduledExecutorService executor;
    private final JobRunnableFactory jobHandlerFactory;
    private final long initialPollInterval;
    private final BackoffSupplier backoffSupplier;
    private final AtomicBoolean acquiringJobs = new AtomicBoolean(true);
    private final AtomicReference<JobPoller> claimableJobPoller;
    private final AtomicBoolean isPollScheduled = new AtomicBoolean(false);
    private volatile long pollInterval;

    public JobWorkerImpl(int maxJobsActive, ScheduledExecutorService executor, Duration pollInterval, JobRunnableFactory jobHandlerFactory, JobPoller jobPoller, BackoffSupplier backoffSupplier) {
        this.maxJobsActive = maxJobsActive;
        this.activationThreshold = Math.round((float)maxJobsActive * 0.3f);
        this.remainingJobs = new AtomicInteger(0);
        this.executor = executor;
        this.jobHandlerFactory = jobHandlerFactory;
        this.initialPollInterval = pollInterval.toMillis();
        this.backoffSupplier = backoffSupplier;
        this.claimableJobPoller = new AtomicReference<JobPoller>(jobPoller);
        this.pollInterval = this.initialPollInterval;
        this.schedulePoll();
    }

    @Override
    public boolean isOpen() {
        return this.acquiringJobs.get();
    }

    @Override
    public boolean isClosed() {
        return !this.isOpen() && this.claimableJobPoller.get() != null && this.remainingJobs.get() <= 0;
    }

    @Override
    public void close() {
        this.acquiringJobs.set(false);
    }

    private void schedulePoll() {
        if (this.isPollScheduled.compareAndSet(false, true)) {
            this.executor.schedule(this::onScheduledPoll, this.pollInterval, TimeUnit.MILLISECONDS);
        }
    }

    private void onScheduledPoll() {
        this.isPollScheduled.set(false);
        int actualRemainingJobs = this.remainingJobs.get();
        if (this.shouldPoll(actualRemainingJobs)) {
            this.tryPoll();
        }
    }

    private boolean shouldPoll(int remainingJobs) {
        return this.acquiringJobs.get() && remainingJobs <= this.activationThreshold;
    }

    private void tryPoll() {
        this.tryClaimJobPoller().ifPresent(poller -> {
            try {
                this.poll((JobPoller)poller);
            }
            catch (Exception error) {
                LOG.warn("Unexpected failure to activate jobs", (Throwable)error);
                this.backoff((JobPoller)poller, error);
            }
        });
    }

    private Optional<JobPoller> tryClaimJobPoller() {
        return Optional.ofNullable(this.claimableJobPoller.getAndSet(null));
    }

    private void releaseJobPoller(JobPoller jobPoller) {
        this.claimableJobPoller.set(jobPoller);
    }

    private void poll(JobPoller jobPoller) {
        int actualRemainingJobs = this.remainingJobs.get();
        if (!this.shouldPoll(actualRemainingJobs)) {
            LOG.trace("Expected to activate for jobs, but still enough remain. Reschedule poll.");
            this.releaseJobPoller(jobPoller);
            this.schedulePoll();
            return;
        }
        int maxJobsToActivate = this.maxJobsActive - actualRemainingJobs;
        jobPoller.poll(maxJobsToActivate, this::handleJob, activatedJobs -> this.onPollSuccess(jobPoller, activatedJobs), error -> this.onPollError(jobPoller, (Throwable)error), this::isOpen);
    }

    private void onPollSuccess(JobPoller jobPoller, int activatedJobs) {
        this.releaseJobPoller(jobPoller);
        int actualRemainingJobs = this.remainingJobs.addAndGet(activatedJobs);
        this.pollInterval = this.initialPollInterval;
        if (actualRemainingJobs <= 0) {
            this.schedulePoll();
        }
    }

    private void onPollError(JobPoller jobPoller, Throwable error) {
        this.backoff(jobPoller, error);
    }

    private void backoff(JobPoller jobPoller, Throwable error) {
        long prevInterval = this.pollInterval;
        try {
            this.pollInterval = this.backoffSupplier.supplyRetryDelay(prevInterval);
        }
        catch (Exception e) {
            LOG.warn(SUPPLY_RETRY_DELAY_FAILURE_MESSAGE, (Throwable)e);
            this.pollInterval = DEFAULT_BACKOFF_SUPPLIER.supplyRetryDelay(prevInterval);
        }
        LOG.debug("Failed to activate jobs due to {}, delay retry for {} ms", (Object)error.getMessage(), (Object)this.pollInterval);
        this.releaseJobPoller(jobPoller);
        this.schedulePoll();
    }

    private void handleJob(ActivatedJob job) {
        this.executor.execute(this.jobHandlerFactory.create(job, this::handleJobFinished));
    }

    private void handleJobFinished() {
        int actualRemainingJobs = this.remainingJobs.decrementAndGet();
        if (!this.isPollScheduled.get() && this.shouldPoll(actualRemainingJobs)) {
            this.tryPoll();
        }
    }
}

