/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.snapshots.broker.impl;

import io.zeebe.snapshots.broker.SnapshotId;
import io.zeebe.snapshots.broker.impl.FileBasedSnapshotMetadata;
import io.zeebe.snapshots.broker.impl.FileBasedSnapshotStore;
import io.zeebe.snapshots.broker.impl.SnapshotChecksum;
import io.zeebe.snapshots.broker.impl.SnapshotChunkUtil;
import io.zeebe.snapshots.raft.PersistedSnapshot;
import io.zeebe.snapshots.raft.ReceivedSnapshot;
import io.zeebe.snapshots.raft.SnapshotChunk;
import io.zeebe.util.FileUtil;
import io.zeebe.util.sched.ActorControl;
import io.zeebe.util.sched.future.ActorFuture;
import io.zeebe.util.sched.future.CompletableActorFuture;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileBasedReceivedSnapshot
implements ReceivedSnapshot {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileBasedReceivedSnapshot.class);
    private static final boolean FAILED = false;
    private static final boolean SUCCESS = true;
    private final Path directory;
    private final ActorControl actor;
    private final FileBasedSnapshotStore snapshotStore;
    private final FileBasedSnapshotMetadata metadata;
    private long expectedSnapshotChecksum;
    private int expectedTotalCount;

    FileBasedReceivedSnapshot(FileBasedSnapshotMetadata metadata, Path directory, FileBasedSnapshotStore snapshotStore, ActorControl actor) {
        this.metadata = metadata;
        this.snapshotStore = snapshotStore;
        this.directory = directory;
        this.actor = actor;
        this.expectedSnapshotChecksum = Long.MIN_VALUE;
        this.expectedTotalCount = Integer.MIN_VALUE;
    }

    @Override
    public long index() {
        return this.metadata.getIndex();
    }

    @Override
    public ActorFuture<Boolean> apply(SnapshotChunk snapshotChunk) throws IOException {
        return this.actor.call(() -> this.applyInternal(snapshotChunk));
    }

    private boolean containsChunk(String chunkId) {
        return Files.exists(this.directory.resolve(chunkId), new LinkOption[0]);
    }

    private boolean applyInternal(SnapshotChunk snapshotChunk) throws IOException {
        if (this.containsChunk(snapshotChunk.getChunkName())) {
            return true;
        }
        long currentSnapshotChecksum = snapshotChunk.getSnapshotChecksum();
        if (this.isSnapshotIdInvalid(snapshotChunk.getSnapshotId())) {
            return false;
        }
        if (this.isSnapshotChecksumInvalid(currentSnapshotChecksum)) {
            return false;
        }
        int currentTotalCount = snapshotChunk.getTotalCount();
        if (this.isTotalCountInvalid(currentTotalCount)) {
            return false;
        }
        String snapshotId = snapshotChunk.getSnapshotId();
        String chunkName = snapshotChunk.getChunkName();
        if (this.snapshotStore.hasSnapshotId(snapshotId)) {
            LOGGER.debug("Ignore snapshot snapshotChunk {}, because snapshot {} already exists.", (Object)chunkName, (Object)snapshotId);
            return true;
        }
        if (this.isChunkChecksumInvalid(snapshotChunk, snapshotId, chunkName)) {
            return false;
        }
        Path tmpSnapshotDirectory = this.directory;
        FileUtil.ensureDirectoryExists((Path)tmpSnapshotDirectory);
        Path snapshotFile = tmpSnapshotDirectory.resolve(chunkName);
        if (Files.exists(snapshotFile, new LinkOption[0])) {
            LOGGER.debug("Received a snapshot snapshotChunk which already exist '{}'.", (Object)snapshotFile);
            return false;
        }
        LOGGER.debug("Consume snapshot snapshotChunk {} of snapshot {}", (Object)chunkName, (Object)snapshotId);
        return this.writeReceivedSnapshotChunk(snapshotChunk, snapshotFile);
    }

    private boolean isChunkChecksumInvalid(SnapshotChunk snapshotChunk, String snapshotId, String chunkName) {
        long actualChecksum;
        long expectedChecksum = snapshotChunk.getChecksum();
        if (expectedChecksum != (actualChecksum = SnapshotChunkUtil.createChecksum(snapshotChunk.getContent()))) {
            LOGGER.warn("Expected to have checksum {} for snapshot chunk {} ({}), but calculated {}", new Object[]{expectedChecksum, chunkName, snapshotId, actualChecksum});
            return true;
        }
        return false;
    }

    private boolean isSnapshotChecksumInvalid(long currentSnapshotChecksum) {
        if (this.expectedSnapshotChecksum == Long.MIN_VALUE) {
            this.expectedSnapshotChecksum = currentSnapshotChecksum;
        }
        if (this.expectedSnapshotChecksum != currentSnapshotChecksum) {
            LOGGER.warn("Expected snapshot chunk with equal snapshot checksum {}, but got chunk with snapshot checksum {}.", (Object)this.expectedSnapshotChecksum, (Object)currentSnapshotChecksum);
            return true;
        }
        return false;
    }

    private boolean isTotalCountInvalid(int currentTotalCount) {
        if (this.expectedTotalCount == Integer.MIN_VALUE) {
            this.expectedTotalCount = currentTotalCount;
        }
        if (this.expectedTotalCount != currentTotalCount) {
            LOGGER.warn("Expected snapshot chunk with equal snapshot total count {}, but got chunk with total count {}.", (Object)this.expectedTotalCount, (Object)currentTotalCount);
            return true;
        }
        return false;
    }

    private boolean isSnapshotIdInvalid(String snapshotId) {
        Optional<FileBasedSnapshotMetadata> receivedSnapshotId = FileBasedSnapshotMetadata.ofFileName(snapshotId);
        if (receivedSnapshotId.isEmpty()) {
            return true;
        }
        return this.metadata.compareTo(receivedSnapshotId.get()) != 0;
    }

    private boolean writeReceivedSnapshotChunk(SnapshotChunk snapshotChunk, Path snapshotFile) throws IOException {
        Files.write(snapshotFile, snapshotChunk.getContent(), StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
        LOGGER.trace("Wrote replicated snapshot chunk to file {}", (Object)snapshotFile);
        return true;
    }

    @Override
    public ActorFuture<Void> abort() {
        CompletableActorFuture abortFuture = new CompletableActorFuture();
        this.actor.run(() -> {
            this.abortInternal();
            abortFuture.complete(null);
        });
        return abortFuture;
    }

    @Override
    public ActorFuture<PersistedSnapshot> persist() {
        CompletableActorFuture future = new CompletableActorFuture();
        this.actor.call(() -> this.persistInternal((CompletableActorFuture<PersistedSnapshot>)future));
        return future;
    }

    @Override
    public SnapshotId snapshotId() {
        return this.metadata;
    }

    private void abortInternal() {
        try {
            LOGGER.debug("DELETE dir {}", (Object)this.directory);
            FileUtil.deleteFolder((Path)this.directory);
        }
        catch (NoSuchFileException nsfe) {
            LOGGER.debug("Tried to delete pending dir {}, but doesn't exist. Either was already removed or no chunk was applied until now.", (Object)this.directory, (Object)nsfe);
        }
        catch (IOException e) {
            LOGGER.warn("Failed to delete pending snapshot {}", (Object)this, (Object)e);
        }
        finally {
            this.snapshotStore.removePendingSnapshot(this);
        }
    }

    private void persistInternal(CompletableActorFuture<PersistedSnapshot> future) {
        if (this.snapshotStore.hasSnapshotId(this.metadata.getSnapshotIdAsString())) {
            this.abortInternal();
            future.complete((Object)this.snapshotStore.getLatestSnapshot().orElseThrow());
            return;
        }
        Object[] files = this.directory.toFile().listFiles();
        try {
            Objects.requireNonNull(files, "No chunks have been applied yet");
        }
        catch (Exception e) {
            future.completeExceptionally((Throwable)e);
            return;
        }
        if (files.length != this.expectedTotalCount) {
            future.completeExceptionally((Throwable)new IllegalStateException(String.format("Expected '%d' chunk files for this snapshot, but found '%d'. Files are: %s.", this.expectedSnapshotChecksum, files.length, Arrays.toString(files))));
            return;
        }
        if (!this.verifyChecksums(future)) {
            return;
        }
        try {
            PersistedSnapshot value = this.snapshotStore.newSnapshot(this.metadata, this.directory);
            future.complete((Object)value);
        }
        catch (Exception e) {
            future.completeExceptionally((Throwable)e);
        }
    }

    private boolean verifyChecksums(CompletableActorFuture<PersistedSnapshot> future) {
        try {
            if (SnapshotChecksum.verify(this.directory)) {
                return true;
            }
            future.completeExceptionally((Throwable)new IllegalStateException("Snapshot is corrupted. Checksum does not match"));
            return false;
        }
        catch (IOException e) {
            future.completeExceptionally((Throwable)new UncheckedIOException("Unexpected exception on calculating snapshot checksum.", e));
            return false;
        }
    }

    public String toString() {
        return "FileBasedReceivedSnapshot{directory=" + this.directory + ", snapshotStore=" + this.snapshotStore + ", metadata=" + this.metadata + "}";
    }
}

