/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.snapshots.broker.impl;

import io.zeebe.snapshots.broker.ConstructableSnapshotStore;
import io.zeebe.snapshots.broker.SnapshotId;
import io.zeebe.snapshots.broker.impl.FileBasedReceivedSnapshot;
import io.zeebe.snapshots.broker.impl.FileBasedSnapshot;
import io.zeebe.snapshots.broker.impl.FileBasedSnapshotMetadata;
import io.zeebe.snapshots.broker.impl.FileBasedTransientSnapshot;
import io.zeebe.snapshots.broker.impl.SnapshotChecksum;
import io.zeebe.snapshots.broker.impl.SnapshotMetrics;
import io.zeebe.snapshots.raft.PersistableSnapshot;
import io.zeebe.snapshots.raft.PersistedSnapshot;
import io.zeebe.snapshots.raft.PersistedSnapshotListener;
import io.zeebe.snapshots.raft.ReceivableSnapshotStore;
import io.zeebe.snapshots.raft.ReceivedSnapshot;
import io.zeebe.snapshots.raft.TransientSnapshot;
import io.zeebe.util.FileUtil;
import io.zeebe.util.sched.Actor;
import io.zeebe.util.sched.future.ActorFuture;
import io.zeebe.util.sched.future.CompletableActorFuture;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FileBasedSnapshotStore
extends Actor
implements ConstructableSnapshotStore,
ReceivableSnapshotStore {
    private static final String RECEIVING_DIR_FORMAT = "%s-%d";
    private static final Logger LOGGER = LoggerFactory.getLogger(FileBasedSnapshotStore.class);
    private final Path snapshotsDirectory;
    private final Path pendingDirectory;
    private final Set<PersistedSnapshotListener> listeners;
    private final SnapshotMetrics snapshotMetrics;
    private final AtomicReference<FileBasedSnapshot> currentPersistedSnapshotRef;
    private final AtomicLong receivingSnapshotStartCount;
    private final Set<PersistableSnapshot> pendingSnapshots = new HashSet<PersistableSnapshot>();
    private final String actorName;

    public FileBasedSnapshotStore(int nodeId, int partitionId, SnapshotMetrics snapshotMetrics, Path snapshotsDirectory, Path pendingDirectory) {
        this.snapshotsDirectory = snapshotsDirectory;
        this.pendingDirectory = pendingDirectory;
        this.snapshotMetrics = snapshotMetrics;
        this.receivingSnapshotStartCount = new AtomicLong();
        this.listeners = new CopyOnWriteArraySet<PersistedSnapshotListener>();
        this.actorName = FileBasedSnapshotStore.buildActorName((int)nodeId, (String)"SnapshotStore", (int)partitionId);
        this.currentPersistedSnapshotRef = new AtomicReference<FileBasedSnapshot>(this.loadLatestSnapshot(snapshotsDirectory));
        this.purgePendingSnapshotsDirectory();
    }

    public String getName() {
        return this.actorName;
    }

    public void close() {
        this.listeners.clear();
    }

    private FileBasedSnapshot loadLatestSnapshot(Path snapshotDirectory) {
        FileBasedSnapshot latestPersistedSnapshot = null;
        ArrayList<FileBasedSnapshot> snapshots = new ArrayList<FileBasedSnapshot>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(snapshotDirectory);){
            for (Path path : stream) {
                FileBasedSnapshot snapshot = this.collectSnapshot(path);
                if (snapshot == null) continue;
                snapshots.add(snapshot);
                if (latestPersistedSnapshot != null && snapshot.getMetadata().compareTo(latestPersistedSnapshot.getMetadata()) < 0) continue;
                latestPersistedSnapshot = snapshot;
            }
            if (latestPersistedSnapshot != null) {
                snapshots.remove(latestPersistedSnapshot);
                if (!snapshots.isEmpty()) {
                    LOGGER.debug("Purging snapshots older than {}", latestPersistedSnapshot);
                    snapshots.forEach(oldSnapshot -> {
                        LOGGER.debug("Deleting snapshot {}", oldSnapshot);
                        oldSnapshot.delete();
                    });
                }
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return latestPersistedSnapshot;
    }

    private FileBasedSnapshot collectSnapshot(Path path) throws IOException {
        Optional<FileBasedSnapshotMetadata> optionalMeta = FileBasedSnapshotMetadata.ofPath(path);
        if (optionalMeta.isPresent()) {
            FileBasedSnapshotMetadata metadata = optionalMeta.get();
            try {
                if (SnapshotChecksum.verify(path)) {
                    return new FileBasedSnapshot(path, metadata);
                }
                LOGGER.warn("Cannot load snapshot in {}. The checksum stored does not match the checksum calculated.", (Object)path);
            }
            catch (Exception e) {
                LOGGER.warn("Could not load snapshot in {}", (Object)path, (Object)e);
            }
        } else {
            LOGGER.warn("Expected snapshot file format to be %d-%d-%d-%d, but was {}", (Object)path);
        }
        return null;
    }

    private void purgePendingSnapshotsDirectory() {
        try (Stream<Path> files = Files.list(this.pendingDirectory);){
            files.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(this::purgePendingSnapshot);
        }
        catch (IOException e) {
            LOGGER.error("Failed to purge pending snapshots, which may result in unnecessary disk usage and should be monitored", (Throwable)e);
        }
    }

    @Override
    public boolean hasSnapshotId(String id) {
        Optional<PersistedSnapshot> optLatestSnapshot = this.getLatestSnapshot();
        if (optLatestSnapshot.isPresent()) {
            PersistedSnapshot snapshot = optLatestSnapshot.get();
            return snapshot.getPath().getFileName().toString().equals(id);
        }
        return false;
    }

    @Override
    public Optional<PersistedSnapshot> getLatestSnapshot() {
        return Optional.ofNullable((PersistedSnapshot)this.currentPersistedSnapshotRef.get());
    }

    @Override
    public ActorFuture<Void> purgePendingSnapshots() {
        CompletableActorFuture abortFuture = new CompletableActorFuture();
        this.actor.run(() -> {
            List abortedAll = this.pendingSnapshots.stream().map(PersistableSnapshot::abort).collect(Collectors.toList());
            this.actor.runOnCompletion(abortedAll, error -> {
                if (error == null) {
                    abortFuture.complete(null);
                } else {
                    abortFuture.completeExceptionally(error);
                }
            });
        });
        return abortFuture;
    }

    @Override
    public ActorFuture<Boolean> addSnapshotListener(PersistedSnapshotListener listener) {
        return this.actor.call(() -> this.listeners.add(listener));
    }

    @Override
    public ActorFuture<Boolean> removeSnapshotListener(PersistedSnapshotListener listener) {
        return this.actor.call(() -> this.listeners.remove(listener));
    }

    @Override
    public long getCurrentSnapshotIndex() {
        return this.getLatestSnapshot().map(PersistedSnapshot::getIndex).orElse(0L);
    }

    @Override
    public ActorFuture<Void> delete() {
        return this.actor.call(() -> {
            this.currentPersistedSnapshotRef.set(null);
            try {
                LOGGER.debug("DELETE FOLDER {}", (Object)this.snapshotsDirectory);
                FileUtil.deleteFolder((Path)this.snapshotsDirectory);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            try {
                LOGGER.debug("DELETE FOLDER {}", (Object)this.pendingDirectory);
                FileUtil.deleteFolder((Path)this.pendingDirectory);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        });
    }

    @Override
    public ReceivedSnapshot newReceivedSnapshot(String snapshotId) {
        Optional<FileBasedSnapshotMetadata> optMetadata = FileBasedSnapshotMetadata.ofFileName(snapshotId);
        FileBasedSnapshotMetadata metadata = optMetadata.orElseThrow(() -> new IllegalArgumentException("Expected snapshot id in a format like 'index-term-processedPosition-exportedPosition', got '" + snapshotId + "'."));
        long nextStartCount = this.receivingSnapshotStartCount.incrementAndGet();
        String pendingDirectoryName = String.format(RECEIVING_DIR_FORMAT, metadata.getSnapshotIdAsString(), nextStartCount);
        Path pendingSnapshotDir = this.pendingDirectory.resolve(pendingDirectoryName);
        FileBasedReceivedSnapshot newPendingSnapshot = new FileBasedReceivedSnapshot(metadata, pendingSnapshotDir, this, this.actor);
        this.addPendingSnapshot(newPendingSnapshot);
        return newPendingSnapshot;
    }

    @Override
    public Optional<TransientSnapshot> newTransientSnapshot(long index, long term, long processedPosition, long exportedPosition) {
        FileBasedSnapshotMetadata newSnapshotId = new FileBasedSnapshotMetadata(index, term, processedPosition, exportedPosition);
        FileBasedSnapshot currentSnapshot = this.currentPersistedSnapshotRef.get();
        if (currentSnapshot != null && currentSnapshot.getMetadata().compareTo(newSnapshotId) == 0) {
            LOGGER.debug("Previous snapshot was taken for the same processed position {} and exported position {}, will not take snapshot.", (Object)processedPosition, (Object)exportedPosition);
            return Optional.empty();
        }
        Path directory = this.buildPendingSnapshotDirectory(newSnapshotId);
        FileBasedTransientSnapshot newPendingSnapshot = new FileBasedTransientSnapshot(newSnapshotId, directory, this, this.actor);
        this.addPendingSnapshot(newPendingSnapshot);
        return Optional.of(newPendingSnapshot);
    }

    private void addPendingSnapshot(PersistableSnapshot pendingSnapshot) {
        this.actor.call(() -> this.pendingSnapshots.add(pendingSnapshot));
    }

    void removePendingSnapshot(PersistableSnapshot pendingSnapshot) {
        this.pendingSnapshots.remove(pendingSnapshot);
    }

    private void observeSnapshotSize(PersistedSnapshot persistedSnapshot) {
        try (DirectoryStream<Path> contents = Files.newDirectoryStream(persistedSnapshot.getPath());){
            long totalSize = 0L;
            long totalCount = 0L;
            for (Path path : contents) {
                if (!Files.isRegularFile(path, new LinkOption[0])) continue;
                long size = Files.size(path);
                this.snapshotMetrics.observeSnapshotFileSize(size);
                totalSize += size;
                ++totalCount;
            }
            this.snapshotMetrics.observeSnapshotSize(totalSize);
            this.snapshotMetrics.observeSnapshotChunkCount(totalCount);
        }
        catch (IOException e) {
            LOGGER.warn("Failed to observe size for snapshot {}", (Object)persistedSnapshot, (Object)e);
        }
    }

    private void purgePendingSnapshots(SnapshotId cutoffIndex) {
        LOGGER.debug("Search for orphaned snapshots below oldest valid snapshot with index {} in {}", (Object)cutoffIndex, (Object)this.pendingDirectory);
        this.pendingSnapshots.stream().filter(pendingSnapshot -> pendingSnapshot.snapshotId().compareTo(cutoffIndex) < 0).forEach(PersistableSnapshot::abort);
        try (DirectoryStream<Path> pendingSnapshotsDirectories = Files.newDirectoryStream(this.pendingDirectory);){
            for (Path pendingSnapshot2 : pendingSnapshotsDirectories) {
                this.purgePendingSnapshot(cutoffIndex, pendingSnapshot2);
            }
        }
        catch (IOException e) {
            LOGGER.warn("Failed to delete orphaned snapshots, could not list pending directory {}", (Object)this.pendingDirectory, (Object)e);
        }
    }

    private void purgePendingSnapshot(SnapshotId cutoffIndex, Path pendingSnapshot) {
        Optional<FileBasedSnapshotMetadata> optionalMetadata = FileBasedSnapshotMetadata.ofPath(pendingSnapshot);
        if (optionalMetadata.isPresent() && optionalMetadata.get().compareTo(cutoffIndex) < 0) {
            try {
                FileUtil.deleteFolder((Path)pendingSnapshot);
                LOGGER.debug("Deleted orphaned snapshot {}", (Object)pendingSnapshot);
            }
            catch (IOException e) {
                LOGGER.warn("Failed to delete orphaned snapshot {}, risk using unnecessary disk space", (Object)pendingSnapshot, (Object)e);
            }
        }
    }

    public Path getPath() {
        return this.snapshotsDirectory;
    }

    private boolean isCurrentSnapshotNewer(FileBasedSnapshotMetadata metadata) {
        FileBasedSnapshot persistedSnapshot = this.currentPersistedSnapshotRef.get();
        return persistedSnapshot != null && persistedSnapshot.getMetadata().compareTo(metadata) >= 0;
    }

    PersistedSnapshot newSnapshot(FileBasedSnapshotMetadata metadata, Path directory) {
        boolean failed;
        FileBasedSnapshot currentPersistedSnapshot = this.currentPersistedSnapshotRef.get();
        if (this.isCurrentSnapshotNewer(metadata)) {
            LOGGER.debug("Snapshot is older then {} already exists", (Object)currentPersistedSnapshot);
            this.purgePendingSnapshots(metadata);
            return currentPersistedSnapshot;
        }
        Path destination = this.buildSnapshotDirectory(metadata);
        this.moveToSnapshotDirectory(directory, destination);
        FileBasedSnapshot newPersistedSnapshot = new FileBasedSnapshot(destination, metadata);
        boolean bl = failed = !this.currentPersistedSnapshotRef.compareAndSet(currentPersistedSnapshot, newPersistedSnapshot);
        if (failed) {
            String errorMessage = "Expected that last snapshot is '%s', which should be replace with '%s', but last snapshot was '%s'.";
            throw new ConcurrentModificationException(String.format("Expected that last snapshot is '%s', which should be replace with '%s', but last snapshot was '%s'.", currentPersistedSnapshot, newPersistedSnapshot, this.currentPersistedSnapshotRef.get()));
        }
        this.snapshotMetrics.incrementSnapshotCount();
        this.observeSnapshotSize(newPersistedSnapshot);
        LOGGER.debug("Purging snapshots older than {}", (Object)newPersistedSnapshot);
        if (currentPersistedSnapshot != null) {
            LOGGER.debug("Deleting snapshot {}", (Object)currentPersistedSnapshot);
            currentPersistedSnapshot.delete();
        }
        this.purgePendingSnapshots(newPersistedSnapshot.getMetadata());
        this.listeners.forEach(listener -> listener.onNewSnapshot(newPersistedSnapshot));
        LOGGER.debug("Created new snapshot {}", (Object)newPersistedSnapshot);
        return newPersistedSnapshot;
    }

    private void moveToSnapshotDirectory(Path directory, Path destination) {
        try {
            this.tryAtomicDirectoryMove(directory, destination);
        }
        catch (FileAlreadyExistsException e) {
            LOGGER.debug("Expected to move snapshot from {} to {}, but it already exists", new Object[]{directory, destination, e});
        }
        catch (IOException e) {
            try {
                if (Files.exists(destination, new LinkOption[0])) {
                    FileUtil.deleteFolder((Path)destination);
                }
            }
            catch (IOException ioException) {
                LOGGER.error("Failed to delete snapshot directory {} after atomic move failed.", (Object)destination, (Object)ioException);
            }
            throw new UncheckedIOException(e);
        }
    }

    private void purgePendingSnapshot(Path pendingSnapshot) {
        try {
            FileUtil.deleteFolder((Path)pendingSnapshot);
            LOGGER.debug("Deleted not completed (orphaned) snapshot {}", (Object)pendingSnapshot);
        }
        catch (IOException e) {
            LOGGER.error("Failed to delete not completed (orphaned) snapshot {}", (Object)pendingSnapshot, (Object)e);
        }
    }

    private void tryAtomicDirectoryMove(Path directory, Path destination) throws IOException {
        try {
            Files.move(directory, destination, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (AtomicMoveNotSupportedException e) {
            LOGGER.warn("Atomic move not supported. Moving the snapshot files non-atomically.");
            Files.move(directory, destination, new CopyOption[0]);
        }
    }

    private Path buildPendingSnapshotDirectory(SnapshotId id) {
        return this.pendingDirectory.resolve(id.getSnapshotIdAsString());
    }

    private Path buildSnapshotDirectory(FileBasedSnapshotMetadata metadata) {
        return this.snapshotsDirectory.resolve(metadata.getSnapshotIdAsString());
    }

    SnapshotMetrics getSnapshotMetrics() {
        return this.snapshotMetrics;
    }
}

