/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.util.health;

import io.zeebe.util.health.FailureListener;
import io.zeebe.util.health.HealthMonitor;
import io.zeebe.util.health.HealthMonitorable;
import io.zeebe.util.health.HealthStatus;
import io.zeebe.util.sched.ActorControl;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import org.slf4j.Logger;

public class CriticalComponentsHealthMonitor
implements HealthMonitor {
    private static final Duration HEALTH_MONITORING_PERIOD = Duration.ofSeconds(60L);
    private final Map<String, HealthMonitorable> monitoredComponents = new HashMap<String, HealthMonitorable>();
    private final Map<String, HealthStatus> componentHealth = new HashMap<String, HealthStatus>();
    private volatile HealthStatus healthStatus = HealthStatus.UNHEALTHY;
    private final ActorControl actor;
    private final Logger log;
    private FailureListener failureListener;

    public CriticalComponentsHealthMonitor(ActorControl actor, Logger log) {
        this.actor = actor;
        this.log = log;
    }

    @Override
    public void startMonitoring() {
        this.actor.runAtFixedRate(HEALTH_MONITORING_PERIOD, this::updateHealth);
    }

    @Override
    public void monitorComponent(String componentName) {
        this.actor.run(() -> this.componentHealth.put(componentName, HealthStatus.UNHEALTHY));
    }

    @Override
    public void removeComponent(String componentName) {
        this.actor.run(() -> {
            this.monitoredComponents.remove(componentName);
            this.componentHealth.remove(componentName);
        });
    }

    @Override
    public void registerComponent(String componentName, HealthMonitorable component) {
        this.actor.run(() -> {
            this.monitoredComponents.put(componentName, component);
            component.addFailureListener(new ComponentFailureListener(componentName));
            this.componentHealth.put(componentName, component.getHealthStatus());
            this.calculateHealth();
        });
    }

    @Override
    public HealthStatus getHealthStatus() {
        return this.healthStatus;
    }

    @Override
    public void addFailureListener(FailureListener failureListener) {
        this.actor.run(() -> {
            this.failureListener = failureListener;
        });
    }

    private void updateHealth() {
        this.componentHealth.keySet().forEach(component -> this.componentHealth.put((String)component, this.getHealth((String)component)));
        this.calculateHealth();
    }

    private void calculateHealth() {
        HealthStatus status = this.componentHealth.containsValue((Object)HealthStatus.UNHEALTHY) ? HealthStatus.UNHEALTHY : HealthStatus.HEALTHY;
        HealthStatus previousStatus = this.healthStatus;
        this.healthStatus = status;
        if (previousStatus != status) {
            switch (status) {
                case HEALTHY: {
                    if (this.failureListener != null) {
                        this.failureListener.onRecovered();
                    }
                    this.log.debug("The components are healthy. The current health status of components: {}", this.componentHealth);
                    break;
                }
                case UNHEALTHY: {
                    if (this.failureListener != null) {
                        this.failureListener.onFailure();
                    }
                    this.log.debug("Detected unhealthy components. The current health status of components: {}", this.componentHealth);
                    break;
                }
                default: {
                    this.log.warn("Unknown health status {}", (Object)status);
                }
            }
        }
    }

    private HealthStatus getHealth(String componentName) {
        HealthMonitorable component = this.monitoredComponents.get(componentName);
        if (component != null) {
            return component.getHealthStatus();
        }
        return HealthStatus.UNHEALTHY;
    }

    class ComponentFailureListener
    implements FailureListener {
        private final String componentName;

        ComponentFailureListener(String componentName) {
            this.componentName = componentName;
        }

        @Override
        public void onFailure() {
            CriticalComponentsHealthMonitor.this.actor.run(this::onComponentFailure);
        }

        @Override
        public void onRecovered() {
            CriticalComponentsHealthMonitor.this.actor.run(this::onComponentRecovered);
        }

        private void onComponentFailure() {
            CriticalComponentsHealthMonitor.this.log.error("{} failed, marking it as unhealthy", (Object)this.componentName);
            CriticalComponentsHealthMonitor.this.componentHealth.computeIfPresent(this.componentName, (k, v) -> HealthStatus.UNHEALTHY);
            CriticalComponentsHealthMonitor.this.calculateHealth();
        }

        private void onComponentRecovered() {
            CriticalComponentsHealthMonitor.this.log.info("{} recovered, marking it as healthy", (Object)this.componentName);
            CriticalComponentsHealthMonitor.this.componentHealth.computeIfPresent(this.componentName, (k, v) -> HealthStatus.HEALTHY);
            CriticalComponentsHealthMonitor.this.calculateHealth();
        }
    }
}

