/*
 * Decompiled with CFR 0.152.
 */
package io.zeebe.journal.file;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import io.zeebe.journal.Journal;
import io.zeebe.journal.JournalReader;
import io.zeebe.journal.JournalRecord;
import io.zeebe.journal.StorageException;
import io.zeebe.journal.file.JournalIndex;
import io.zeebe.journal.file.JournalMetrics;
import io.zeebe.journal.file.JournalSegment;
import io.zeebe.journal.file.JournalSegmentDescriptor;
import io.zeebe.journal.file.JournalSegmentFile;
import io.zeebe.journal.file.SegmentedJournalBuilder;
import io.zeebe.journal.file.SegmentedJournalReader;
import io.zeebe.journal.file.SegmentedJournalWriter;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.NavigableMap;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentSkipListMap;
import org.agrona.DirectBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentedJournal
implements Journal {
    public static final long ASQN_IGNORE = -1L;
    private static final int SEGMENT_BUFFER_FACTOR = 3;
    private static final int FIRST_SEGMENT_ID = 1;
    private static final int INITIAL_INDEX = 1;
    private final JournalMetrics journalMetrics;
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final String name;
    private final File directory;
    private final int maxSegmentSize;
    private final int maxEntrySize;
    private final NavigableMap<Long, JournalSegment> segments = new ConcurrentSkipListMap<Long, JournalSegment>();
    private final Collection<SegmentedJournalReader> readers = Sets.newConcurrentHashSet();
    private volatile JournalSegment currentSegment;
    private volatile boolean open = true;
    private final long minFreeDiskSpace;
    private final JournalIndex journalIndex;
    private final SegmentedJournalWriter writer;

    public SegmentedJournal(String name, File directory, int maxSegmentSize, int maxEntrySize, long minFreeSpace, JournalIndex journalIndex) {
        this.name = (String)Preconditions.checkNotNull((Object)name, (Object)"name cannot be null");
        this.directory = (File)Preconditions.checkNotNull((Object)directory, (Object)"directory cannot be null");
        this.maxSegmentSize = maxSegmentSize;
        this.maxEntrySize = maxEntrySize;
        this.journalMetrics = new JournalMetrics(name);
        this.minFreeDiskSpace = minFreeSpace;
        this.journalIndex = journalIndex;
        this.open();
        this.writer = new SegmentedJournalWriter(this);
    }

    public static SegmentedJournalBuilder builder() {
        return new SegmentedJournalBuilder();
    }

    @Override
    public JournalRecord append(long asqn, DirectBuffer data) {
        return this.writer.append(asqn, data);
    }

    @Override
    public JournalRecord append(DirectBuffer data) {
        return this.writer.append(-1L, data);
    }

    @Override
    public void append(JournalRecord record) {
        this.writer.append(record);
    }

    @Override
    public void deleteAfter(long indexExclusive) {
        this.writer.deleteAfter(indexExclusive);
    }

    @Override
    public void deleteUntil(long index) {
        Map.Entry<Long, JournalSegment> segmentEntry = this.segments.floorEntry(index);
        if (segmentEntry != null) {
            SortedMap<Long, JournalSegment> compactSegments = this.segments.headMap(segmentEntry.getValue().index());
            if (compactSegments.isEmpty()) {
                this.log.debug("No segments can be deleted with index < {} (first log index: {})", (Object)index, (Object)this.getFirstIndex());
                return;
            }
            this.log.debug("{} - Deleting log up from {} up to {} (removing {} segments)", new Object[]{this.name, this.getFirstIndex(), ((JournalSegment)compactSegments.get(compactSegments.lastKey())).index(), compactSegments.size()});
            for (JournalSegment segment : compactSegments.values()) {
                this.log.trace("{} - Deleting segment: {}", (Object)this.name, (Object)segment);
                segment.close();
                segment.delete();
                this.journalMetrics.decSegmentCount();
            }
            compactSegments.clear();
            this.journalIndex.deleteUntil(index);
            this.resetHead(this.getFirstSegment().index());
        }
    }

    @Override
    public void reset(long nextIndex) {
        this.journalIndex.clear();
        this.writer.reset(nextIndex);
    }

    @Override
    public long getLastIndex() {
        return this.writer.getLastIndex();
    }

    @Override
    public long getFirstIndex() {
        if (!this.segments.isEmpty()) {
            return this.segments.firstEntry().getValue().index();
        }
        return 0L;
    }

    @Override
    public boolean isEmpty() {
        return this.writer.getNextIndex() - this.getFirstSegment().index() == 0L;
    }

    @Override
    public void flush() {
        this.writer.flush();
    }

    @Override
    public JournalReader openReader() {
        SegmentedJournalReader reader = new SegmentedJournalReader(this);
        this.readers.add(reader);
        return reader;
    }

    @Override
    public boolean isOpen() {
        return this.open;
    }

    @Override
    public void close() {
        this.segments.values().forEach(segment -> {
            this.log.debug("Closing segment: {}", segment);
            segment.close();
        });
        this.currentSegment = null;
        this.open = false;
    }

    private synchronized void open() {
        long startTime = System.currentTimeMillis();
        for (JournalSegment segment : this.loadSegments()) {
            this.segments.put(segment.descriptor().index(), segment);
            this.journalMetrics.incSegmentCount();
        }
        if (!this.segments.isEmpty()) {
            this.currentSegment = this.segments.lastEntry().getValue();
        } else {
            JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(1L).withMaxSegmentSize(this.maxSegmentSize).build();
            this.currentSegment = this.createSegment(descriptor);
            this.currentSegment.descriptor().update(System.currentTimeMillis());
            this.segments.put(1L, this.currentSegment);
            this.journalMetrics.incSegmentCount();
        }
        this.journalMetrics.observeJournalOpenDuration(System.currentTimeMillis() - startTime);
    }

    private void assertOpen() {
        Preconditions.checkState((this.currentSegment != null ? 1 : 0) != 0, (Object)"journal not open");
    }

    private void assertDiskSpace() {
        if (this.directory().getUsableSpace() < Math.max(this.maxSegmentSize() * 3L, this.minFreeDiskSpace)) {
            throw new StorageException.OutOfDiskSpace("Not enough space to allocate a new journal segment");
        }
    }

    private long maxSegmentSize() {
        return this.maxSegmentSize;
    }

    private File directory() {
        return this.directory;
    }

    private synchronized void resetCurrentSegment() {
        JournalSegment lastSegment = this.getLastSegment();
        if (lastSegment != null) {
            this.currentSegment = lastSegment;
        } else {
            JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(1L).withMaxSegmentSize(this.maxSegmentSize).build();
            this.currentSegment = this.createSegment(descriptor);
            this.segments.put(1L, this.currentSegment);
            this.journalMetrics.incSegmentCount();
        }
    }

    JournalSegment resetSegments(long index) {
        this.assertOpen();
        for (JournalSegment segment : this.segments.values()) {
            segment.close();
            segment.delete();
            this.journalMetrics.decSegmentCount();
        }
        this.segments.clear();
        JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(1L).withIndex(index).withMaxSegmentSize(this.maxSegmentSize).build();
        this.currentSegment = this.createSegment(descriptor);
        this.segments.put(index, this.currentSegment);
        this.journalMetrics.incSegmentCount();
        return this.currentSegment;
    }

    JournalSegment getFirstSegment() {
        this.assertOpen();
        Map.Entry<Long, JournalSegment> segment = this.segments.firstEntry();
        return segment != null ? segment.getValue() : null;
    }

    JournalSegment getLastSegment() {
        this.assertOpen();
        Map.Entry<Long, JournalSegment> segment = this.segments.lastEntry();
        return segment != null ? segment.getValue() : null;
    }

    synchronized JournalSegment getNextSegment() {
        this.assertOpen();
        this.assertDiskSpace();
        JournalSegment lastSegment = this.getLastSegment();
        JournalSegmentDescriptor descriptor = JournalSegmentDescriptor.builder().withId(lastSegment != null ? lastSegment.descriptor().id() + 1L : 1L).withIndex(this.currentSegment.lastIndex() + 1L).withMaxSegmentSize(this.maxSegmentSize).build();
        this.currentSegment = this.createSegment(descriptor);
        this.segments.put(descriptor.index(), this.currentSegment);
        this.journalMetrics.incSegmentCount();
        return this.currentSegment;
    }

    JournalSegment getNextSegment(long index) {
        Map.Entry<Long, JournalSegment> nextSegment = this.segments.higherEntry(index);
        return nextSegment != null ? nextSegment.getValue() : null;
    }

    synchronized JournalSegment getSegment(long index) {
        this.assertOpen();
        if (this.currentSegment != null && index > this.currentSegment.index()) {
            return this.currentSegment;
        }
        Map.Entry<Long, JournalSegment> segment = this.segments.floorEntry(index);
        if (segment != null) {
            return segment.getValue();
        }
        return this.getFirstSegment();
    }

    synchronized void removeSegment(JournalSegment segment) {
        this.segments.remove(segment.index());
        this.journalMetrics.decSegmentCount();
        segment.close();
        segment.delete();
        this.resetCurrentSegment();
    }

    JournalSegment createSegment(JournalSegmentDescriptor descriptor) {
        FileChannel channel;
        RandomAccessFile raf;
        File segmentFile = JournalSegmentFile.createSegmentFile(this.name, this.directory, descriptor.id());
        try {
            raf = new RandomAccessFile(segmentFile, "rw");
            raf.setLength(descriptor.maxSegmentSize());
            channel = raf.getChannel();
        }
        catch (IOException e) {
            throw new StorageException(e);
        }
        ByteBuffer buffer = ByteBuffer.allocate(64);
        descriptor.copyTo(buffer);
        buffer.flip();
        try {
            channel.write(buffer);
        }
        catch (IOException e) {
            throw new StorageException(e);
        }
        finally {
            try {
                channel.close();
                raf.close();
            }
            catch (IOException e) {
                this.log.warn("Unexpected IOException on closing", (Throwable)e);
            }
        }
        JournalSegment segment = this.newSegment(new JournalSegmentFile(segmentFile), descriptor);
        this.log.debug("Created segment: {}", (Object)segment);
        return segment;
    }

    protected JournalSegment newSegment(JournalSegmentFile segmentFile, JournalSegmentDescriptor descriptor) {
        return new JournalSegment(segmentFile, descriptor, this.maxEntrySize, this.journalIndex);
    }

    private JournalSegment loadSegment(long segmentId) {
        JournalSegment journalSegment;
        block8: {
            File segmentFile = JournalSegmentFile.createSegmentFile(this.name, this.directory, segmentId);
            ByteBuffer buffer = ByteBuffer.allocate(64);
            FileChannel channel = this.openChannel(segmentFile);
            try {
                channel.read(buffer);
                buffer.flip();
                JournalSegmentDescriptor descriptor = new JournalSegmentDescriptor(buffer);
                JournalSegment segment = this.newSegment(new JournalSegmentFile(segmentFile), descriptor);
                this.log.debug("Loaded disk segment: {} ({})", (Object)descriptor.id(), (Object)segmentFile.getName());
                journalSegment = segment;
                if (channel == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (channel != null) {
                        try {
                            channel.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new StorageException(e);
                }
            }
            channel.close();
        }
        return journalSegment;
    }

    private FileChannel openChannel(File file) {
        try {
            return FileChannel.open(file.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        }
        catch (IOException e) {
            throw new StorageException(e);
        }
    }

    protected Collection<JournalSegment> loadSegments() {
        this.directory.mkdirs();
        TreeMap<Long, JournalSegment> segments = new TreeMap<Long, JournalSegment>();
        for (File file : this.directory.listFiles(File::isFile)) {
            if (!JournalSegmentFile.isSegmentFile(this.name, file)) continue;
            JournalSegmentFile segmentFile = new JournalSegmentFile(file);
            ByteBuffer buffer = ByteBuffer.allocate(64);
            try (FileChannel channel = this.openChannel(file);){
                channel.read(buffer);
                buffer.flip();
            }
            catch (IOException e) {
                throw new StorageException(e);
            }
            JournalSegmentDescriptor descriptor = new JournalSegmentDescriptor(buffer);
            JournalSegment segment = this.loadSegment(descriptor.id());
            this.log.debug("Found segment: {} ({})", (Object)segment.descriptor().id(), (Object)segmentFile.file().getName());
            segments.put(segment.index(), segment);
        }
        JournalSegment previousSegment = null;
        boolean corrupted = false;
        Iterator iterator = segments.entrySet().iterator();
        while (iterator.hasNext()) {
            JournalSegment segment = (JournalSegment)iterator.next().getValue();
            if (previousSegment != null && previousSegment.lastIndex() != segment.index() - 1L) {
                this.log.warn("Journal is inconsistent. {} is not aligned with prior segment {}", (Object)segment.file().file(), (Object)previousSegment.file().file());
                corrupted = true;
            }
            if (corrupted) {
                segment.close();
                segment.delete();
                iterator.remove();
            }
            previousSegment = segment;
        }
        return segments.values();
    }

    public void closeReader(SegmentedJournalReader segmentedJournalReader) {
        this.readers.remove(segmentedJournalReader);
    }

    void resetHead(long index) {
        for (SegmentedJournalReader reader : this.readers) {
            if (reader.getNextIndex() > index) continue;
            reader.seek(index);
        }
    }

    void resetTail(long index) {
        for (SegmentedJournalReader reader : this.readers) {
            if (reader.getNextIndex() >= index) {
                reader.seek(index);
                continue;
            }
            reader.seek(reader.getNextIndex());
        }
    }

    public JournalMetrics getJournalMetrics() {
        return this.journalMetrics;
    }

    public JournalIndex getJournalIndex() {
        return this.journalIndex;
    }
}

