/*
 * Copyright The OpenZipkin Authors
 * SPDX-License-Identifier: Apache-2.0
 */
package zipkin2.reporter.libthrift;

import java.io.Closeable;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.thrift.TConfiguration;
import org.apache.thrift.TException;
import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.transport.TSocket;
import org.apache.thrift.transport.TTransportException;
import org.apache.thrift.transport.layered.TFramedTransport;

final class ScribeClient implements Closeable {
  static final Logger logger = Logger.getLogger(ScribeClient.class.getName());
  static final byte[] category = new byte[] {'z', 'i', 'p', 'k', 'i', 'n'};

  final String host;
  final int port;
  final int socketTimeout;
  final int connectTimeout;

  volatile TSocket socket;
  volatile TBinaryProtocol prot;

  /**
   * This defers opening a socket until the first call to {@link #log}, to
   * accommodate a server that's down when the client initializes.
   */
  ScribeClient(String host, int port, int socketTimeout, int connectTimeout) {
    this.host = host;
    this.port = port;
    this.socketTimeout = socketTimeout;
    this.connectTimeout = connectTimeout;
  }

  static int messageSizeInBytes(int spanSizeInBytes) {
    return InternalScribeCodec.messageSizeInBytes(category, spanSizeInBytes);
  }

  static int messageSizeInBytes(List<byte[]> encodedSpans) {
    return InternalScribeCodec.messageSizeInBytes(category, encodedSpans);
  }

  private int seqid_;

  boolean log(List<byte[]> encodedSpans) throws TException {
    try {
      // Starting in version 0.14, TSocket opens a socket inside its
      // constructor, which we defer so that the server can be initially down.
      if (socket == null) {
        synchronized (this) {
          if (socket == null) {
            socket = new TSocket(new TConfiguration(), host, port, socketTimeout, connectTimeout);
            prot = new TBinaryProtocol(new TFramedTransport(socket));
          }
        }
      }

      if (!socket.isOpen()) socket.open();
      InternalScribeCodec.writeLogRequest(category, encodedSpans, ++seqid_, prot);
      prot.getTransport().flush();
      return InternalScribeCodec.readLogResponse(seqid_, prot);
    } catch (TTransportException e) {
      logger.log(Level.FINE, "Transport exception. recreating socket", e);
      close();
      seqid_ = 0;
      throw e;
    }
  }

  @Override public void close() {
    TSocket maybe = this.socket;
    if (maybe != null) maybe.close();
  }
}
