/*
 * Copyright (2020) The Delta Lake Project Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.delta.tables.execution

import scala.collection.Map

import org.apache.spark.sql.delta.{ DeltaErrors, DeltaHistoryManager, DeltaLog, PreprocessTableUpdate }
import org.apache.spark.sql.delta.commands.{ DeleteCommand, DeltaGenerateCommand, VacuumCommand }
import org.apache.spark.sql.delta.util.AnalysisHelper
import io.delta.tables.DeltaTable

import org.apache.spark.sql.{ functions, Column, DataFrame }
import org.apache.spark.sql.catalyst.TableIdentifier
import org.apache.spark.sql.catalyst.analysis.UnresolvedAttribute
import org.apache.spark.sql.catalyst.expressions.{ Expression, SubqueryExpression }
import org.apache.spark.sql.catalyst.plans.logical._

/**
  * Interface to provide the actual implementations of DeltaTable operations.
  */
trait DeltaTableOperations extends AnalysisHelper { self: DeltaTable =>

  protected def executeDelete(condition: Option[Expression]): Unit = {
    val delete = DeltaDelete(self.toDF.queryExecution.analyzed, condition)

    // current DELETE does not support subquery,
    // and the reason why perform checking here is that
    // we want to have more meaningful exception messages,
    // instead of having some random msg generated by executePlan().
    subqueryNotSupportedCheck(condition, "DELETE")

    val qe             = sparkSession.sessionState.executePlan(delete)
    val resolvedDelete = qe.analyzed.asInstanceOf[DeltaDelete]
    val deleteCommand  = DeleteCommand(resolvedDelete)
    deleteCommand.run(sparkSession)
  }

  protected def toStrColumnMap(map: Map[String, String]): Map[String, Column] = {
    map.toSeq.map { case (k, v) => k -> functions.expr(v) }.toMap
  }

  protected def makeUpdateTable(
    target: DeltaTable,
    onCondition: Option[Column],
    setColumns: Seq[(String, Column)]
  ): DeltaUpdateTable = {
    val updateColumns     = setColumns.map { x => UnresolvedAttribute.quotedString(x._1) }
    val updateExpressions = setColumns.map { x => x._2.expr }
    val condition         = onCondition.map { _.expr }
    DeltaUpdateTable(target.toDF.queryExecution.analyzed, updateColumns, updateExpressions, condition)
  }

  protected def executeHistory(deltaLog: DeltaLog, limit: Option[Int]): DataFrame = {
    val history = new DeltaHistoryManager(deltaLog)
    val spark   = self.toDF.sparkSession
    spark.createDataFrame(history.getHistory(limit))
  }

  protected def executeGenerate(tblIdentifier: String, mode: String): Unit = {
    val tableId: TableIdentifier = sparkSession.sessionState.sqlParser
      .parseTableIdentifier(tblIdentifier)
    val generate                 = DeltaGenerateCommand(mode, tableId)
    generate.run(sparkSession)
  }

  protected def executeUpdate(set: Map[String, Column], condition: Option[Column]): Unit = {
    val setColumns = set.map { case (col, expr) => (col, expr) }.toSeq

    // Current UPDATE does not support subquery,
    // and the reason why perform checking here is that
    // we want to have more meaningful exception messages,
    // instead of having some random msg generated by executePlan().
    subqueryNotSupportedCheck(condition.map { _.expr }, "UPDATE")

    val update         = makeUpdateTable(self, condition, setColumns)
    val resolvedUpdate =
      DeltaUpdateTable.resolveReferences(update, tryResolveReferences(sparkSession)(_, update))
    val updateCommand  = PreprocessTableUpdate(sparkSession.sessionState.conf)(resolvedUpdate)
    updateCommand.run(sparkSession)
  }

  private def subqueryNotSupportedCheck(condition: Option[Expression], op: String): Unit = {
    condition.foreach { cond =>
      if (SubqueryExpression.hasSubquery(cond)) {
        throw DeltaErrors.subqueryNotSupportedException(op, cond)
      }
    }
  }

  protected def executeVacuum(deltaLog: DeltaLog, retentionHours: Option[Double]): DataFrame = {
    VacuumCommand.gc(sparkSession, deltaLog, false, retentionHours)
    sparkSession.emptyDataFrame
  }

  protected def sparkSession = self.toDF.sparkSession
}
