package it.unibo.tuprolog.utils.impl

import it.unibo.tuprolog.utils.Cache
import it.unibo.tuprolog.utils.Optional
import it.unibo.tuprolog.utils.buffered
import kotlin.jvm.Synchronized

internal class SimpleLRUCache<K, V>(override val capacity: Int) : Cache<K, V> {

    init {
        require(capacity > 0)
    }

    private val cache = LinkedHashMap<K, V>()

    @Synchronized
    override fun set(key: K, value: V): Optional<out Pair<K, V>> {
        val evicted = removeLeastRecentIfNecessary()
        cache[key] = value
        return evicted
    }

    private fun removeLeastRecent(): Optional<out Pair<K, V>> {
        val entry = cache.iterator().next()
        cache.remove(entry.key)
        return Optional.some(entry.toPair())
    }

    private fun removeLeastRecentIfNecessary(): Optional<out Pair<K, V>> {
        return if (cache.size >= capacity) {
            removeLeastRecent()
        } else {
            return Optional.none()
        }
    }

    @Synchronized
    override fun get(key: K): Optional<out V> =
        if (cache.containsKey(key)) {
            Optional.of(cache[key])
        } else {
            Optional.none()
        }

    @Synchronized
    override fun toMap(): Map<K, V> =
        cache.toMap()

    @Synchronized
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other == null || this::class != other::class) return false

        other as SimpleLRUCache<*, *>

        if (capacity != other.capacity) return false
        if (cache != other.cache) return false

        return true
    }

    @Synchronized
    override fun hashCode(): Int {
        var result = capacity
        result = 31 * result + cache.hashCode()
        return result
    }

    @Synchronized
    override fun toSequence(): Sequence<Pair<K, V>> {
        return cache.entries.asSequence().map { it.toPair() }.buffered()
    }

    @Synchronized
    override fun toString(): String {
        return "SimpleLRUCache(${toSequence().map { "${it.first} = ${it.second}" }.joinToString(", ")})"
    }

    override val size: Int
        @Synchronized
        get() = cache.size
}
