/*
	* Copyright (C) 2002-2017 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/
package it.unimi.dsi.fastutil.bytes;

import java.util.List;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Collection;
import java.util.NoSuchElementException;

/**
 * An abstract class providing basic methods for lists implementing a
 * type-specific list interface.
 *
 * <P>
 * As an additional bonus, this class implements on top of the list operations a
 * type-specific stack.
 */
public abstract class AbstractByteList extends AbstractByteCollection implements ByteList, ByteStack {
	protected AbstractByteList() {
	}

	/**
	 * Ensures that the given index is nonnegative and not greater than the list
	 * size.
	 *
	 * @param index
	 *            an index.
	 * @throws IndexOutOfBoundsException
	 *             if the given index is negative or greater than the list size.
	 */
	protected void ensureIndex(final int index) {
		if (index < 0) throw new IndexOutOfBoundsException("Index (" + index + ") is negative");
		if (index > size()) throw new IndexOutOfBoundsException("Index (" + index + ") is greater than list size (" + (size()) + ")");
	}

	/**
	 * Ensures that the given index is nonnegative and smaller than the list
	 * size.
	 *
	 * @param index
	 *            an index.
	 * @throws IndexOutOfBoundsException
	 *             if the given index is negative or not smaller than the list
	 *             size.
	 */
	protected void ensureRestrictedIndex(final int index) {
		if (index < 0) throw new IndexOutOfBoundsException("Index (" + index + ") is negative");
		if (index >= size()) throw new IndexOutOfBoundsException("Index (" + index + ") is greater than or equal to list size (" + (size()) + ")");
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation always throws an
	 * {@link UnsupportedOperationException}.
	 */
	@Override
	public void add(final int index, final byte k) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the type-specific version of
	 * {@link List#add(int, Object)}.
	 */
	@Override
	public boolean add(final byte k) {
		add(size(), k);
		return true;
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation always throws an
	 * {@link UnsupportedOperationException}.
	 */
	@Override
	public byte removeByte(int i) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation always throws an
	 * {@link UnsupportedOperationException}.
	 */
	@Override
	public byte set(final int index, final byte k) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Adds all of the elements in the specified collection to this list
	 * (optional operation).
	 */
	@Override
	public boolean addAll(int index, final Collection<? extends Byte> c) {
		ensureIndex(index);
		final Iterator<? extends Byte> i = c.iterator();
		final boolean retVal = i.hasNext();
		while (i.hasNext())
			add(index++, i.next());
		return retVal;
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the type-specific version of
	 * {@link List#addAll(int, Collection)}.
	 */
	@Override
	public boolean addAll(final Collection<? extends Byte> c) {
		return addAll(size(), c);
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the new covariantly stronger generic
	 * method {@link #listIterator()}.
	 * 
	 * @deprecated As of <code>fastutil</code> 5, replaced by
	 *             {@link #listIterator()}.
	 */
	@Deprecated
	@Override
	public ByteListIterator byteListIterator() {
		return listIterator();
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the new covariantly stronger generic
	 * method {@link #listIterator(int)}.
	 * 
	 * @deprecated As of <code>fastutil</code> 5, replaced by
	 *             {@link #listIterator()}.
	 */
	@Deprecated
	@Override
	public ByteListIterator byteListIterator(final int index) {
		return listIterator(index);
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to {@link #listIterator()}.
	 */
	@Override
	public ByteListIterator iterator() {
		return listIterator();
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to {@link #listIterator(int)
	 * listIterator(0)}.
	 */
	@Override
	public ByteListIterator listIterator() {
		return listIterator(0);
	}

	/**
	 * {@inheritDoc}
	 * <P>
	 * This implementation is based on the random-access methods.
	 */
	@Override
	public ByteListIterator listIterator(final int index) {
		ensureIndex(index);
		return new AbstractByteListIterator() {
			int pos = index, last = -1;

			@Override
			public boolean hasNext() {
				return pos < AbstractByteList.this.size();
			}

			@Override
			public boolean hasPrevious() {
				return pos > 0;
			}

			@Override
			public byte nextByte() {
				if (!hasNext()) throw new NoSuchElementException();
				return AbstractByteList.this.getByte(last = pos++);
			}

			@Override
			public byte previousByte() {
				if (!hasPrevious()) throw new NoSuchElementException();
				return AbstractByteList.this.getByte(last = --pos);
			}

			@Override
			public int nextIndex() {
				return pos;
			}

			@Override
			public int previousIndex() {
				return pos - 1;
			}

			@Override
			public void add(byte k) {
				AbstractByteList.this.add(pos++, k);
				last = -1;
			}

			@Override
			public void set(byte k) {
				if (last == -1) throw new IllegalStateException();
				AbstractByteList.this.set(last, k);
			}

			@Override
			public void remove() {
				if (last == -1) throw new IllegalStateException();
				AbstractByteList.this.removeByte(last);
				/*
				 * If the last operation was a next(), we are removing an
				 * element *before* us, and we must decrease pos
				 * correspondingly.
				 */
				if (last < pos) pos--;
				last = -1;
			}
		};
	}

	/**
	 * Returns true if this list contains the specified element.
	 * <p>
	 * This implementation delegates to {@code indexOf()}.
	 * 
	 * @see List#contains(Object)
	 */
	@Override
	public boolean contains(final byte k) {
		return indexOf(k) >= 0;
	}

	@Override
	public int indexOf(final byte k) {
		final ByteListIterator i = listIterator();
		byte e;
		while (i.hasNext()) {
			e = i.nextByte();
			if (((k) == (e))) return i.previousIndex();
		}
		return -1;
	}

	@Override
	public int lastIndexOf(final byte k) {
		ByteListIterator i = listIterator(size());
		byte e;
		while (i.hasPrevious()) {
			e = i.previousByte();
			if (((k) == (e))) return i.nextIndex();
		}
		return -1;
	}

	@Override
	public void size(final int size) {
		int i = size();
		if (size > i) while (i++ < size)
			add(((byte) 0));
		else while (i-- != size)
			remove(i);
	}

	@Override
	public ByteList subList(final int from, final int to) {
		ensureIndex(from);
		ensureIndex(to);
		if (from > to) throw new IndexOutOfBoundsException("Start index (" + from + ") is greater than end index (" + to + ")");
		return new ByteSubList(this, from, to);
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the new covariantly stronger generic
	 * method {@link #subList(int, int)}.
	 * 
	 * @deprecated As of <code>fastutil</code> 5, replaced by
	 *             {@link #subList(int, int)}.
	 */
	@Deprecated
	@Override
	public ByteList byteSubList(final int from, final int to) {
		return subList(from, to);
	}

	/**
	 * {@inheritDoc}
	 *
	 * <P>
	 * This is a trivial iterator-based implementation. It is expected that
	 * implementations will override this method with a more optimized version.
	 */
	@Override
	public void removeElements(final int from, final int to) {
		ensureIndex(to);
		ByteListIterator i = listIterator(from);
		int n = to - from;
		if (n < 0) throw new IllegalArgumentException("Start index (" + from + ") is greater than end index (" + to + ")");
		while (n-- != 0) {
			i.nextByte();
			i.remove();
		}
	}

	/**
	 * {@inheritDoc}
	 *
	 * <P>
	 * This is a trivial iterator-based implementation. It is expected that
	 * implementations will override this method with a more optimized version.
	 */
	@Override
	public void addElements(int index, final byte a[], int offset, int length) {
		ensureIndex(index);
		if (offset < 0) throw new ArrayIndexOutOfBoundsException("Offset (" + offset + ") is negative");
		if (offset + length > a.length) throw new ArrayIndexOutOfBoundsException("End index (" + (offset + length) + ") is greater than array length (" + a.length + ")");
		while (length-- != 0)
			add(index++, a[offset++]);
	}

	/**
	 * {@inheritDoc}
	 *
	 * <P>
	 * This implementation delegates to the analogous method for array
	 * fragments.
	 */
	@Override
	public void addElements(final int index, final byte a[]) {
		addElements(index, a, 0, a.length);
	}

	/**
	 * {@inheritDoc}
	 *
	 * <P>
	 * This is a trivial iterator-based implementation. It is expected that
	 * implementations will override this method with a more optimized version.
	 */
	@Override
	public void getElements(final int from, final byte a[], int offset, int length) {
		ByteListIterator i = listIterator(from);
		if (offset < 0) throw new ArrayIndexOutOfBoundsException("Offset (" + offset + ") is negative");
		if (offset + length > a.length) throw new ArrayIndexOutOfBoundsException("End index (" + (offset + length) + ") is greater than array length (" + a.length + ")");
		if (from + length > size()) throw new IndexOutOfBoundsException("End index (" + (from + length) + ") is greater than list size (" + size() + ")");
		while (length-- != 0)
			a[offset++] = i.nextByte();
	}

	/**
	 * {@inheritDoc}
	 * <P>
	 * This implementation delegates to {@link #removeElements(int, int)}.
	 */
	@Override
	public void clear() {
		removeElements(0, size());
	}

	private boolean valEquals(final Object a, final Object b) {
		return a == null ? b == null : a.equals(b);
	}

	/**
	 * Returns the hash code for this list, which is identical to
	 * {@link java.util.List#hashCode()}.
	 *
	 * @return the hash code for this list.
	 */
	@Override
	public int hashCode() {
		ByteIterator i = iterator();
		int h = 1, s = size();
		while (s-- != 0) {
			byte k = i.nextByte();
			h = 31 * h + (k);
		}
		return h;
	}

	@Override
	public boolean equals(final Object o) {
		if (o == this) return true;
		if (!(o instanceof List)) return false;
		final List<?> l = (List<?>) o;
		int s = size();
		if (s != l.size()) return false;
		if (l instanceof ByteList) {
			final ByteListIterator i1 = listIterator(),
					i2 = ((ByteList) l).listIterator();
			while (s-- != 0)
				if (i1.nextByte() != i2.nextByte()) return false;
			return true;
		}
		final ListIterator<?> i1 = listIterator(), i2 = l.listIterator();
		while (s-- != 0)
			if (!valEquals(i1.next(), i2.next())) return false;
		return true;
	}

	/**
	 * Compares this list to another object. If the argument is a
	 * {@link java.util.List}, this method performs a lexicographical
	 * comparison; otherwise, it throws a <code>ClassCastException</code>.
	 *
	 * @param l
	 *            a list.
	 * @return if the argument is a {@link java.util.List}, a negative integer,
	 *         zero, or a positive integer as this list is lexicographically
	 *         less than, equal to, or greater than the argument.
	 * @throws ClassCastException
	 *             if the argument is not a list.
	 */

	@Override
	public int compareTo(final List<? extends Byte> l) {
		if (l == this) return 0;
		if (l instanceof ByteList) {
			final ByteListIterator i1 = listIterator(),
					i2 = ((ByteList) l).listIterator();
			int r;
			byte e1, e2;
			while (i1.hasNext() && i2.hasNext()) {
				e1 = i1.nextByte();
				e2 = i2.nextByte();
				if ((r = (Byte.compare((e1), (e2)))) != 0) return r;
			}
			return i2.hasNext() ? -1 : (i1.hasNext() ? 1 : 0);
		}
		ListIterator<? extends Byte> i1 = listIterator(), i2 = l.listIterator();
		int r;
		while (i1.hasNext() && i2.hasNext()) {
			if ((r = ((Comparable<? super Byte>) i1.next()).compareTo(i2.next())) != 0) return r;
		}
		return i2.hasNext() ? -1 : (i1.hasNext() ? 1 : 0);
	}

	@Override
	public void push(byte o) {
		add(o);
	}

	@Override
	public byte popByte() {
		if (isEmpty()) throw new NoSuchElementException();
		return removeByte(size() - 1);
	}

	@Override
	public byte topByte() {
		if (isEmpty()) throw new NoSuchElementException();
		return getByte(size() - 1);
	}

	@Override
	public byte peekByte(int i) {
		return getByte(size() - 1 - i);
	}

	/**
	 * Removes a single instance of the specified element from this collection,
	 * if it is present (optional operation).
	 * <p>
	 * This implementation delegates to {@code indexOf()}.
	 * 
	 * @see List#remove(Object)
	 */
	@Override
	public boolean rem(byte k) {
		int index = indexOf(k);
		if (index == -1) return false;
		removeByte(index);
		return true;
	}

	@Override
	public boolean addAll(int index, final ByteCollection c) {
		ensureIndex(index);
		final ByteIterator i = c.iterator();
		final boolean retVal = i.hasNext();
		while (i.hasNext())
			add(index++, i.nextByte());
		return retVal;
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the type-specific version of
	 * {@link List#addAll(int, Collection)}.
	 */
	@Override
	public boolean addAll(final int index, final ByteList l) {
		return addAll(index, (ByteCollection) l);
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the type-specific version of
	 * {@link List#addAll(int, Collection)}.
	 */
	public boolean addAll(final ByteCollection c) {
		return addAll(size(), c);
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the type-specific list version of
	 * {@link List#addAll(int, Collection)}.
	 */
	public boolean addAll(final ByteList l) {
		return addAll(size(), l);
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the corresponding type-specific method.
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public void add(final int index, final Byte ok) {
		add(index, ok.byteValue());
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the corresponding type-specific method.
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Byte set(final int index, final Byte ok) {
		return (Byte.valueOf(set(index, ok.byteValue())));
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the corresponding type-specific method.
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Byte get(final int index) {
		return (Byte.valueOf(getByte(index)));
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the corresponding type-specific method.
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public int indexOf(final Object ok) {
		return indexOf(((((Byte) (ok)).byteValue())));
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the corresponding type-specific method.
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public int lastIndexOf(final Object ok) {
		return lastIndexOf(((((Byte) (ok)).byteValue())));
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the corresponding type-specific method.
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Byte remove(final int index) {
		return (Byte.valueOf(removeByte(index)));
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the corresponding type-specific method.
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public void push(Byte o) {
		push(o.byteValue());
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the corresponding type-specific method.
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Byte pop() {
		return Byte.valueOf(popByte());
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the corresponding type-specific method.
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Byte top() {
		return Byte.valueOf(topByte());
	}

	/**
	 * {@inheritDoc}
	 *
	 * <p>
	 * This implementation delegates to the corresponding type-specific method.
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Byte peek(int i) {
		return Byte.valueOf(peekByte(i));
	}

	@Override
	public String toString() {
		final StringBuilder s = new StringBuilder();
		final ByteIterator i = iterator();
		int n = size();
		byte k;
		boolean first = true;
		s.append("[");
		while (n-- != 0) {
			if (first) first = false;
			else s.append(", ");
			k = i.nextByte();
			s.append(String.valueOf(k));
		}
		s.append("]");
		return s.toString();
	}

	/** A class implementing a sublist view. */
	public static class ByteSubList extends AbstractByteList implements java.io.Serializable {
		private static final long serialVersionUID = -7046029254386353129L;
		/** The list this sublist restricts. */
		protected final ByteList l;
		/** Initial (inclusive) index of this sublist. */
		protected final int from;
		/** Final (exclusive) index of this sublist. */
		protected int to;

		public ByteSubList(final ByteList l, final int from, final int to) {
			this.l = l;
			this.from = from;
			this.to = to;
		}

		private boolean assertRange() {
			assert from <= l.size();
			assert to <= l.size();
			assert to >= from;
			return true;
		}

		@Override
		public boolean add(final byte k) {
			l.add(to, k);
			to++;
			assert assertRange();
			return true;
		}

		@Override
		public void add(final int index, final byte k) {
			ensureIndex(index);
			l.add(from + index, k);
			to++;
			assert assertRange();
		}

		@Override
		public boolean addAll(final int index, final Collection<? extends Byte> c) {
			ensureIndex(index);
			to += c.size();
			return l.addAll(from + index, c);
		}

		@Override
		public byte getByte(int index) {
			ensureRestrictedIndex(index);
			return l.getByte(from + index);
		}

		@Override
		public byte removeByte(int index) {
			ensureRestrictedIndex(index);
			to--;
			return l.removeByte(from + index);
		}

		@Override
		public byte set(int index, byte k) {
			ensureRestrictedIndex(index);
			return l.set(from + index, k);
		}

		@Override
		public int size() {
			return to - from;
		}

		@Override
		public void getElements(final int from, final byte[] a, final int offset, final int length) {
			ensureIndex(from);
			if (from + length > size()) throw new IndexOutOfBoundsException("End index (" + from + length + ") is greater than list size (" + size() + ")");
			l.getElements(this.from + from, a, offset, length);
		}

		@Override
		public void removeElements(final int from, final int to) {
			ensureIndex(from);
			ensureIndex(to);
			l.removeElements(this.from + from, this.from + to);
			this.to -= (to - from);
			assert assertRange();
		}

		@Override
		public void addElements(int index, final byte a[], int offset, int length) {
			ensureIndex(index);
			l.addElements(this.from + index, a, offset, length);
			this.to += length;
			assert assertRange();
		}

		@Override
		public ByteListIterator listIterator(final int index) {
			ensureIndex(index);
			return new AbstractByteListIterator() {
				int pos = index, last = -1;

				public boolean hasNext() {
					return pos < size();
				}

				public boolean hasPrevious() {
					return pos > 0;
				}

				public byte nextByte() {
					if (!hasNext()) throw new NoSuchElementException();
					return l.getByte(from + (last = pos++));
				}

				public byte previousByte() {
					if (!hasPrevious()) throw new NoSuchElementException();
					return l.getByte(from + (last = --pos));
				}

				public int nextIndex() {
					return pos;
				}

				public int previousIndex() {
					return pos - 1;
				}

				public void add(byte k) {
					if (last == -1) throw new IllegalStateException();
					ByteSubList.this.add(pos++, k);
					last = -1;
					assert assertRange();
				}

				public void set(byte k) {
					if (last == -1) throw new IllegalStateException();
					ByteSubList.this.set(last, k);
				}

				public void remove() {
					if (last == -1) throw new IllegalStateException();
					ByteSubList.this.removeByte(last);
					/*
					 * If the last operation was a next(), we are removing an
					 * element *before* us, and we must decrease pos
					 * correspondingly.
					 */
					if (last < pos) pos--;
					last = -1;
					assert assertRange();
				}
			};
		}

		@Override
		public ByteList subList(final int from, final int to) {
			ensureIndex(from);
			ensureIndex(to);
			if (from > to) throw new IllegalArgumentException("Start index (" + from + ") is greater than end index (" + to + ")");
			return new ByteSubList(this, from, to);
		}

		@Override
		public boolean rem(byte k) {
			int index = indexOf(k);
			if (index == -1) return false;
			to--;
			l.removeByte(from + index);
			assert assertRange();
			return true;
		}

		@Override
		public boolean addAll(final int index, final ByteCollection c) {
			ensureIndex(index);
			return super.addAll(index, c);
		}

		@Override
		public boolean addAll(final int index, final ByteList l) {
			ensureIndex(index);
			return super.addAll(index, l);
		}
	}
}
