/*
	* Copyright (C) 2010-2017 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/
package it.unimi.dsi.fastutil.floats;
import java.util.Iterator;
import java.util.Collection;
import java.util.NoSuchElementException;
import it.unimi.dsi.fastutil.BigList;
import it.unimi.dsi.fastutil.BigListIterator;
/**  An abstract class providing basic methods for big lists implementing a type-specific big list interface. */
public abstract class AbstractFloatBigList extends AbstractFloatCollection implements FloatBigList , FloatStack {
	protected AbstractFloatBigList() {}
	/** Ensures that the given index is nonnegative and not greater than this big-list size.
	 *
	 * @param index an index.
	 * @throws IndexOutOfBoundsException if the given index is negative or greater than this big-list size.
	 */
	protected void ensureIndex(final long index) {
	 if (index < 0) throw new IndexOutOfBoundsException("Index (" + index + ") is negative");
	 if (index > size64()) throw new IndexOutOfBoundsException("Index (" + index + ") is greater than list size (" + (size64()) + ")");
	}
	/** Ensures that the given index is nonnegative and smaller than this big-list size.
	 *
	 * @param index an index.
	 * @throws IndexOutOfBoundsException if the given index is negative or not smaller than this big-list size.
	 */
	protected void ensureRestrictedIndex(final long index) {
	 if (index < 0) throw new IndexOutOfBoundsException("Index (" + index + ") is negative");
	 if (index >= size64()) throw new IndexOutOfBoundsException("Index (" + index + ") is greater than or equal to list size (" + (size64()) + ")");
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation always throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public void add(final long index, final float k) {
	 throw new UnsupportedOperationException();
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the type-specific version of {@link BigList#add(long, Object)}.
	 */
	@Override
	public boolean add(final float k) {
	 add(size64(), k);
	 return true;
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation always throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public float removeFloat(long i) {
	 throw new UnsupportedOperationException();
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the type-specific version of {@link BigList#remove(long)}.
	 * @deprecated This method should have never been implemented, as it is not
	 * part of {@link BigList}. It will be removed in the future.
	 */
	@Deprecated
	public float removeFloat(int i) {
	 return removeFloat((long)i);
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation always throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public float set(final long index, final float k) {
	 throw new UnsupportedOperationException();
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the type-specific version of {@link BigList#set(long, Object)}.
	 * @deprecated This method should have never been implemented, as it is not
	 * part of {@link BigList}. It will be removed in the future.
	 */
	@Deprecated
	public float set(final int index, final float k) {
	 return set((long)index, k);
	}
	/** Adds all of the elements in the specified collection to this list (optional operation). */
	@Override
	public boolean addAll(long index, final Collection<? extends Float> c) {
	 ensureIndex(index);
	 final Iterator<? extends Float> i = c.iterator();
	 final boolean retVal = i.hasNext();
	 while(i.hasNext()) add(index++, i.next());
	 return retVal;
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the type-specific version of {@link BigList#addAll(long, Collection)}.
	 * @deprecated This method should have never been implemented, as it is not
	 * part of {@link BigList}. It will be removed in the future.
	 */
	@Deprecated
	public boolean addAll(int index, final Collection<? extends Float> c) {
	 return addAll((long)index, c);
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the type-specific version of {@link BigList#addAll(long, Collection)}.
	 */
	public boolean addAll(final Collection<? extends Float> c) {
	 return addAll(size64(), c);
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to {@link #listIterator()}.
	 */
	@Override
	public FloatBigListIterator iterator() {
	 return listIterator();
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to {@link BigList#listIterator(long) listIterator(0)}.
	 */
	@Override
	public FloatBigListIterator listIterator() {
	 return listIterator(0L);
	}
	/** {@inheritDoc}
	 * <P>This implementation is based on the random-access methods. */
	@Override
	public FloatBigListIterator listIterator(final long index) {
	 ensureIndex(index);
	 return new AbstractFloatBigListIterator () {
	   long pos = index, last = -1;
	   @Override
	   public boolean hasNext() { return pos < AbstractFloatBigList.this.size64(); }
	   @Override
	   public boolean hasPrevious() { return pos > 0; }
	   @Override
	   public float nextFloat() { if (! hasNext()) throw new NoSuchElementException(); return AbstractFloatBigList.this.getFloat(last = pos++); }
	   @Override
	   public float previousFloat() { if (! hasPrevious()) throw new NoSuchElementException(); return AbstractFloatBigList.this.getFloat(last = --pos); }
	   @Override
	   public long nextIndex() { return pos; }
	   @Override
	   public long previousIndex() { return pos - 1; }
	   @Override
	   public void add(float k) {
	    AbstractFloatBigList.this.add(pos++, k);
	    last = -1;
	   }
	   @Override
	   public void set(float k) {
	    if (last == -1) throw new IllegalStateException();
	    AbstractFloatBigList.this.set(last, k);
	   }
	   @Override
	   public void remove() {
	    if (last == -1) throw new IllegalStateException();
	    AbstractFloatBigList.this.removeFloat(last);
	    /* If the last operation was a next(), we are removing an element *before* us, and we must decrease pos correspondingly. */
	    if (last < pos) pos--;
	    last = -1;
	   }
	  };
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to {@link BigList#listIterator(long)}.
	 * @deprecated This method should have never been implemented, as it is not
	 * part of {@link BigList}. It will be removed in the future.
	 */
	@Deprecated
	public FloatBigListIterator listIterator(final int index) {
	 return listIterator((long)index);
	}
	/** Returns true if this list contains the specified element.
	 * <p>This implementation delegates to {@code indexOf()}.
	 * @see BigList#contains(Object)
	 */
	@Override
	public boolean contains(final float k) {
	 return indexOf(k) >= 0;
	}
	@Override
	public long indexOf(final float k) {
	 final FloatBigListIterator i = listIterator();
	 float e;
	 while(i.hasNext()) {
	  e = i.nextFloat();
	  if (( Float.floatToIntBits(k) == Float.floatToIntBits(e) )) return i.previousIndex();
	 }
	 return -1;
	}
	@Override
	public long lastIndexOf(final float k) {
	 FloatBigListIterator i = listIterator(size64());
	 float e;
	 while(i.hasPrevious()) {
	  e = i.previousFloat();
	  if (( Float.floatToIntBits(k) == Float.floatToIntBits(e) )) return i.nextIndex();
	 }
	 return -1;
	}
	@Override
	public void size(final long size) {
	 long i = size64();
	 if (size > i) while(i++ < size) add((0));
	 else while(i-- != size) remove(i);
	}
	/**
	 * <p>This implementation delegates to {@link BigList#size(long)}.
	 * @deprecated This method should have never been implemented, as it is not
	 * part of {@link BigList}. It will be removed in the future.
	 */
	@Deprecated
	public void size(final int size) {
	 size((long)size);
	}
	@Override
	public FloatBigList subList(final long from, final long to) {
	 ensureIndex(from);
	 ensureIndex(to);
	 if (from > to) throw new IndexOutOfBoundsException("Start index (" + from + ") is greater than end index (" + to + ")");
	 return new FloatSubList (this, from, to);
	}
	/** {@inheritDoc}
	 *
	 * <P>This is a trivial iterator-based implementation. It is expected that
	 * implementations will override this method with a more optimized version.
	 */
	@Override
	public void removeElements(final long from, final long to) {
	 ensureIndex(to);
	 FloatBigListIterator i = listIterator(from);
	 long n = to - from;
	 if (n < 0) throw new IllegalArgumentException("Start index (" + from + ") is greater than end index (" + to + ")");
	 while(n-- != 0) {
	  i.nextFloat();
	  i.remove();
	 }
	}
	/** {@inheritDoc}
	 *
	 * <P>This is a trivial iterator-based implementation. It is expected that
	 * implementations will override this method with a more optimized version.
	 */
	@Override
	public void addElements(long index, final float a[][], long offset, long length) {
	 ensureIndex(index);
	 FloatBigArrays.ensureOffsetLength(a, offset, length);
	 while(length-- != 0) add(index++, FloatBigArrays.get(a, offset++));
	}
	/** {@inheritDoc}
	 *
	 * <P>This implementation delegates to the analogous method for big-array fragments.
	 */
	@Override
	public void addElements(final long index, final float a[][]) {
	 addElements(index, a, 0, FloatBigArrays.length(a));
	}
	/** {@inheritDoc}
	 *
	 * <P>This is a trivial iterator-based implementation. It is expected that
	 * implementations will override this method with a more optimized version.
	 */
	@Override
	public void getElements(final long from, final float a[][], long offset, long length) {
	 FloatBigListIterator i = listIterator(from);
	 FloatBigArrays.ensureOffsetLength(a, offset, length);
	 if (from + length > size64()) throw new IndexOutOfBoundsException("End index (" + (from + length) + ") is greater than list size (" + size64() + ")");
	 while(length-- != 0) FloatBigArrays.set(a, offset++, i.nextFloat());
	}
	/** {@inheritDoc}
	 * <P>This implementation delegates to {@link #removeElements(long, long)}.*/
	@Override
	public void clear() {
	 removeElements(0, size64());
	}
	/** {@inheritDoc}
	 *
	 * <P>This implementation delegates to {@link #size64()}.
	 * @deprecated Please use the {@link #size64()} instead. */
	@Override
	@Deprecated
	public int size() {
	 return (int)Math.min(Integer.MAX_VALUE, size64());
	}
	private boolean valEquals(final Object a, final Object b) {
	 return a == null ? b == null : a.equals(b);
	}
	/** Returns the hash code for this big list, which is identical to {@link java.util.List#hashCode()}.
	 *
	 * @return the hash code for this big list.
	 */
	@Override
	public int hashCode() {
	 FloatIterator i = iterator();
	 int h = 1;
	 long s = size64();
	 while (s-- != 0) {
	  float k = i.nextFloat();
	  h = 31 * h + it.unimi.dsi.fastutil.HashCommon.float2int(k);
	 }
	 return h;
	}
	@Override
	public boolean equals(final Object o) {
	 if (o == this) return true;
	 if (! (o instanceof BigList)) return false;
	 final BigList<?> l = (BigList<?>)o;
	 long s = size64();
	 if (s != l.size64()) return false;
	 if (l instanceof FloatBigList) {
	  final FloatBigListIterator i1 = listIterator(), i2 = ((FloatBigList )l).listIterator();
	  while(s-- != 0) if (i1.nextFloat() != i2.nextFloat()) return false;
	  return true;
	 }
	 final BigListIterator<?> i1 = listIterator(), i2 = l.listIterator();
	 while(s-- != 0) if (! valEquals(i1.next(), i2.next())) return false;
	 return true;
	}
	/** Compares this big list to another object. If the
	 * argument is a {@link BigList}, this method performs a lexicographical comparison; otherwise,
	 * it throws a <code>ClassCastException</code>.
	 *
	 * @param l a big list.
	 * @return if the argument is a {@link BigList}, a negative integer,
	 * zero, or a positive integer as this list is lexicographically less than, equal
	 * to, or greater than the argument.
	 * @throws ClassCastException if the argument is not a big list.
	 */

	@Override
	public int compareTo(final BigList<? extends Float> l) {
	 if (l == this) return 0;
	 if (l instanceof FloatBigList) {
	  final FloatBigListIterator i1 = listIterator(), i2 = ((FloatBigList )l).listIterator();
	  int r;
	  float e1, e2;
	  while(i1.hasNext() && i2.hasNext()) {
	   e1 = i1.nextFloat();
	   e2 = i2.nextFloat();
	   if ((r = ( Float.compare((e1),(e2)) )) != 0) return r;
	  }
	  return i2.hasNext() ? -1 : (i1.hasNext() ? 1 : 0);
	 }
	 BigListIterator<? extends Float> i1 = listIterator(), i2 = l.listIterator();
	 int r;
	 while(i1.hasNext() && i2.hasNext()) {
	  if ((r = ((Comparable<? super Float>)i1.next()).compareTo(i2.next())) != 0) return r;
	 }
	 return i2.hasNext() ? -1 : (i1.hasNext() ? 1 : 0);
	}
	@Override
	public void push(float o) {
	 add(o);
	}
	@Override
	public float popFloat() {
	 if (isEmpty()) throw new NoSuchElementException();
	 return removeFloat(size64() - 1);
	}
	@Override
	public float topFloat() {
	 if (isEmpty()) throw new NoSuchElementException();
	 return getFloat(size64() - 1);
	}
	@Override
	public float peekFloat(int i) {
	 return getFloat(size64() - 1 - i);
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the type-specific version of {@link BigList#get(long)}.
	 * @deprecated This method should have never been implemented, as it is not
	 * part of {@link BigList}. It will be removed in the future.
	 */
	@Deprecated
	public float getFloat(final int index) {
	 return getFloat((long)index);
	}
	/** Removes a single instance of the specified element from this collection, if it is present (optional operation).
	 * <p>This implementation delegates to {@code indexOf()}.
	 * @see BigList#remove(Object)
	 */
	@Override
	public boolean rem(float k) {
	 long index = indexOf(k);
	 if (index == -1) return false;
	 removeFloat(index);
	 return true;
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the type-specific version of {@link #addAll(long, Collection)}.
	 */
	@Override
	public boolean addAll(final long index, final FloatCollection c) {
	 return addAll(index, (Collection<? extends Float>)c);
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the type-specific version of {@link #addAll(long, Collection)}.
	 */
	@Override
	public boolean addAll(final long index, final FloatBigList l) {
	 return addAll(index, (FloatCollection)l);
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the type-specific version of {@link #addAll(long, Collection)}.
	 */
	@Override
	public boolean addAll(final FloatCollection c) {
	 return addAll(size64(), c);
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the type-specific list version of {@link #addAll(long, Collection)}.
	 */
	@Override
	public boolean addAll(final FloatBigList l) {
	 return addAll(size64(), l);
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public void add(final long index, final Float ok) {
	 add(index, ok.floatValue());
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Float set(final long index, final Float ok) {
	 return (Float.valueOf(set(index, ok.floatValue())));
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Float get(final long index) {
	 return (Float.valueOf(getFloat(index)));
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public long indexOf(final Object ok) {
	 return indexOf(((((Float)(ok)).floatValue())));
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public long lastIndexOf(final Object ok) {
	 return lastIndexOf(((((Float)(ok)).floatValue())));
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated This method should have never been implemented, as it is not
	 * part of {@link BigList}. It will be removed in the future.
	 */
	@Deprecated
	public Float remove(final int index) {
	 return (Float.valueOf(removeFloat(index)));
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Float remove(final long index) {
	 return (Float.valueOf(removeFloat(index)));
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public void push(Float o) {
	 push(o.floatValue());
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Float pop() {
	 return Float.valueOf(popFloat());
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Float top() {
	 return Float.valueOf(topFloat());
	}
	/** {@inheritDoc}
	 *
	 * <p>This implementation delegates to the corresponding type-specific method.
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	public Float peek(int i) {
	 return Float.valueOf(peekFloat(i));
	}
	@Override
	public String toString() {
	 final StringBuilder s = new StringBuilder();
	 final FloatIterator i = iterator();
	 long n = size64();
	 float k;
	 boolean first = true;
	 s.append("[");
	 while(n-- != 0) {
	  if (first) first = false;
	  else s.append(", ");
	  k = i.nextFloat();
	   s.append(String.valueOf(k));
	 }
	 s.append("]");
	 return s.toString();
	}
	/** A class implementing a sublist view. */
	public static class FloatSubList extends AbstractFloatBigList implements java.io.Serializable {
	 private static final long serialVersionUID = -7046029254386353129L;
	 /** The list this sublist restricts. */
	 protected final FloatBigList l;
	 /** Initial (inclusive) index of this sublist. */
	 protected final long from;
	 /** Final (exclusive) index of this sublist. */
	 protected long to;
	 public FloatSubList(final FloatBigList l, final long from, final long to) {
	  this.l = l;
	  this.from = from;
	  this.to = to;
	 }
	 private boolean assertRange() {
	  assert from <= l.size64();
	  assert to <= l.size64();
	  assert to >= from;
	  return true;
	 }
	 @Override
	 public boolean add(final float k) {
	  l.add(to, k);
	  to++;
	  assert assertRange();
	  return true;
	 }
	 @Override
	 public void add(final long index, final float k) {
	  ensureIndex(index);
	  l.add(from + index, k);
	  to++;
	  assert assertRange();
	 }
	 @Override
	 public boolean addAll(final long index, final Collection<? extends Float> c) {
	  ensureIndex(index);
	  to += c.size();
	  return l.addAll(from + index, c);
	 }
	 @Override
	 public float getFloat(long index) {
	  ensureRestrictedIndex(index);
	  return l.getFloat(from + index);
	 }
	 @Override
	 public float removeFloat(long index) {
	  ensureRestrictedIndex(index);
	  to--;
	  return l.removeFloat(from + index);
	 }
	 @Override
	 public float set(long index, float k) {
	  ensureRestrictedIndex(index);
	  return l.set(from + index, k);
	 }
	 @Override
	 public long size64() {
	  return to - from;
	 }
	 @Override
	 public void getElements(final long from, final float[][] a, final long offset, final long length) {
	  ensureIndex(from);
	  if (from + length > size64()) throw new IndexOutOfBoundsException("End index (" + from + length + ") is greater than list size (" + size64() + ")");
	  l.getElements(this.from + from, a, offset, length);
	 }
	 @Override
	 public void removeElements(final long from, final long to) {
	  ensureIndex(from);
	  ensureIndex(to);
	  l.removeElements(this.from + from, this.from + to);
	  this.to -= (to - from);
	  assert assertRange();
	 }
	 @Override
	 public void addElements(final long index, final float a[][], long offset, long length) {
	  ensureIndex(index);
	  l.addElements(this.from + index, a, offset, length);
	  this.to += length;
	  assert assertRange();
	 }
	 @Override
	 public FloatBigListIterator listIterator(final long index) {
	  ensureIndex(index);
	  return new AbstractFloatBigListIterator () {
	    long pos = index, last = -1;
	    public boolean hasNext() { return pos < size64(); }
	    public boolean hasPrevious() { return pos > 0; }
	    public float nextFloat() { if (! hasNext()) throw new NoSuchElementException(); return l.getFloat(from + (last = pos++)); }
	    public float previousFloat() { if (! hasPrevious()) throw new NoSuchElementException(); return l.getFloat(from + (last = --pos)); }
	    public long nextIndex() { return pos; }
	    public long previousIndex() { return pos - 1; }
	    public void add(float k) {
	     if (last == -1) throw new IllegalStateException();
	     FloatSubList.this.add(pos++, k);
	     last = -1;
	     assert assertRange();
	    }
	    public void set(float k) {
	     if (last == -1) throw new IllegalStateException();
	     FloatSubList.this.set(last, k);
	    }
	    public void remove() {
	     if (last == -1) throw new IllegalStateException();
	     FloatSubList.this.removeFloat(last);
	     /* If the last operation was a next(), we are removing an element *before* us, and we must decrease pos correspondingly. */
	     if (last < pos) pos--;
	     last = -1;
	     assert assertRange();
	    }
	   };
	 }
	 @Override
	 public FloatBigList subList(final long from, final long to) {
	  ensureIndex(from);
	  ensureIndex(to);
	  if (from > to) throw new IllegalArgumentException("Start index (" + from + ") is greater than end index (" + to + ")");
	  return new FloatSubList (this, from, to);
	 }
	 @Override
	 public boolean rem(float k) {
	  long index = indexOf(k);
	  if (index == -1) return false;
	  to--;
	  l.removeFloat(from + index);
	  assert assertRange();
	  return true;
	 }
	 @Override
	 public boolean addAll(final long index, final FloatCollection c) {
	  ensureIndex(index);
	  return super.addAll(index, c);
	 }
	 @Override
	 public boolean addAll(final long index, final FloatBigList l) {
	  ensureIndex(index);
	  return super.addAll(index, l);
	 }
	}
}
