package it.unimi.dsi.mg4j.document;

/*		 
 * MG4J: Managing Gigabytes for Java
 *
 * Copyright (C) 2009-2011 Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */


import it.unimi.dsi.fastutil.objects.Reference2ObjectMap;
import it.unimi.dsi.lang.ObjectParser;

import java.io.IOException;
import java.io.InputStream;

/** A collection that exhibits a contiguous subsets of documents from a given collection.
 * 
 * <p>This class provides several string-based constructors that use the {@link ObjectParser}
 * conventions; they can be used to generate easily subcollections from the command line.
 * 
 * @author Sebastiano Vigna
 *
 */

public class SubDocumentCollection extends AbstractDocumentCollection {
	/** The underlying document collection. */
	final DocumentCollection underlyingCollection;
	/** The first document (inclusive) in this subcollection. */
	final int first;
	/** The last document (exclusive) in this subcollection. */
	final int last;

	/** Creates a new subcollection.
	 * 
	 * @param underlyingCollection the underlying document collection.
	 * @param first the first document (inclusive) in the subcollection.
	 * @param last the last document (exclusive) in this subcollection.
	 */
	public SubDocumentCollection( DocumentCollection underlyingCollection, int first, int last ) {
		this.underlyingCollection = underlyingCollection;
		this.first = first;
		this.last = last;
	}

	/** Creates a new subcollection starting from a given document.
	 * 
	 * <p>The new subcollection will contain all documents from the given one onwards.
	 * 
	 * @param underlyingCollection the underlying document collection.
	 * @param first the first document (inclusive) in the subcollection.
	 */
	public SubDocumentCollection( DocumentCollection underlyingCollection, int first ) {
		this( underlyingCollection, first, underlyingCollection.size() );
	}

	/** Creates a new subcollection.
	 * 
	 * @param underlyingCollectionBasename the basename of the underlying document collection.
	 * @param first the first document (inclusive) in the subcollection.
	 * @param last the last document (exclusive) in this subcollection.
	 */
	public SubDocumentCollection( String underlyingCollectionBasename, String first, String last ) throws NumberFormatException, IllegalArgumentException, SecurityException, IOException, ClassNotFoundException {
		this( (DocumentCollection)AbstractDocumentSequence.load( underlyingCollectionBasename ),
				Integer.parseInt( first ), Integer.parseInt( last ) );
	}

	/** Creates a new subcollection starting from a given document.
	 * 
	 * <p>The new subcollection will contain all documents from the given one onwards.
	 * 
	 * @param underlyingCollectionBasename the basename of the underlying document collection.
	 * @param first the first document (inclusive) in the subcollection.
	 */
	public SubDocumentCollection( String underlyingCollectionBasename, String first ) throws NumberFormatException, IllegalArgumentException, SecurityException, IOException, ClassNotFoundException {
		this( (DocumentCollection)AbstractDocumentSequence.load( underlyingCollectionBasename ),
				Integer.parseInt( first ) );
	}
	
	public DocumentCollection copy() {
		return new SubDocumentCollection( underlyingCollection.copy(), first, last );
	}

	public Document document( int index ) throws IOException {
		ensureDocumentIndex( index );
		return underlyingCollection.document( first + index );
	}

	public int size() {
		return last - first;
	}

	public Reference2ObjectMap<Enum<?>, Object> metadata( int index ) throws IOException {
		ensureDocumentIndex( index );
		return underlyingCollection.metadata( first + index );
	}

	public InputStream stream( int index ) throws IOException {
		ensureDocumentIndex( index );
		return underlyingCollection.stream( first + index );
	}

	public DocumentFactory factory() {
		return underlyingCollection.factory();
	}
}
