package it.unimi.dsi.mg4j.index;

/*		 
 * MG4J: Managing Gigabytes for Java
 *
 * Copyright (C) 2007-2011 Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import it.unimi.dsi.fastutil.ints.IntList;
import it.unimi.dsi.fastutil.longs.LongList;
import it.unimi.dsi.io.ByteBufferInputStream;
import it.unimi.dsi.io.InputBitStream;
import it.unimi.dsi.mg4j.index.CompressionFlags.Coding;
import it.unimi.dsi.mg4j.index.payload.Payload;
import it.unimi.dsi.mg4j.tool.PartitionLexically;
import it.unimi.dsi.util.PrefixMap;
import it.unimi.dsi.util.Properties;
import it.unimi.dsi.util.StringMap;

import java.nio.MappedByteBuffer;

/** A local memory-mapped bistream index.
 * 
 * <p>Memory-mapped indices are created by mapping the index file into memory
 * using a {@link MappedByteBuffer}. The main advantage over an {@link InMemoryIndex}
 * is that only the most frequently used parts of the index will be loaded in core memory.
 * 
 * <p>Note that due to insurmountable Java limitations, it is impossible to map an index larger than
 * 2GiB. However, you can {@linkplain PartitionLexically partition lexically} an index so that
 * the resulting segments are smaller than 2GiB, and modify the property file of the resulting
 * cluster so that the URIs of the local indices require memory mapping. This will effectively memory-map
 * the whole index. 
 *  
 * @author Sebastiano Vigna
 * @since 1.2
 */

public class MemoryMappedIndex extends BitStreamIndex {
	private static final long serialVersionUID = 0L;
	
	/** The byte buffer containing the index. */
	protected final ByteBufferInputStream index;

	public MemoryMappedIndex( final ByteBufferInputStream index, final int numberOfDocuments, final int numberOfTerms, final long numberOfPostings,
			final long numberOfOccurrences, final int maxCount, final Payload payload, final Coding frequencyCoding, final Coding pointerCoding, final Coding countCoding, final Coding positionCoding, final int quantum, final int height, 
			final TermProcessor termProcessor, final String field, final Properties properties, final StringMap<? extends CharSequence> termMap, final PrefixMap<? extends CharSequence> prefixMap, final IntList sizes, final LongList offsets ) {
		super( numberOfDocuments, numberOfTerms, numberOfPostings, numberOfOccurrences, maxCount, payload, frequencyCoding, pointerCoding, countCoding, positionCoding, quantum, height, -1, termProcessor, field, properties, termMap, prefixMap, sizes, offsets );
		this.index = index;
	}

	@Override
	public InputBitStream getInputBitStream( int bufferSizeUnused ) {
		return new InputBitStream( getInputStream() );
	}
	
	@Override
	public ByteBufferInputStream getInputStream() {
		return index.copy();
	}
}
