package it.unimi.dsi.mg4j.index.cluster;
/*		 
 * MG4J: Managing Gigabytes for Java
 *
 * Copyright (C) 2006-2011 Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */
import it.unimi.dsi.fastutil.ints.IntIterator;
import it.unimi.dsi.mg4j.index.Index;
import it.unimi.dsi.mg4j.index.IndexIterator;
import it.unimi.dsi.mg4j.index.payload.Payload;
import it.unimi.dsi.mg4j.search.visitor.DocumentIteratorVisitor;

import java.io.IOException;


/** An index iterator concatenating iterators from local indices.
 *  
 * @author Alessandro Arrabito
 * @author Sebastiano Vigna
 */

public class DocumentalConcatenatedClusterIndexIterator extends DocumentalConcatenatedClusterDocumentIterator implements IndexIterator {
	/** A cached copy of the iterators, to avoid type casts. */
	protected final IndexIterator[] indexIterator;
	/** The precomputed frequency. */
	protected final int frequency;
	/** The reference index. */
	protected final DocumentalConcatenatedCluster index;
	/** The term associated with this index iterator. */
	protected String term;
	/** The id associated with this index iterator. */
	protected int id;
	
	public DocumentalConcatenatedClusterIndexIterator( final DocumentalClusterIndexReader indexReader, final IndexIterator[] indexIterator, final int[] usedIndex ) throws IOException {
		super( indexReader, indexIterator, usedIndex );
		this.index = (DocumentalConcatenatedCluster)indexReader.index;
		this.indexIterator = indexIterator;
		int t = 0;
		for ( int i = indexIterator.length; i-- != 0; ) t += indexIterator[ i ].frequency();
		frequency = t;
	}

	public DocumentalConcatenatedClusterIndexIterator term( final CharSequence term ) {
		this.term = term == null ? null : term.toString();
		return this;
	}

	public String term() { 
		return term;
	}

	public DocumentalConcatenatedClusterIndexIterator id( final int id ) {
		this.id = id;
		return this;
	}
	
	public int id() {
		return id;
	}
	
	public Index index() {
		return index;
	}
	
	public int frequency() {
		return frequency;
	}

	public Payload payload() throws IOException {
		if ( lastIterator < 0 ) throw new IllegalStateException( "There is no current payload: nextDocument() has never been called" );
		return indexIterator[ lastIterator ].payload();
	}
	
	public int count() throws IOException {
		if ( lastIterator < 0 ) throw new IllegalStateException( "There is no current count: nextDocument() has never been called" );
		return indexIterator[ lastIterator ].count();
	}

	public IntIterator positions() throws IOException {
		if ( lastIterator < 0 ) throw new IllegalStateException( "There are no current positions: nextDocument() has never been called" );
		return indexIterator[ lastIterator ].positions();
	}

	public int positions( int[] positions ) throws IOException {
		if ( lastIterator < 0 ) throw new IllegalStateException( "There are no current positions: nextDocument() has never been called" );
		return indexIterator[ lastIterator ].positions( positions );
	}

	public int[] positionArray() throws IOException {
		if ( lastIterator < 0 ) throw new IllegalStateException( "There are no current positions: nextDocument() has never been called" );
		return indexIterator[ lastIterator ].positionArray();
	}

	public <T> T accept( final DocumentIteratorVisitor<T> visitor ) throws IOException {
		return visitor.visit( this );
	}
	
	public <T> T acceptOnTruePaths( final DocumentIteratorVisitor<T> visitor ) throws IOException {
		return visitor.visit( this );
	}

	public int termNumber() {
		// ALERT: to be implemented
		throw new UnsupportedOperationException( "To be implemented" );
	}
	
	public IndexIterator weight( final double weight ) {
		super.weight( weight );
		return this;
	}
	
}
