package it.unimi.dsi.mg4j.index.remote;

/*		 
 * MG4J: Managing Gigabytes for Java
 *
 * Copyright (C) 2006-2011 Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import it.unimi.dsi.fastutil.ints.IntList;
import it.unimi.dsi.fastutil.longs.LongList;
import it.unimi.dsi.io.InputBitStream;
import it.unimi.dsi.mg4j.index.BitStreamIndex;
import it.unimi.dsi.mg4j.index.IndexReader;
import it.unimi.dsi.mg4j.index.TermProcessor;
import it.unimi.dsi.mg4j.index.CompressionFlags.Coding;
import it.unimi.dsi.mg4j.index.payload.Payload;
import it.unimi.dsi.util.PrefixMap;
import it.unimi.dsi.util.Properties;
import it.unimi.dsi.util.StringMap;

import java.io.IOException;
import java.io.InputStream;
import java.net.SocketAddress;

/** A remote bitstream-based index.
 *
 * <p>Remote bitstream indices (as opposed to {@linkplain it.unimi.dsi.mg4j.index.remote.RemoteIndex remote indices}
 * export the actual bitstream by means of a reader. Of course, this is possible only if the
 * remoted indices is a {@link it.unimi.dsi.mg4j.index.BitStreamIndex}.
 * 
 * @author Sebastiano Vigna
 * @since 1.1
 */

public class RemoteBitStreamIndex extends BitStreamIndex {
	static final long serialVersionUID = 0L;
	
	// TODO: find a way to use index-specific default buffer sizes
	/** The default buffer size. */
	@SuppressWarnings("hiding")
	public final static int DEFAULT_BUFFER_SIZE = 512;
	/** The address of the socket associated with this index. */
	public final SocketAddress address;
			
	public RemoteBitStreamIndex( final SocketAddress address, int numberOfDocuments, int numberOfTerms, final long numberOfPostings, final long numberOfOccurrences, int maxCount,
			Payload payload, Coding frequencyCoding, Coding pointerCoding, Coding countCoding, Coding positionCoding, int quantum, int height, int bufferSize, TermProcessor termProcessor, String field, Properties properties, StringMap<? extends CharSequence> termMap, PrefixMap<? extends CharSequence> prefixMap,
			IntList sizes, LongList offsets ) {
		super( numberOfDocuments, numberOfTerms, numberOfPostings, numberOfOccurrences, maxCount, payload, frequencyCoding, pointerCoding, countCoding, positionCoding, quantum, height, bufferSize, termProcessor, field, properties,
				termMap, prefixMap, sizes, offsets );
		this.address = address;
	}
	
	@Override
	public InputBitStream getInputBitStream( final int bufferSize ) throws IOException {
		return new InputBitStream( new RemoteInputStream( address ), bufferSize );
	}

	@Override
	public InputStream getInputStream() throws IOException {
		return new RemoteInputStream( address );
	}

	@Override
	public IndexReader getReader( final int bufferSize ) throws IOException {
		// TODO: we should use generic classes
		try {
			return getConstructor().newInstance( this, getInputBitStream( bufferSize == -1 ? this.bufferSize : bufferSize ) );
		}
		catch ( IOException e ) {
			throw e;
		}
		catch ( Exception e ) {
			throw new RuntimeException( e );
		}
	}
}
