package it.unimi.dsi.mg4j.index.remote;

/*		 
 * MG4J: Managing Gigabytes for Java
 *
 * Copyright (C) 2006-2011 Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import it.unimi.dsi.Util;
import it.unimi.dsi.fastutil.longs.AbstractLongList;
import it.unimi.dsi.fastutil.longs.LongList;

import java.io.EOFException;
import java.io.IOException;
import java.io.Serializable;
import java.net.Socket;
import java.net.SocketAddress;

import org.apache.log4j.Logger;


/** A class accessing remotely a {@link it.unimi.dsi.fastutil.longs.LongList}. 
 * 
 * @author Alessandro Arrabito
 * @author Sebastiano Vigna
 */
public class RemoteOffsetList extends AbstractLongList implements Serializable {
	static final long serialVersionUID = 2L;
	
	/** A remote server connection, lazily initialised at the first remote call. */
	private transient RemoteIndexServerConnection connection;
	/** The address of the socket assigned to the server thread. */
	private SocketAddress address;
	/** The size of the list, cached locally. */
	private int size;
	
	/** Creates a new remote long list.
	 * @param addr the address of the socket assigned to the server thread.
	 * @param size the size of the list.
	 */
	public RemoteOffsetList( final SocketAddress addr, final int size ) {
		address = addr;
		this.size = size;
	}
	
	public long getLong( int index ) {
		try {
			if ( connection == null ) connection = new RemoteIndexServerConnection( address, IndexServer.GET_OFFSET_LIST );
			connection.outputStream.writeInt( index );
			connection.outputStream.flush();
			return connection.inputStream.readLong();
		}
		catch ( IOException e ) {
			throw new RuntimeException( e );
		}
	}

	public int size() {
		return size;
	}

	public static class ServerThread extends it.unimi.dsi.mg4j.index.remote.ServerThread {
		private final static boolean DEBUG = false;
		private final static Logger LOGGER = Util.getLogger( ServerThread.class );
		/** The remoted list. */
		private final LongList list;

		public ServerThread( final Socket socket, final LongList list ) throws IOException {
			super( socket );
			this.list = list;
		}

		public void run() {
			try {
				int index;
				for ( ;; ) {
					index = inputStream.readInt();
					if ( DEBUG ) LOGGER.debug( "Received request for index " + index );
					outputStream.writeLong( list.getLong( index ) );
					outputStream.flush();
				}
			}
			catch ( EOFException e ) {
				LOGGER.warn( "The socket has been closed" );
			}
			catch ( Exception e ) {
				LOGGER.fatal( e, e );
			}
		}
	}
}
