package it.unimi.dsi.mg4j.search;

/*		 
 * MG4J: Managing Gigabytes for Java
 *
 * Copyright (C) 2007-2011 Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import it.unimi.dsi.fastutil.ints.AbstractIntIterator;
import it.unimi.dsi.fastutil.ints.IntIterator;
import it.unimi.dsi.fastutil.objects.AbstractObjectIterator;
import it.unimi.dsi.util.Interval;

import java.io.IOException;
import java.util.Iterator;
import java.util.NoSuchElementException;

/** An abstract iterator on documents that implements {@link IntIterator#hasNext() hasNext()}
 * and {@link IntIterator#nextInt() nextInt()} using {@link DocumentIterator#nextDocument() nextDocument()},
 * and provides support for the {@link DocumentIterator#weight()}/{@link DocumentIterator#weight(double)} methods.
 *
 * <p>As explained elsewhere, since MG4J 1.2 the iteration logic has been made fully lazy,
 * and the standard {@link IntIterator} methods are available as a commodity; however, their use
 * in performance-sensitive environments is strongly discouraged. The fully lazy
 * implementation needs some bridging to be accessible using {@link java.util}'s semi-lazy
 * {@linkplain Iterator iterators}, and this class provides the necessary code. In MG4J 4.0 the class
 * has been redesigned and is not backward compatible, but this should not be a problem unless
 * you implemented your own document iterators.
 * 
 * <p>Instances of this class expect implementation to keep track of the {@linkplain #curr current document}
 * of the iterator. The special value -1 denotes an iterator that has not still been accessed,
 * and the special value {@link DocumentIterator#END_OF_LIST} denotes an iterator that has been exhausted. 
 * 
 * <p>This class keeps track of whether it is {@linkplain #ahead}, that is, {@link #hasNext()} has been called
 * but {@link #nextInt()} or {@link #nextDocument()} have not yet: in this case, {@link #curr} has
 * not been returned yet by {@link #nextInt()} or {@link #nextDocument()}. {@link #ahead} may
 * be true only if {@link #curr} is neither -1, nor {@link DocumentIterator#END_OF_LIST}.
 * 
 * <p>Concrete subclasses must implement a
 * {@link #nextDocumentInternal()} method that implements only the true iterator logic, forgetting about {@link #ahead}
 * and assuming the {@link #curr} is not {@link DocumentIterator#END_OF_LIST}. Moreover, {@link #ahead} must be always
 * set to false in {@link #skipTo(int)}.
 * 
 * <p>Methods performing actions depending on the last document returned should throw an {@link IllegalStateException}
 * if called when {@link #ahead} is true, or if {@link #curr} is -1 or {@link DocumentIterator#END_OF_LIST}. 
 * You just need to call {@link #ensureOnADocument()}.
 * 
 * <p>Finally, {@link #toNextDocument(int)} will turn the value of {@link #curr} into a suitable return value
 * for {@link #nextDocument()} (as {@link DocumentIterator#END_OF_LIST} needs to be massaged).
 */

public abstract class AbstractDocumentIterator extends AbstractIntIterator implements DocumentIterator {
	/** The current document of the iterator. The special value -1 means that that 
	 * @see AbstractDocumentIterator */
	protected int curr = -1;
	/** Whether this iterator is ahead.
	 * @see AbstractDocumentIterator */
	protected boolean ahead;
	/** The weight of this iterator. */
	protected double weight = 1;
	
	/** Turns the value of the argument into a valid return value of {@link #nextDocument()}
	 * 
	 * @param curr a value for {@link #curr}, including possibly {@link DocumentIterator#END_OF_LIST}.
	 * @return the correct return value for {@link #nextDocument()}.
	 */
	protected static int toNextDocument( final int curr ) {
		return ( curr + 1 ) & 0x80000000 | curr;
	}
	
	/** Turns a value returned by {@link #nextDocument()} into a valid value for {@link #curr}.
	 * 
	 * @param d a value returned by {@link #nextDocument()}.
	 * @return the correct return value for {@link #curr}.
	 */
	protected static int fromNextDocument( final int d ) {
		return d & 0x7FFFFFFF;
	}
	
	public double weight() {
		return weight;
	}
	
	public DocumentIterator weight( final double weight ) {
		this.weight = weight;
		return this;
	}
	
	/** Invokes {@link DocumentIterator#intervalIterator()}
	 * 
	 * @return {@link DocumentIterator#intervalIterator()}.
	 */
	public IntervalIterator iterator() {
		try {
			return intervalIterator();
		}
		catch ( IOException e ) {
			throw new RuntimeException( e );
		}
	}

	/** Checks whether {@link #ahead} is true; if not, sets {@link #ahead} if {#nextDocument()} returns a document.
	 *
	 * @return true if {@link #ahead} is true or {@link #nextDocument()} returns a document.
	 */
	public boolean hasNext() {
		if ( ! ahead  ) try {
			ahead = nextDocument() != -1;
		}
		catch ( IOException e ) {
			throw new RuntimeException( e );
		}
		return ahead; 
	}
	
	/** Checks whether there is an element to be returned; if so, sets {@link #ahead} to false and returns {@link #curr}.
	 *
	 * @return {@link #curr}.
	 */
	@Deprecated
	public int nextInt() {
		if ( ! hasNext() ) throw new NoSuchElementException();
		ahead = false;
		return curr;
	}

	protected abstract int nextDocumentInternal() throws IOException;
	
	public int nextDocument() throws IOException {
		if ( ahead ) {
			// We already know what to return
			ahead = false;
			return curr;
		}

		if ( curr == END_OF_LIST ) return -1;
		
		return nextDocumentInternal();
	}
	
	protected final void ensureOnADocument() {
		// This catches ahead || curr == END_OF_LIST || curr == -1.
		if ( ahead || ( curr | 0x80000000 ) == -1 ) throw new IllegalStateException();
	}
	
	/** Returns the current document.
	 * 
	 * @return {@link #curr}.
	 * @throws IllegalStateException if {@link #ahead} is true;
	 */
	
	public int document() {
		if ( ahead ) throw new IllegalStateException();
		return curr; 
	}
	
	protected abstract static class AbstractIntervalIterator extends AbstractObjectIterator<Interval> implements IntervalIterator {
		/** The next document to be returned, if it has already been peeked at by {@link #hasNext()},
		 * or <code>null</code>. */
		protected Interval next;
		
		/** Checks whether {@link #next} has been already set; if not, peeks at the interval returned by {@link IntervalIterator#nextInterval() nextInterval()}.
		 *
		 * @return true if {@link #next} is not <code>null</code> or if {@link IntervalIterator#nextInterval() nextInterval()} has returned a valid interval.
		 */
		public boolean hasNext() {
			if ( next == null ) try {
				next = nextInterval();
			}
			catch ( IOException e ) {
				throw new RuntimeException( e );
			}
			return next != null;
		}
		
		/** Checks whether there is an interval to be returned, sets
		 * {@link #next} to <code>null</code>, and returns its previous value.
		 *
		 * @return the next interval, as cached by {@link #hasNext()}.
		 */
		@Deprecated
		public Interval next() {
			if ( ! hasNext() ) throw new NoSuchElementException();
			final Interval result = next;
			next = null;
			return result;
		}
	}
}
