package it.unimi.dsi.mg4j.search;

/*		 
 * MG4J: Managing Gigabytes for Java
 *
 * Copyright (C) 2003-2011 Paolo Boldi and Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import java.io.IOException;

/** An abstract document iterator helping in the implementation of {@link it.unimi.dsi.mg4j.search.ConsecutiveDocumentIterator}
 * and {@link it.unimi.dsi.mg4j.search.OrderedAndDocumentIterator}. 
 */

public abstract class AbstractOrderedIntervalDocumentIterator extends AbstractIntersectionDocumentIterator {

	public final static boolean DEBUG = false;
	public final static boolean ASSERTS = false;
	
	/** Creates a new abstract document iterator.
	 * 
	 * @param documentIterator the underlying document iterators (at least one). The must all return the same
	 * singleton set as {@link DocumentIterator#indices()}.
	 * @throws IOException 
	 */
	protected AbstractOrderedIntervalDocumentIterator( final DocumentIterator[] documentIterator ) throws IOException {
		super( documentIterator );
		if ( soleIndex == null ) throw new IllegalArgumentException();
	}

	protected int nextDocumentInternal() throws IOException {
		while( super.nextDocumentInternal() != -1 && ! intervalIterator().hasNext() );
		return toNextDocument( curr );
	}
	
	public int skipTo( final int n ) throws IOException {
		ahead = false;
		if ( curr >= n ) return curr;

		// We first try to get an aligned set of iterators.
		final int res = super.skipTo( n );
		// If this doesn't work, be bail out.
		if ( res == END_OF_LIST ) return END_OF_LIST;

		// Otherwise, we must manually check that there are consecutive intervals.
		if ( intervalIterator().hasNext() ) return res;
		// If not, we invalidate and check whether there is another possible document.
		nextDocumentInternal();
		return curr;
	}


	/** An abstract interval iterator helping in the implementation of 
	 * {@link ConsecutiveDocumentIterator} and {@link OrderedAndDocumentIterator}
	 * interval iterator member classes. */
	
	protected abstract class AbstractOrderedIntervalIterator extends AbstractCompositeIntervalIterator implements IntervalIterator {
		public AbstractOrderedIntervalIterator() {
			super( n );
		}
	}

	/** An abstract interval iterator helping in the implementation of 
	 * {@link ConsecutiveDocumentIterator} and {@link OrderedAndDocumentIterator}
	 * internal interval iterator member classes. The difference with {@link AbstractOrderedIntervalDocumentIterator}
	 * is that this class assumes that all document iterators are actually index iterators.
	 * The algorithms in this (very common) case can be significantly simplified, obtaining
	 * a large gain in performance. */
	
	protected abstract class AbstractOrderedIndexIntervalIterator extends AbstractCompositeIndexIntervalIterator implements IntervalIterator {
		public AbstractOrderedIndexIntervalIterator() {
			super( n );
		}
	}
}
