package it.unimi.dsi.mg4j.search;

/*		 
 * MG4J: Managing Gigabytes for Java
 *
 * Copyright (C) 2003-2011 Paolo Boldi and Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import it.unimi.dsi.fastutil.ints.IntSet;
import it.unimi.dsi.util.Interval;

import java.io.IOException;
import java.util.Iterator;
                                                                                                                                                    
/** An iterator over {@linkplain Interval intervals}.
 *  Apart for the usual methods of a (type-specific) iterator, it
 *  has a special (optional) {@link #reset()} method that allows one to reset
 *  the iterator: the exact meaning of this operation is decided by the
 *  implementing classes. Typically, after a {@link #reset()}, one can
 *  iterate over a new sequence.
 *  
 *  <p><strong>Warning</strong>: from MG4J 1.2, most methods throw an {@link IOException}
 *  (such exceptions used to be catched and wrapped into a {@link RuntimeException}).
 *
 *  <P>This interface also specifies a method {@link #extent()} returning
 *  a positive integer that is supposed to approximate the minimum possible
 *  length of an interval returned by this iterator. This method returns -1
 *  if this extent cannot be computed.
 */

public interface IntervalIterator extends Iterator<Interval> {

	/** Resets the internal state of this iterator for a new document. 
	 *
	 * <P>To reduce object creation, interval iterators are usually created in a lazy
	 * fashion by document iterator when they are needed. However, this implies that
	 * every time the document iterator is moved, some internal state of the interval iterator must be reset
	 * (e.g., because on the new document some of the component interval iterators are now
	 * {@link IntervalIterators#TRUE}).
	 */
	public void reset() throws IOException;       
	
	/** Returns an approximation of a lower bound for the length of an interval
	 *  returned by this iterator.
 	 *
	 *  @return an approximation of a lower bound for the length of an interval.
	 */
	public int extent();
	
	/** Returns the next interval provided by this interval iterator, or <code>null</code> if no more intervals are available.
	 * 
	 * <p>This method has been reintroduced in MG4J 1.2 with a different semantics.
	 * The special return value <code>null</code> is used to mark the end of iteration. The reason
	 * for this change is providing <em>fully lazy</em> iteration over intervals. Fully lazy iteration
	 * does not provide an <code>hasNext()</code> method&mdash;you have to actually ask for the next
	 * element and check the return value. Fully lazy iteration is much lighter on method calls (half) and
	 * in most (if not all) MG4J classes leads to a much simpler logic. Moreover, {@link #nextInterval()}
	 * can be specified as throwing an {@link IOException}, which avoids the pernicious proliferation
	 * of try/catch blocks in very short, low-level methods (it was having a detectable impact on performance).
	 *
	 * @return the next interval, or <code>null</code> if no more intervals are available.
	 */
	public Interval nextInterval() throws IOException;

	/** Returns the next interval.
	 * 
	 * @deprecated As of MG4J 1.2, the suggested way of iterating over interval iterators
	 * is {@link #nextInterval()}, which has been reintroduced with a fully lazy semantics.
	 * After a couple of releases, however, this annotation will be removed, as it
	 * is very practical to have interval iterators implementing {@link Iterator Iterator&lt;Interval>}. Its
	 * main purpose is to warn people about performance issues solved by {@link #nextInterval()}.
	 * @see #nextInterval()
	 */
	@Deprecated
	public Interval next();	

	/** Provides the set of terms that span the current interval.
	 * 
	 * <p>For each interval returned by MG4J, there is a set of terms that caused the interval to be returned.
	 * The terms appear inside the interval, and certainly at its extremes.
	 * 
	 * <p>Note that the results of this method must be taken with a grain of salt: there might be different sets of terms
	 * causing the current interval, and only one will be returned. 
	 * 
	 * @param terms a set of integers that will be filled with the terms spanning the current interval.
	 */
	
	public void intervalTerms( IntSet terms );
	
}
                                                           