package it.unimi.dsi.mg4j.search;

/*		 
 * MG4J: Managing Gigabytes for Java
 *
 * Copyright (C) 2003-2011 Paolo Boldi and Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import it.unimi.dsi.fastutil.objects.Reference2ReferenceMap;
import it.unimi.dsi.fastutil.objects.Reference2ReferenceMaps;
import it.unimi.dsi.fastutil.objects.ReferenceSet;
import it.unimi.dsi.fastutil.objects.ReferenceSets;
import it.unimi.dsi.mg4j.index.Index;
import it.unimi.dsi.mg4j.search.visitor.DocumentIteratorVisitor;

/** A document iterator returning all documents with interval iterator {@link IntervalIterators#TRUE}. */

public class TrueDocumentIterator extends AbstractDocumentIterator {
	private final ReferenceSet<Index> indices;
	private final Index soleIndex;
	
	protected TrueDocumentIterator( final Index index ) {
		indices = ReferenceSets.singleton( soleIndex = index );
	}

	/** Creates a true document iterator with given index.
	 * 
	 * @param index the index of this iterator.
	 * @return a true document iterator with given index.
	 */
	public static TrueDocumentIterator getInstance( final Index index ) {
		return new TrueDocumentIterator( index );
	}
	
	public IntervalIterator intervalIterator() {
		return IntervalIterators.TRUE;
	}

	public IntervalIterator intervalIterator( Index index ) {
		return index == soleIndex ? IntervalIterators.TRUE : IntervalIterators.FALSE;
	}

	public Reference2ReferenceMap<Index, IntervalIterator> intervalIterators() {
		return Reference2ReferenceMaps.singleton( soleIndex, IntervalIterators.TRUE );
	}

	public ReferenceSet<Index> indices() {
		return indices;
	}

	protected int nextDocumentInternal() {
		if ( curr < soleIndex.numberOfDocuments - 1 ) return ++curr;
		curr = END_OF_LIST;
		return -1;
	}

	public int skipTo( int n ) {
		ahead = false;
		if ( n <= curr ) return curr;
		return curr = n >= soleIndex.numberOfDocuments ? END_OF_LIST : n;
	}

	public <T> T accept( DocumentIteratorVisitor<T> visitor ) {
		if ( !visitor.visitPre( this ) ) return null;
		return visitor.visitPost( this, null );
	}

	public <T> T acceptOnTruePaths( DocumentIteratorVisitor<T> visitor ) {
		if ( !visitor.visitPre( this ) ) return null;
		return visitor.visitPost( this, null );
	}

	public void dispose() {}
}
