package it.unimi.dsi.mg4j.tool;

import static org.junit.Assert.assertEquals;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import it.unimi.dsi.logging.ProgressLogger;
import it.unimi.dsi.mg4j.document.CompositeDocumentSequence;
import it.unimi.dsi.mg4j.document.DocumentSequence;
import it.unimi.dsi.mg4j.document.StringArrayDocumentCollection;
import it.unimi.dsi.mg4j.index.DiskBasedIndex;
import it.unimi.dsi.mg4j.index.Index;
import it.unimi.dsi.mg4j.index.IndexIterator;
import it.unimi.dsi.mg4j.index.IndexReader;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationTargetException;
import java.net.URISyntaxException;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;


public class PrecomputeAlignmentsTest {
	
	private static String basename;

	@BeforeClass
	public static void setUp() throws IOException, ConfigurationException, SecurityException, URISyntaxException, ClassNotFoundException, InstantiationException, IllegalAccessException, InvocationTargetException, NoSuchMethodException {
		basename = File.createTempFile( PrecomputeAlignmentsTest.class.getSimpleName(), "test" ).getCanonicalPath();

		new IndexBuilder( basename, new CompositeDocumentSequence(
				new DocumentSequence[] {
						new StringArrayDocumentCollection( "a b a c", "d b d d", "e f g", "h" ),
						new StringArrayDocumentCollection( "Y X Y Z", "X Y X Z", "Z Z Z", "" )
				},
				new String[] {
						"alignee",
						"aligner"
				}
			) ).run();
	}

	@AfterClass
	public static void tearDown() {
		for ( Object f : FileUtils.listFiles( new File( basename ).getParentFile(), FileFilterUtils.prefixFileFilter( PrecomputeAlignments.class.getSimpleName() ), null ) )
			( (File)f ).delete();
	}

	@Test
	public void testAlignment() throws ConfigurationException, SecurityException, IOException, URISyntaxException, ClassNotFoundException, InstantiationException, IllegalAccessException, InvocationTargetException, NoSuchMethodException {
		new PrecomputeAlignments( basename, basename + "-alignee", basename + "-aligner", null, ProgressLogger.TEN_SECONDS ).run();
		final Index index = Index.getInstance( basename );
		IndexReader indexReader = index.getReader();
		assertEquals( 4, index.numberOfDocuments ); // There's a fake document.
		IndexIterator documents;
		
		documents = indexReader.documents( 0 ); // a
		assertEquals( 1, documents.nextDocument() ); // X
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );

		documents = indexReader.documents( 1 ); // b
		assertEquals( 0, documents.nextDocument() ); // X
		assertEquals( 1, documents.nextDocument() ); // Y
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );

		documents = indexReader.documents( 2 ); // c
		assertEquals( 2, documents.nextDocument() ); // Z
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );
	
		documents = indexReader.documents( 3 ); // d
		assertEquals( 0, documents.nextDocument() ); // X
		assertEquals( 2, documents.nextDocument() ); // Z
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );

		documents = indexReader.documents( 4 ); // e
		assertEquals( 2, documents.nextDocument() ); // Z
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );

		documents = indexReader.documents( 5 ); // f
		assertEquals( 2, documents.nextDocument() ); // Z
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );

		documents = indexReader.documents( 6 ); // g
		assertEquals( 2, documents.nextDocument() ); // Z
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );

		documents = indexReader.documents( 7 ); // h
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );
	}

	@Test
	public void testSubset() throws ConfigurationException, SecurityException, IOException, URISyntaxException, ClassNotFoundException, InstantiationException, IllegalAccessException, InvocationTargetException, NoSuchMethodException {
		final ObjectArrayList<String> terms = new ObjectArrayList<String>( new String[] { "a", "c", "e", "w" } ); // We even accept terms not in the index.
		new PrecomputeAlignments( basename, basename + "-alignee", basename + "-aligner", terms, ProgressLogger.TEN_SECONDS ).run();
		final Index index = Index.getInstance( basename );
		IndexReader indexReader = index.getReader();
		assertEquals( 4, index.numberOfDocuments ); // There's a fake document.
		assertEquals( 4, index.numberOfTerms );
		IndexIterator documents;

		assertEquals( terms, IOUtils.readLines( new InputStreamReader( new FileInputStream( basename + DiskBasedIndex.TERMS_EXTENSION ), "UTF-8" ) ) );

		documents = indexReader.documents( 0 ); // a
		assertEquals( 1, documents.nextDocument() ); // X
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );

		documents = indexReader.documents( 1 ); // c
		assertEquals( 2, documents.nextDocument() ); // Z
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );
	
		documents = indexReader.documents( 2 ); // e
		assertEquals( 2, documents.nextDocument() ); // Z
		assertEquals( 3, documents.nextDocument() ); // Fake
		assertEquals( -1, documents.nextDocument() );
	}
}
