/*
 * Copyright (c) 2008, 2023 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */

// Contributors:
//     Linda DeMichiel - 2.1
//     Linda DeMichiel - 2.0

package jakarta.persistence;

import java.lang.annotation.Target;
import java.lang.annotation.Retention;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Specifies that a unique constraint is to be included in the
 * generated DDL for a primary or secondary table.
 *
 * <p>Example:
 * {@snippet :
 * @Entity
 * @Table(
 *     name = "EMPLOYEE",
 *     uniqueConstraints = @UniqueConstraint(columnNames = {"EMP_ID", "EMP_NAME"})
 * )
 * public class Employee { ... }
 * }
 *
 * @since 1.0
 */
@Target({}) 
@Retention(RUNTIME)
public @interface UniqueConstraint {

    /**
     * (Optional) The name of the constraint.
     * <p> Defaults to a provider-generated name.
     *
     * @since 2.0
     */
    String name() default "";

    /**
     * (Required) The names of the column which make up the
     * constraint.
     */
    String[] columnNames();

    /**
     * (Optional) A SQL fragment appended to the generated DDL
     * which creates this constraint.
     *
     * @since 3.2
     */
    String options() default "";
}
