/**
 *  Copyright (c) 2011 Terracotta, Inc.
 *  Copyright (c) 2011 Oracle and/or its affiliates.
 *
 *  All rights reserved. Use is subject to license terms.
 */


package javax.cache.event;

/**
 * This enumeration defines valid values for {@link CacheEntryListener} notification scope.
 *
 * A distributed cache may contain multiple CacheManager nodes. Events on a cache instance inside
 * in the same JVM are considered local, other events are not.
 * <p/>
 * A cache will only receive remote events for if it registers a {@link CacheEntryListener} with a remote notification scope.
 * Only those cluster members interested in receiving remote events should register such a listener.
 * <p/>
 * There can be a large cost difference between receipt of local versus remote events thus allowing the choice.
 * Accordingly, the remote notification scope is optional. This capability can be determined using {@link javax.cache.Caching#isSupported(javax.cache.OptionalFeature)}.
 * <p/>
 * @since 1.0
 */
public enum NotificationScope {

    /**
     * Receive only events generated locally.
     */
    LOCAL(true, false),

    /**
     * Receive only events generated by another JVM
     */
    REMOTE(false, true);

    private final boolean deliverLocal;
    private final boolean deliverRemote;

    private NotificationScope(boolean deliverLocal, boolean deliverRemote) {
        this.deliverLocal = deliverLocal;
        this.deliverRemote = deliverRemote;
    }

    /**
     * Determine whether an event should be delivered under this notification scope.
     *
     * @param isRemote Whether the event is local or remote
     * @return true if the event should be delivered to a listener with this notification scope
     */
    public boolean shouldDeliver(boolean isRemote) {
        return (isRemote && deliverRemote) || (!isRemote && deliverLocal);
    }
}
