/**
 *  Copyright (c) 2011-2013 Terracotta, Inc.
 *  Copyright (c) 2011-2013 Oracle and/or its affiliates.
 *
 *  All rights reserved. Use is subject to license terms.
 */

package javax.cache.integration;

import javax.cache.Cache;
import java.util.Map;

/**
 * Used when a cache is read-through or when loading data into a cache via the
 * {@link javax.cache.Cache#loadAll(java.util.Set, boolean,
 * CompletionListener)} method.
 * <p/>
 * @param <K> the type of keys handled by this loader
 * @param <V> the type of values generated by this loader
 * @author Greg Luck
 * @author Yannis Cosmadopoulos
 * @since 1.0
 * @see javax.cache.configuration.Configuration#isReadThrough()
 * @see CacheWriter
 */
public interface CacheLoader<K, V> {

  /**
   * Loads an object. Application developers should implement this
   * method to customize the loading of cache object. This method is called
   * by the caching service when the requested object is not in the cache. If
   * the object can't be loaded <code>null</code> should be returned.
   *
   * @param key the key identifying the object being loaded
   * @return The entry for the object that is to be stored in the cache or
   *         <code>null</code> if the object can't be loaded
   */
  Cache.Entry<K, V> load(K key);

  /**
   * Loads multiple objects. Application developers should implement this
   * method to customize the loading of cache object. This method is called
   * by the caching service when the requested object is not in the cache. If
   * an object can't be loaded, it is not returned in the resulting map.
   * <p/>
   *
   * @param keys keys identifying the values to be loaded
   * @return A map of key, values to be stored in the cache.
   */
  Map<K, V> loadAll(Iterable<? extends K> keys);
}
