package javax.slee.management;

import javax.management.ObjectName;

/**
 * This class identifies a notification such as an {@link AlarmNotification alarm} or
 * {@link TraceNotification trace} notification as being generated from some internal
 * subsystem in the SLEE.
 * @since SLEE 1.1
 */
public final class SubsystemNotification extends AbstractNotificationSource implements NotificationSource {
    /**
     * The JMX notification type of alarm notifications that are generated in response
     * to an internal component or subsystem of the SLEE interacting with the
     * {@link javax.slee.facilities.AlarmFacility} (or a similar internal interface).
     * <p>
     * The notification type is equal to the string "javax.slee.management.alarm.subsystem".
     */
    public static final String ALARM_NOTIFICATION_TYPE = "javax.slee.management.alarm.subsystem";

    /**
     * The JMX notification type of trace notifications that are generated in response
     * to an internal component or subsystem of the SLEE interacting with a
     * {@link javax.slee.facilities.Tracer} object (or a similar internal interface).
     * <p>
     * The notification type is equal to the string "javax.slee.management.trace.subsystem".
     */
    public static final String TRACE_NOTIFICATION_TYPE = "javax.slee.management.trace.subsystem";

    /**
     * The JMX notification type of usage notifications that are generated by a
     * {@link javax.slee.usage.UsageMBean} containing a <code>SubsystemNotification</code>
     * as a notification source.
     * <p>
     * The notification type is equal to the string "javax.slee.management.usage.subsystem".
     */
    public static final String USAGE_NOTIFICATION_TYPE = "javax.slee.management.usage.subsystem";

    /**
     * The JMX Object Name property key that identifies the name of the internal subsystem
     * in a Usage MBean whose {@link javax.slee.usage.UsageMBean#NOTIFICATION_SOURCE_KEY}
     * property has a value equal to {@link #USAGE_NOTIFICATION_TYPE}.  This key is
     * equal to the string "subsystemName".
     * @see javax.slee.usage.UsageMBean#BASE_OBJECT_NAME
     * @see javax.slee.usage.UsageMBean#NOTIFICATION_SOURCE_KEY
     * @since SLEE 1.1
     */
    public static final String SUBSYSTEM_NAME_KEY = "subsystemName";


    /**
     * Create a new <code>SubsystemNotification</code> object that uniquely identifies an
     * internal component or subsystem of the SLEE.
     * @param subsystemName the name of the SLEE internal component or subsystem.  Subsystem
     *        names may be any vendor-defined string.
     * @throws NullPointerException if <code>subsystemName</code> is <code>null</code>.
     */
    public SubsystemNotification(String subsystemName) {
        if (subsystemName == null) throw new NullPointerException("entityName is null");
        this.subsystemName = subsystemName;
    }

    /**
     * Get the name of the internal component or subsystem of the SLEE identified by this
     * notification source.
     * @return the name of the internal component or subsystem.
     */
    public String getSubsystemName() {
        return subsystemName;
    }

    /**
     * Get the JMX notification type of alarm notifications generated in response
     * to a SLEE internal component or subsystem interacting with the Alarm Facility.
     * @return the string defined by {@link #ALARM_NOTIFICATION_TYPE}.
     */
    public String getAlarmNotificationType() {
        return ALARM_NOTIFICATION_TYPE;
    }

    /**
     * Get the JMX notification type of trace notifications generated in response
     * to a SLEE internal component or subsystem interacting with the Trace Facility.
     * @return the string defined by {@link #TRACE_NOTIFICATION_TYPE}.
     */
    public String getTraceNotificationType() {
        return TRACE_NOTIFICATION_TYPE;
    }

    /**
     * Get the JMX notification type of usage notifications generated in response
     * to a SLEE internal component or subsystem interacting with its usage parameters.
     * @return the string defined by {@link #USAGE_NOTIFICATION_TYPE}.
     */
    public String getUsageNotificationType() {
        return USAGE_NOTIFICATION_TYPE;
    }

    /**
     * Get a JMX Object Name property string that uniquely identifies the specified SLEE
     * internal component or subsystem, suitable for inclusion in the Object Name of a Usage
     * MBean. This method makes use of the {@link javax.management.ObjectName#quote}</code>
     * method to ensure that the name is valid for inclusion as property values in an Object
     * Name.
     * <p>
     * This method can be used as follows to manually construct a complete Object Name
     * for a Usage MBean:
     * <br><ul><code>
     *     ObjectName name = new ObjectName(<br>
     *     &nbsp;&nbsp;&nbsp;&nbsp;{@link javax.slee.usage.UsageMBean#BASE_OBJECT_NAME} + "," +<br>
     *     &nbsp;&nbsp;&nbsp;&nbsp;{@link javax.slee.usage.UsageMBean#USAGE_PARAMETER_SET_NAME_KEY} + "=" + ObjectName.quote(paramSetName) + "," &nbsp; // optional<br>
     *     &nbsp;&nbsp;&nbsp;&nbsp;{@link javax.slee.usage.UsageMBean#NOTIFICATION_SOURCE_KEY} + "=" + {@link #USAGE_NOTIFICATION_TYPE SubsystemNotification.USAGE_NOTIFICATION_TYPE} + "," +<br>
     *     &nbsp;&nbsp;&nbsp;&nbsp;{@link #getUsageMBeanProperties(String) SubsystemNotification.getUsageMBeanProperties(subsystemName)}<br>
     *     );
     * </code></ul>
     * @param subsystemName the name of the SLEE internal component or subsystem.
     * @return an Object Name property string that uniquely identifies the specified
     *        internal component or subsystem.
     * @throws NullPointerException if <code>subsystem</code> is <code>null</code>.
     */
    public static String getUsageMBeanProperties(String subsystemName) {
        if (subsystemName == null) throw new NullPointerException("subsystemName is null");
        return SUBSYSTEM_NAME_KEY + '=' + ObjectName.quote(subsystemName);
    }

    /**
     * Get a JMX Object Name property string that uniquely identifies the SLEE internal
     * component or subsystem of this notification source, suitable for inclusion in
     * the Object Name of a Usage MBean.
     * <p>
     * This method is equivalent to {@link #getUsageMBeanProperties(String) getUsageMBeanProperties(getSubsystemName())}.
     * @return an Object Name property string that uniquely identifies the
     *        internal component or subsystem.
     * @return an Object Name property string that uniquely identifies the SLEE
     *        internal component or subsystem of this notification source.
     */
    public String getUsageMBeanProperties() {
        return getUsageMBeanProperties(subsystemName);
    }

    /**
     * Compare this notification source for equality with another object.
     * @param obj the object to compare this with.
     * @return <code>true</code> if <code>obj</code> is an instance of this class and
     *        contains the subsystem name as this, <code>false</code> otherwise.
     */
    public boolean equals(Object obj) {
        if (obj == this) return true;
        if (!(obj instanceof SubsystemNotification)) return false;

        return this.subsystemName.equals(((SubsystemNotification)obj).subsystemName);
    }

    /**
     * Get a hash code value for this notification source.
     * @return a hash code value for this notification source.
     */
    public int hashCode() {
        return subsystemName.hashCode();
    }

    /**
     * Get a string representation for this notification source.
     * @return a string representation for this notification source.
     * @see Object#toString()
     */
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("SubsystemNotification[subsystem=").append(subsystemName).append(']');
        return buf.toString();
    }

    /**
     * Compare this notification source with the specified object for order.
     * Returns a negative integer, zero, or a positive integer if this object
     * is less than, equal to, or greater than the specified object.
     * <p>
     * If <code>obj</code> is a <code>SubsystemNotification</code>, order is
     * determined by comparing the encapsulated subsystem name.  Otherwise, if
     * <code>obj</code> is a <code>NotificationSource</code>, ordering is determined
     * by comparing the class name of this class with the class name of <code>obj</code>.
     * <p>
     * @param obj the object to compare this with.
     * @return a negative integer, zero, or a positive integer if this notification
     *        source is considered less than, equal to, or greater than the
     *        specified object.
     * @throws ClassCastException if <code>obj</code> does not implement the
     *        {@link NotificationSource} interface.
     * @see Comparable#compareTo(Object)
     */
    public int compareTo(Object obj) {
        // can't compare with null
        if (obj == null) throw new NullPointerException("obj is null");
        if (obj == this) return 0;

        if (obj instanceof SubsystemNotification) {
            // compare the profile table name
            SubsystemNotification that = (SubsystemNotification)obj;
            return this.subsystemName.compareTo(that.subsystemName);
        }
        else {
            return super.compareTo(TYPE, obj);
        }
    }


    // protected

    protected String getClassName() {
        return TYPE;
    }


    private final String subsystemName;

    // constant to avoid expensive getClass() invocations at runtime
    private static final String TYPE = SubsystemNotification.class.getName();
}
