/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.metadata.spi.retrieval;

import java.lang.annotation.Annotation;

import org.jboss.metadata.spi.MetaData;

/**
 * MetaDataRetrievalToMetaDataBridge.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 46146 $
 */
public class MetaDataRetrievalToMetaDataBridge implements MetaData
{
   /** The meta data retrieval method */
   private MetaDataRetrieval retrieval;
   
   public MetaDataRetrievalToMetaDataBridge(MetaDataRetrieval retrieval)
   {
      if (retrieval == null)
         throw new IllegalArgumentException("Null retrieval");
      this.retrieval = retrieval;
   }

   public long getValidTime()
   {
      ValidTime validTime = retrieval.getValidTime();
      return validTime.getValidTime();
   }

   public <T extends Annotation> T getAnnotation(Class<T> annotationType)
   {
      if (annotationType == null)
         throw new IllegalArgumentException("Null annotationType");
      AnnotationItem<T> item = retrieval.retrieveAnnotation(annotationType);
      if (item == null)
         return null;
      return item.getValue();
   }

   public Annotation[] getAnnotations()
   {
      AnnotationsItem item = retrieval.retrieveAnnotations();
      if (item == null)
         return NO_ANNOTATIONS;
      return item.getValue();
   }

   public Annotation[] getLocalAnnotations()
   {
      AnnotationsItem item = retrieval.retrieveLocalAnnotations();
      if (item == null)
         return NO_ANNOTATIONS;
      return item.getValue();
   }

   public Object[] getMetaData()
   {
      MetaDatasItem item = retrieval.retrieveMetaData();
      if (item == null)
         return NO_METADATA;
      return item.getValue();
   }

   public Object[] getLocalMetaData()
   {
      MetaDatasItem item = retrieval.retrieveLocalMetaData();
      if (item == null)
         return NO_METADATA;
      return item.getValue();
   }

   public <T> T getMetaData(Class<T> type)
   {
      if (type == null)
         throw new IllegalArgumentException("Null type");
      MetaDataItem<T> item = retrieval.retrieveMetaData(type);
      if (item == null)
         return null;
      return item.getValue();
   }

   public <T> T getMetaData(String name, Class<T> type)
   {
      if (name == null)
         throw new IllegalArgumentException("Null name");
      if (type == null)
         throw new IllegalArgumentException("Null type");
      MetaDataItem item = retrieval.retrieveMetaData(name);
      if (item == null)
         return null;
      return type.cast(item.getValue());
   }

   public Object getMetaData(String name)
   {
      if (name == null)
         throw new IllegalArgumentException("Null name");
      MetaDataItem item = retrieval.retrieveMetaData(name);
      if (item == null)
         return null;
      return item.getValue();
   }

   public boolean isAnnotationPresent(Class<? extends Annotation> annotationType)
   {
      return getAnnotation(annotationType) != null;
   }

   public boolean isMetaDataPresent(Class<?> type)
   {
      return getMetaData(type) != null;
   }

   public boolean isMetaDataPresent(String name, Class<?> type)
   {
      if (type == null)
         throw new IllegalArgumentException("Null type");
       Object value = getMetaData(name);
       if (value == null)
          return false;
       return type.isInstance(value);
   }

   public boolean isMetaDataPresent(String name)
   {
      return getMetaData(name) != null;
   }

   /**
    * Get the meta data retrieval
    * 
    * @return the retrieval
    */
   protected MetaDataRetrieval getMetaDataRetrieval()
   {
      return retrieval;
   }
}
