package org.jboss.maven.plugins.deploy;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.channels.FileChannel;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.License;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;

/**
 * Maven plugin for deploying to jboss repository.  By default
 * it will bind to the deploy phase.
 * 
 * @phase deploy
 * @goal jboss-deploy
 * 
 */
public class JBossDeployMojo extends AbstractMojo
{

   private static String fileSep = System.getProperty("file.separator");

   /**
    * The Maven Project Object
    *
    * @parameter expression="${project}"
    * @required
    * @readonly
    */
   protected MavenProject project;

   /**
    * Source control information.
    * @parameter expression="${project.scm}"
    * @readonly
    */
   private org.apache.maven.model.Scm scm;

   /**
    * License information.
    * @parameter expression="${project.licenses}"
    * @required
    * @readonly
    */
   private List licenses;

   //private org.apache.maven.model.License license;

   /**
    * The Maven Plugin Object
    *
    * @parameter expression="${plugin.components}"
    * @required
    * @readonly
    */
   protected List pluginComponents;

   /**
    * Maven project properties.
    *
    * @parameter expression="${project.properties}"
    * @required
    * @readonly
    */
   protected Properties projectProperties;

   /**
    * The plugin dependencies.
    *
    * @parameter expression="${plugin.artifacts}"
    * @required
    * @readonly
    */
   private List pluginArtifacts;

   /**
    * The directory for compiled classes.
    *
    * @parameter expression="${project.build.directory}"
    * @required
    * @readonly
    */
   private File targetDirectory;

   /**
    * Project artifacts.
    *
    * @parameter expression="${project.artifact}"
    * @required
    * @readonly
    * @todo this is an export variable, really
    */
   private Artifact projectArtifact;

   /**
    * This is the location that the jboss repository (non-maven) files will be copied to.
    * @parameter
    * @required
    */
   private String jbossDeployRoot;

   /**
    * This is the map of components that this project depends on.
    * @parameter
    */
   private Map imports;

   /**
    * This is the list of artifacts to be added to the classpath.
    * @parameter
    */
   private List exports;

   /**
    * Main execution path of the plugin.  Generates component-info.xml, and copies jar files to repository location.
    */
   public void execute() throws MojoExecutionException
   {
      this.getLog().debug("Executing JBoss deploy plugin");

      // Check that the project is using jar packaging because we don't handle other types yet.
      if (!project.getPackaging().equals("jar"))
      {
         getLog().info("Ignoring project with packaging type: " + project.getPackaging());
         return;
      }

      // Initialize local variables
      String artifactFileName = project.getArtifactId() + "." + project.getPackaging();
      String artifactFilePath = targetDirectory.toString() + fileSep + artifactFileName;
      String artifactSourcesFilePath = targetDirectory.toString() + fileSep + project.getArtifactId() + "-sources"
            + "." + project.getPackaging();
      String artifactJavadocFilePath = targetDirectory.toString() + fileSep + project.getArtifactId() + "-javadoc"
            + "." + project.getPackaging();
      String artifactTestsFilePath = targetDirectory.toString() + fileSep + project.getArtifactId() + "-tests" + "."
            + project.getPackaging();
      String deployToPath = jbossDeployRoot + fileSep + project.getGroupId() + fileSep + project.getArtifactId()
            + fileSep + project.getVersion() + fileSep;

      // Load template file
      InputStream templateInputStream = this.getClass().getResourceAsStream("component-info-template.txt");

      if (templateInputStream == null)
      {
         this.getLog().error("Unable to load template file");
         return;
      }
      StringBuffer compInfoTemplate = new StringBuffer(1000);
      try
      {
         BufferedReader reader = new BufferedReader(new InputStreamReader(templateInputStream));
         char[] buf = new char[1000];
         int numRead = 0;
         while ((numRead = reader.read(buf)) != -1)
         {
            compInfoTemplate.append(String.valueOf(buf, 0, numRead));
         }
         reader.close();
      }
      catch (IOException ioe)
      {
         this.getLog().error("Unable to load component info template.");
         this.getLog().error(ioe.toString());
         throw new MojoExecutionException(ioe.getMessage());
      }

      // Evaluate the place holders in the template
      String componentId = project.getGroupId() + "/" + project.getArtifactId();
      evaluateVariable(compInfoTemplate, "project.name", componentId + "-component-info");
      evaluateVariable(compInfoTemplate, "component.id", componentId);
      evaluateVariable(compInfoTemplate, "project.version", project.getVersion());
      if (licenses.size() == 0)
      {
         getLog().warn("No license specified in pom.xml.  Default lgpl license will be used.");
         evaluateVariable(compInfoTemplate, "project.license", "lgpl");
      }
      else
      {
         License firstLicense = (License) licenses.get(0);
         evaluateVariable(compInfoTemplate, "project.license", firstLicense.getName());
      }
      if (project.getDescription() == null || project.getDescription().equals(""))
      {
         project.setDescription(project.getId());
      }
      evaluateVariable(compInfoTemplate, "project.description", project.getDescription());
      if (scm == null || scm.getConnection() == null)
      {
         evaluateVariable(compInfoTemplate, "project.scm", "");
      }
      else
      {
         String scmConnStr = scm.getConnection().replaceFirst("scm:", "");
         evaluateVariable(compInfoTemplate, "project.scm", "scm=\"" + scmConnStr + "\"");
      }

      // Create and set list of artifacts
      StringBuffer artifacts = new StringBuffer();
      artifacts.append("    <artifact id=\"" + artifactFileName + "\"/>\n");
      if (new File(artifactTestsFilePath).exists())
      {
         artifacts.append("    <artifact id=\"" + project.getArtifactId() + "-tests." + project.getPackaging()
               + "\"/>\n");
      }
      if (new File(artifactSourcesFilePath).exists())
      {
         artifacts.append("    <artifact id=\"" + project.getArtifactId() + "-sources." + project.getPackaging()
               + "\"/>\n");
      }
      if (new File(artifactJavadocFilePath).exists())
      {
         artifacts.append("    <artifact id=\"" + project.getArtifactId() + "-javadoc." + project.getPackaging()
               + "\"/>\n");
      }
      evaluateVariable(compInfoTemplate, "artifacts", artifacts.toString());

      // Create and set list of includes
      StringBuffer exportsString = new StringBuffer();
      exportsString.append("      <include input=\"" + artifactFileName + "\"/>\n");
      if (exports!=null) {
         for(Object export : exports) {
            exportsString.append("      <include input=\"" + export + "\"/>\n");
         }
      }
      evaluateVariable(compInfoTemplate, "includes", exportsString.toString());

      // Get list of imports
      //TODO: Maybe this can be determined based on the dependencies, but for now we just have to set it 
      //      in the plugin configuration 
      //      List dependencies = project.getDependencies();
      if (imports == null)
      {
         evaluateVariable(compInfoTemplate, "imports", "");
      }
      else
      {
         StringBuffer importsString = new StringBuffer();
         Set componentNames = imports.keySet();
         for (Object component : componentNames)
         {
            importsString.append("    <import componentref=\"" + component + "\">\n");
            importsString.append("      <compatible version=\"" + imports.get(component) + "\"/>\n");
            importsString.append("    </import>\n");
         }
         evaluateVariable(compInfoTemplate, "imports", importsString.toString());
      }
      
      // Write the component info file
      try
      {
         File deployToDir = new File(deployToPath);
         if (!deployToDir.exists())
         {
            deployToDir.mkdirs();
         }
         File compInfoFile = new File(deployToPath + "component-info.xml");
         FileWriter fw = new FileWriter(compInfoFile);
         fw.write(compInfoTemplate.toString());
         fw.close();
      }
      catch (IOException ioe)
      {
         ioe.printStackTrace();
      }

      // Deploy lib directory.
      getLog().debug("Copying from: " + artifactFilePath);
      getLog().info("Deploying to: " + deployToPath);
      String deployToPathLib = deployToPath + "lib" + fileSep;
      try
      {
         copyFileToDir(artifactFilePath, deployToPathLib);
         copyFileToDir(artifactTestsFilePath, deployToPathLib);
         copyFileToDir(artifactSourcesFilePath, deployToPathLib);
         copyFileToDir(artifactJavadocFilePath, deployToPathLib);
      }
      catch (IOException ioe)
      {
         throw new MojoExecutionException("Problem copying artifacts: " + ioe.getMessage());
      }
   }

   /**
    * Copy a file from the source path to the destination.
    * 
    * @param src The file to copy.  If the source file does not exist, the method does nothing.
    * @param dest The directory to copy the file.  If the dest path does not exist,
    *     any necessary folders will be created.
    */
   public void copyFileToDir(String src, String dest) throws IOException
   {
      File srcFile = new File(src);
      if (!srcFile.exists())
      {
         return;
      }
      File destDir = new File(dest);
      if (!destDir.exists())
      {
         destDir.mkdirs();
      }
      File destFile = new File(destDir.getCanonicalPath() + fileSep + srcFile.getName());
      FileChannel srcChannel = new FileInputStream(srcFile).getChannel();
      FileChannel dstChannel = new FileOutputStream(destFile).getChannel();
      dstChannel.transferFrom(srcChannel, 0, srcChannel.size());
      srcChannel.close();
      dstChannel.close();
      this.getLog().debug("Copied to file: " + destFile.getCanonicalPath());
   }

   /**
    * Replace all instances of var with value in the given buffer.
    * For example if var is "project.name", this method will search
    * the buffer for "${project.name}" and replace it with the value
    * provided.
    * 
    * @param buffer The buffer to be modified
    * @param var The name of the variable to be evaluated.
    * @param value The replacement string.
    */
   public void evaluateVariable(StringBuffer buffer, String var, String value)
   {
      int start = 0;
      while ((start = buffer.indexOf("${" + var + "}")) != -1)
      {
         int end = start + ("${" + var + "}").length();
         buffer.replace(start, end, value);
      }
   }

}
