package org.jboss.maven.plugins.deploy;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.License;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.xml.sax.SAXException;

/**
 * Maven plugin for deploying to jboss repository.  By default
 * it will bind to the deploy phase.
 * 
 * @phase deploy
 * @goal jboss-deploy
 * 
 */
public class JBossDeployMojo extends AbstractMojo
{

   private static String fileSep = System.getProperty("file.separator");

   /**
    * The Maven Project Object
    *
    * @parameter expression="${project}"
    * @required
    * @readonly
    */
   protected MavenProject project;

   /**
    * The Maven Project Helper Object
    *
    * @component
    * @required
    */
   protected org.apache.maven.project.MavenProjectHelper projectHelper;

   /**
    * Source control information.
    * @parameter expression="${project.scm}"
    * @readonly
    */
   private org.apache.maven.model.Scm scm;

   /**
    * License information.
    * @parameter expression="${project.licenses}"
    * @required
    * @readonly
    */
   private List licenses;

   //private org.apache.maven.model.License license;

   /**
    * The Maven Plugin Object
    *
    * @parameter expression="${plugin.components}"
    * @required
    * @readonly
    */
   protected List pluginComponents;

   /**
    * Maven project properties.
    *
    * @parameter expression="${project.properties}"
    * @required
    * @readonly
    */
   protected Properties projectProperties;

   /**
    * This is the location that the jboss repository (non-maven) files will be copied to.
    * @parameter
    * @required
    */
   private String jbossDeployRoot;

   /**
    * If set to true, this will tell the plugin to generate a listing of imports based on the
    * project dependencies.  Defaults to false;
    * @parameter
    */
   private boolean generateImports;

   /**
    * This is the map of components that this project depends on.  If generateImports is set
    * to true, the generated imports will be appended to the imports specified here.
    * @parameter
    */
   private Map imports;

   /**
    * This is the list of artifacts to be added to the classpath.
    * @parameter
    */
   private List exports;

   /**
    * Main execution path of the plugin.  Generates component-info.xml, and copies jar files to repository location.
    */
   public void execute() throws MojoExecutionException
   {
      this.getLog().debug("Executing JBoss deploy plugin");

      // Check that there is at least one artifact to deploy
      if ((project.getArtifact() == null || project.getArtifact().getFile() == null)
            && project.getAttachedArtifacts().size() == 0)
      {
         this.getLog().info("No artifacts to deploy for project:" + project.getName());
         return;
      }

      // Initialize some local variables
      File deployToDir = new File(jbossDeployRoot + fileSep + project.getGroupId().replace('.', '/') + fileSep
            + project.getArtifactId() + fileSep + project.getVersion() + fileSep);
      File deployToLibDir = new File(deployToDir.getAbsolutePath() + fileSep + "lib" + fileSep);

      // Load template file
      InputStream templateInputStream = this.getClass().getResourceAsStream("component-info-template.txt");

      if (templateInputStream == null)
      {
         this.getLog().error("Unable to load template file");
         return;
      }

      StringBuffer compInfoTemplate = new StringBuffer();

      try
      {
         compInfoTemplate.append(IOUtils.toString(templateInputStream));
      }
      catch (IOException ioe)
      {
         this.getLog().error("Unable to load component info template.");
         this.getLog().error(ioe.toString());
         throw new MojoExecutionException(ioe.getMessage());
      }

      // Evaluate the place holders in the template
      String componentId = project.getGroupId() + "/" + project.getArtifactId();
      evaluateVariable(compInfoTemplate, "project.name", componentId + "-component-info");
      evaluateVariable(compInfoTemplate, "component.id", componentId);
      evaluateVariable(compInfoTemplate, "project.version", project.getVersion());
      if (licenses.size() == 0)
      {
         getLog().warn("No license specified in pom.xml.  Default lgpl license will be used.");
         evaluateVariable(compInfoTemplate, "project.license", "lgpl");
      }
      else
      {
         License firstLicense = (License) licenses.get(0);
         evaluateVariable(compInfoTemplate, "project.license", firstLicense.getName());
      }
      if (project.getDescription() == null || project.getDescription().equals(""))
      {
         project.setDescription(project.getId());
      }
      evaluateVariable(compInfoTemplate, "project.description", project.getDescription());

      if (scm == null || scm.getConnection() == null)
      {
         evaluateVariable(compInfoTemplate, "project.scm", "");
      }
      else
      {
         String scmConnStr = scm.getConnection().replaceFirst("scm:", "");
         evaluateVariable(compInfoTemplate, "project.scm", "scm=\"" + scmConnStr + "\"");
      }

      // Get the list of artifacts, copy them to the lib directory, and add them to component-info
      HashSet<String> artifactSet = new HashSet<String>();
      Artifact projectArtifact = project.getArtifact();
      if (projectArtifact != null && projectArtifact.getFile() != null)
      {
         artifactSet.add("    <artifact id=\"" + projectArtifact.getFile().getName() + "\"/>\n");
         try
         {
            this.getLog().debug("Deploying file://" + deployToLibDir + fileSep + projectArtifact.getFile().getName());
            FileUtils.copyFileToDirectory(projectArtifact.getFile(), deployToLibDir);
         }
         catch (IOException ioe)
         {
            throw new MojoExecutionException("Problem copying artifact: " + ioe.getMessage());
         }
      }

      List list = project.getAttachedArtifacts();
      this.getLog().debug("Number of attached artifacts: " + list.size());
      for (Object obj : list)
      {
         Artifact artifact = (Artifact) obj;
         this.getLog().debug(artifact.getArtifactId());
         String artifactLine = "    <artifact id=\"" + artifact.getArtifactId() + "-" + artifact.getClassifier()
               + ".jar" + "\"/>\n";
         if (artifactSet.contains(artifactLine))
         {
            this.getLog().warn(
                  "Duplicate attached artifact found: " + artifact.getArtifactId() + "-" + artifact.getClassifier()
                        + ".jar");
         }
         artifactSet.add(artifactLine);
         try
         {
            this.getLog().debug("Deploying file://" + deployToLibDir + fileSep + artifact.getFile().getName());
            FileUtils.copyFileToDirectory(artifact.getFile(), deployToLibDir);
         }
         catch (IOException ioe)
         {
            throw new MojoExecutionException("Problem copying artifact: " + ioe.getMessage());
         }
      }

      StringBuffer artifacts = new StringBuffer();
      for (String line : artifactSet)
      {
         artifacts.append(line);
      }
      evaluateVariable(compInfoTemplate, "artifacts", artifacts.toString());

      // Create and set list of includes
      StringBuffer exportsString = new StringBuffer();
      if (project.getArtifact().getFile() != null)
      {
         exportsString.append("      <include input=\"" + project.getArtifact().getFile().getName() + "\"/>\n");
      }
      if (exports != null)
      {
         for (Object export : exports)
         {
            exportsString.append("      <include input=\"" + export + "\"/>\n");
         }
      }
      evaluateVariable(compInfoTemplate, "includes", exportsString.toString());

      // Generate list of imports
      StringBuffer importsString = new StringBuffer();
      if (imports != null)
      {
         Set componentNames = imports.keySet();
         for (Object component : componentNames)
         {
            importsString.append("    <import componentref=\"" + component.toString().replace('.', '/') + "\">\n");
            importsString.append("      <compatible version=\"" + imports.get(component) + "\"/>\n");
            importsString.append("    </import>\n");
         }
      }
      if (this.generateImports)
      {
         List dependencies = project.getDependencies();
         for (Object obj : dependencies)
         {
            Dependency dep = (Dependency) obj;
            if (dep.getScope() == null || !dep.getScope().equals("test")) {
               importsString.append("    <import componentref=\"" + dep.getGroupId().replace('.', '/') + "/"
                     + dep.getArtifactId() + "\">\n");
               importsString.append("      <compatible version=\"" + dep.getVersion() + "\"/>\n");
               importsString.append("    </import>\n");
            }
         }
      }
      evaluateVariable(compInfoTemplate, "imports", importsString.toString());

      // Write the component info file
      File compInfoFile = new File(deployToDir + fileSep + "component-info.xml");
      try
      {
         if (!deployToDir.exists())
         {
            deployToDir.mkdirs();
         }
         FileWriter fw = new FileWriter(compInfoFile);
         this.getLog().info("Deploying file: " + compInfoFile.getCanonicalPath());
         fw.write(compInfoTemplate.toString());
         fw.close();
      }
      catch (IOException ioe)
      {
         this.getLog().error(ioe);
      }

      CompInfoVerifier verifier = new CompInfoVerifier();
      try
      {
         verifier.verifyCompInfo(new FileInputStream(compInfoFile));
      }
      catch (SAXException e)
      {
         this.getLog().warn(e);
      }
      catch (FileNotFoundException fnfe)
      {
         fnfe.printStackTrace();
      }
   }

   /**
    * Replace all instances of var with value in the given buffer.
    * For example if var is "project.name", this method will search
    * the buffer for "${project.name}" and replace it with the value
    * provided.
    * 
    * @param buffer The buffer to be modified
    * @param var The name of the variable to be evaluated.
    * @param value The replacement string.
    */
   public void evaluateVariable(StringBuffer buffer, String var, String value)
   {
      int start = 0;
      while ((start = buffer.indexOf("${" + var + "}")) != -1)
      {
         int end = start + ("${" + var + "}").length();
         buffer.replace(start, end, value);
      }
   }

}
