package jp.gopay.sdk.builders.charge;

import jp.gopay.sdk.builders.ResourceMonitor;
import jp.gopay.sdk.builders.ResourcePredicate;
import jp.gopay.sdk.builders.charge.AbstractChargesBuilders.*;
import jp.gopay.sdk.models.common.*;
import jp.gopay.sdk.models.common.Void;
import jp.gopay.sdk.models.request.charge.CaptureReq;
import jp.gopay.sdk.models.request.charge.ChargesReq;
import jp.gopay.sdk.models.request.charge.PatchReq;
import jp.gopay.sdk.models.response.PaginatedList;
import jp.gopay.sdk.models.response.charge.Charge;
import jp.gopay.sdk.resources.ChargesResource;
import jp.gopay.sdk.types.ChargeStatus;
import retrofit2.Call;
import retrofit2.Retrofit;

public abstract class ChargesBuilders {

    public static class ListChargesRequestBuilder
            extends AbstractListChargesRequestBuilder<ListChargesRequestBuilder, ChargesResource, Charge>{

        public ListChargesRequestBuilder(Retrofit retrofit) {
            super(retrofit);
        }

        public ListChargesRequestBuilder(Retrofit retrofit, StoreId storeId) {
            super(retrofit, storeId);
        }

        @Override
        protected Call<PaginatedList<Charge>> getRequest(ChargesResource resource) {
            if (storeId != null) {
                return resource.listAllStoreCharges(storeId, getLimit(), getCursorDirection(), getCursor(), propertySearch.asMap(), metadataSearch);
            } else {
                return resource.listAllCharges(getLimit(), getCursorDirection(), getCursor(), propertySearch.asMap(), metadataSearch);
            }
        }
    }

    public static class GetChargeRequestBuilder
            extends AbstractGetChargeRequestBuilder<GetChargeRequestBuilder, ChargesResource, Charge>{

        public GetChargeRequestBuilder(Retrofit retrofit, StoreId storeId, ChargeId chargeId) {
            super(retrofit, storeId, chargeId);
        }

        @Override
        protected Call<Charge> getRequest(ChargesResource resource) {
            return resource.getStoreCharge(storeId, chargeId, polling);
        }
    }

    public static class CreateChargeRequestBuilder
            extends AbstractCreateChargeRequestBuilder<CreateChargeRequestBuilder, ChargesResource, Charge>{

        public CreateChargeRequestBuilder(Retrofit retrofit, TransactionTokenId transactionTokenId, MoneyLike money, Boolean capture) {
            super(retrofit, transactionTokenId, money, capture);
        }

        @Override
        protected Call<Charge> getRequest(ChargesResource resource) {
            return resource.createCharge(new ChargesReq(transactionTokenId, money, capture, captureAt, metadata), idempotencyKey);
        }
    }

    public static class UpdateChargeRequestBuilder
            extends AbstractUpdateChargeRequestBuilder<UpdateChargeRequestBuilder, ChargesResource, Charge>{

        public UpdateChargeRequestBuilder(Retrofit retrofit, StoreId storeId, ChargeId chargeId) {
            super(retrofit, storeId, chargeId);
        }

        @Override
        protected Call<Charge> getRequest(ChargesResource resource) {
            return resource.updateCharge(storeId, chargeId, new PatchReq(metadata), idempotencyKey);
        }
    }

    public static class CaptureAuthorizedChargeRequestBuilder
            extends AbstractCaptureAuthorizedChargeRequestBuilder<CaptureAuthorizedChargeRequestBuilder, ChargesResource>{

        public CaptureAuthorizedChargeRequestBuilder(Retrofit retrofit, StoreId storeId, ChargeId chargeId, MoneyLike money) {
            super(retrofit, storeId, chargeId, money);
        }

        @Override
        protected Call<Void> getRequest(ChargesResource resource) {
            return resource.capture(storeId, chargeId, new CaptureReq(money), idempotencyKey);
        }
    }

    public static ResourceMonitor<Charge> createChargeCompletionMonitor(Retrofit retrofit, StoreId storeId, ChargeId chargeId) {
        return new ResourceMonitor<>(new GetChargeRequestBuilder(retrofit, storeId, chargeId)
                .withPolling(true),
                new ResourcePredicate<Charge>() {
                    @Override
                    public boolean test(Charge resource) {
                        return resource.getStatus() != ChargeStatus.PENDING;
                    }
                });
    }

}
