package jp.gopay.sdk.builders.store;

import jp.gopay.sdk.builders.IdempotentRetrofitRequestBuilder;
import jp.gopay.sdk.builders.RetrofitRequestBuilder;
import jp.gopay.sdk.builders.RetrofitRequestBuilderPaginated;
import jp.gopay.sdk.models.common.*;
import jp.gopay.sdk.models.common.Void;
import jp.gopay.sdk.models.common.stores.SecurityConfiguration;
import jp.gopay.sdk.models.response.store.*;
import jp.gopay.sdk.models.response.subscription.SubscriptionConfiguration;
import jp.gopay.sdk.types.CardBrand;
import jp.gopay.sdk.types.Country;
import org.threeten.bp.ZoneId;
import retrofit2.Retrofit;

import java.math.BigDecimal;
import java.net.URL;
import java.util.Locale;
import java.util.Map;

public abstract class AbstractStoreBuilders {

    public static abstract class AbstractListStoresRequestBuilder<B extends AbstractListStoresRequestBuilder, R, M extends Store>
            extends RetrofitRequestBuilderPaginated<M, R, B, StoreId> {

        protected String search;

        protected String getSearch() {
            return search;
        }

        public AbstractListStoresRequestBuilder(Retrofit retrofit) {
            super(retrofit);
        }

        public B withSearch(String searchString) {
            this.search = searchString;
            return (B)this;
        }

    }

    public static abstract class AbstractDeleteStoreRequestBuilder<B extends AbstractDeleteStoreRequestBuilder, R>
            extends RetrofitRequestBuilder<Void, R> {

        protected StoreId storeId;

        protected StoreId getStoreId() {
            return storeId;
        }

        public AbstractDeleteStoreRequestBuilder(Retrofit retrofit, StoreId storeId) {
            super(retrofit);
            this.storeId = storeId;
        }

    }

    public static abstract class AbstractGetStoreRequestBuilder<B extends AbstractGetStoreRequestBuilder, R, M extends StoreWithConfiguration>
            extends RetrofitRequestBuilder<M, R> {

        protected StoreId storeId;

        protected StoreId getStoreId() {
            return storeId;
        }

        public AbstractGetStoreRequestBuilder(Retrofit retrofit, StoreId storeId) {
            super(retrofit);
            this.storeId = storeId;
        }

    }

    public static abstract class AbstractUpdateStoreRequestBuilder<B extends AbstractUpdateStoreRequestBuilder, R, M extends StoreWithConfiguration>
            extends IdempotentRetrofitRequestBuilder<M, R, B> {

        protected StoreId storeId;
        protected String name;
        protected URL logoUrl;
        protected Locale language;
        protected Country country;
        protected ZoneId timeZone;
        protected UserTransactionsConfiguration userTransactionsConfiguration;
        protected CardConfiguration cardConfiguration;
        protected QrScanConfiguration qrScanConfiguration;
        protected KonbiniConfiguration convenienceConfiguration;
        protected RecurringTokenConfiguration recurringTokenConfiguration;
        protected SecurityConfiguration securityConfiguration;
        protected Map<CardBrand, BigDecimal> cardBrandPercentFees;
        protected SubscriptionConfiguration subscriptionConfiguration;

        protected StoreId getStoreId() {
            return storeId;
        }

        protected String getName() {
            return name;
        }

        protected URL getLogoUrl() {
            return logoUrl;
        }

        protected Locale getLanguage() {
            return language;
        }

        public ZoneId getTimeZone() {
            return timeZone;
        }

        public UserTransactionsConfiguration getUserTransactionsConfiguration() {
            return userTransactionsConfiguration;
        }

        protected CardConfiguration getCardConfiguration() {
            return cardConfiguration;
        }

        protected QrScanConfiguration getQrScanConfiguration() {
            return qrScanConfiguration;
        }

        protected KonbiniConfiguration getConvenienceConfiguration() {
            return convenienceConfiguration;
        }

        protected RecurringTokenConfiguration getRecurringTokenConfiguration() {
            return recurringTokenConfiguration;
        }

        protected SecurityConfiguration getSecurityConfiguration() {
            return securityConfiguration;
        }

        protected Map<CardBrand, BigDecimal> getCardBrandPercentFees() {
            return cardBrandPercentFees;
        }

        /**
         * The returned type will be changed to {@link Country} on later release
         * @return country
         */
        protected String getCountry() {
            if(country == null){
                return null;
            }
            return country.getAlpha2();
        }

        /**
         * This method will be deleted when the returned type by "getCountry(String)" is changed to {@link Country}
         * @return country enum
         */
        protected Country getCountryEnum() {
            return country;
        }

        public AbstractUpdateStoreRequestBuilder(Retrofit retrofit, StoreId storeId) {
            super(retrofit);
            this.storeId = storeId;
        }

        public B withName(String name) {
            this.name = name;
            return (B)this;
        }

        public B withLogoUrl(URL logoUrl) {
            this.logoUrl = logoUrl;
            return (B)this;
        }

        public B withLanguage(Locale language){
            this.language = language;
            return (B)this;
        }

        @Deprecated
        /**
         * @deprecated
         * This method will be deleted on later release
         */
        public B withCountry(String country) {
            this.country = Country.getCountryByAlpha2(country);
            return (B)this;
        }

        public B withCountry(Country country) {
            this.country = country;
            return (B)this;
        }

        public B withTimeZone(ZoneId timeZone) {
            this.timeZone = timeZone;
            return (B) this;
        }
        public B withUserTransactionsConfiguration(UserTransactionsConfiguration userTransactionsConfiguration) {
            this.userTransactionsConfiguration = userTransactionsConfiguration;
            return (B) this;
        }

        public B withCardConfiguration(CardConfiguration cardConfiguration) {
            this.cardConfiguration = cardConfiguration;
            return (B)this;
        }

        public B withQrScanConfiguration(QrScanConfiguration qrScanConfiguration) {
            this.qrScanConfiguration = qrScanConfiguration;
            return (B)this;
        }

        public B withConvenienceConfiguration(KonbiniConfiguration convenienceConfiguration){
            this.convenienceConfiguration = convenienceConfiguration;
            return (B)this;
        }

        public B withRecurringTokenConfiguration(RecurringTokenConfiguration recurringTokenConfiguration) {
            this.recurringTokenConfiguration = recurringTokenConfiguration;
            return (B)this;
        }

        public B withSecurityConfiguration(SecurityConfiguration securityConfiguration) {
            this.securityConfiguration = securityConfiguration;
            return (B)this;
        }

        public B withCardBrandPercentFees(Map<CardBrand, BigDecimal> cardBrandPercentFees){
            this.cardBrandPercentFees = cardBrandPercentFees;
            return (B)this;
        }

        public B withSubscriptionConfiguration(SubscriptionConfiguration subscriptionConfiguration){
            this.subscriptionConfiguration = subscriptionConfiguration;
            return (B)this;
        }

    }

    public static abstract class AbstractCreateStoreRequestBuilder<B extends AbstractCreateStoreRequestBuilder, R, M extends StoreWithConfiguration>
            extends IdempotentRetrofitRequestBuilder<M, R, B> {

        protected String name;
        protected URL logoUrl;
        protected Locale language;
        protected Country country;
        protected ZoneId timeZone;
        protected UserTransactionsConfiguration userTransactionsConfiguration;
        protected CardConfiguration cardConfiguration;
        protected QrScanConfiguration qrScanConfiguration;
        protected KonbiniConfiguration convenienceConfiguration;
        protected RecurringTokenConfiguration recurringTokenConfiguration;
        protected SecurityConfiguration securityConfiguration;
        protected Map<CardBrand, BigDecimal> cardBrandPercentFees;
        protected SubscriptionConfiguration subscriptionConfiguration;

        protected String getName() {
            return name;
        }

        protected URL getLogoUrl() {
            return logoUrl;
        }

        protected Locale getLanguage() {
            return language;
        }

        public ZoneId getTimeZone() {
            return timeZone;
        }

        public UserTransactionsConfiguration getUserTransactionsConfiguration() {
            return userTransactionsConfiguration;
        }

        protected CardConfiguration getCardConfiguration() {
            return cardConfiguration;
        }

        protected QrScanConfiguration getQrScanConfiguration() {
            return qrScanConfiguration;
        }

        public KonbiniConfiguration getConvenienceConfiguration() {
            return convenienceConfiguration;
        }

        protected RecurringTokenConfiguration getRecurringTokenConfiguration() {
            return recurringTokenConfiguration;
        }

        protected SecurityConfiguration getSecurityConfiguration() {
            return securityConfiguration;
        }

        protected Map<CardBrand, BigDecimal> getCardBrandPercentFees() {
            return cardBrandPercentFees;
        }

        /**
         * The returned type will be changed to {@link Country} on later release
         * @return country
         */
        protected String getCountry() {
            if(country == null){
                return null;
            }
            return country.getAlpha2();
        }

        /**
         * This method will be deleted when the returned type by "getCountry(String)" is changed to {@link Country}
         * @return country enum
         */
        protected Country getCountryEnum() {
            return country;
        }

        public AbstractCreateStoreRequestBuilder(Retrofit retrofit, String name) {
            super(retrofit);
            this.name = name;
        }

        @Deprecated
        /**
         * @deprecated
         * This method will be deleted on later release
         */
        public B withCountry(String country){
            this.country = Country.getCountryByAlpha2(country);
            return (B)this;
        }

        public B withCountry(Country country) {
            this.country = country;
            return (B)this;
        }

        public B withLogoUrl(URL logoUrl) {
            this.logoUrl = logoUrl;
            return (B)this;
        }

        public B withLanguage(Locale language){
            this.language = language;
            return (B)this;
        }

        public B withTimeZone(ZoneId timeZone) {
            this.timeZone = timeZone;
            return (B) this;
        }

        public B withUserTransactionsConfiguration(UserTransactionsConfiguration userTransactionsConfiguration) {
            this.userTransactionsConfiguration = userTransactionsConfiguration;
            return (B) this;
        }
        public B withCardConfiguration(CardConfiguration cardConfiguration) {
            this.cardConfiguration = cardConfiguration;
            return (B)this;
        }

        public B withQrScanConfiguration(QrScanConfiguration qrScanConfiguration) {
            this.qrScanConfiguration = qrScanConfiguration;
            return (B)this;
        }

        public B withConvenienceConfiguration(KonbiniConfiguration convenienceConfiguration){
            this.convenienceConfiguration = convenienceConfiguration;
            return (B)this;
        }

        public B withRecurringTokenConfiguration(RecurringTokenConfiguration recurringTokenConfiguration) {
            this.recurringTokenConfiguration = recurringTokenConfiguration;
            return (B)this;
        }

        public B withSecurityConfiguration(SecurityConfiguration securityConfiguration) {
            this.securityConfiguration = securityConfiguration;
            return (B)this;
        }

        public B withCardBrandPercentFees(Map<CardBrand, BigDecimal> cardBrandPercentFees){
            this.cardBrandPercentFees = cardBrandPercentFees;
            return (B)this;
        }

        public B withSubscriptionConfiguration(SubscriptionConfiguration subscriptionConfiguration){
            this.subscriptionConfiguration = subscriptionConfiguration;
            return (B)this;
        }

    }

    public static abstract class AbstractGetCheckoutInfoRequestBuilder<B extends AbstractGetCheckoutInfoRequestBuilder, R, M extends CheckoutInfo>
                extends RetrofitRequestBuilder<M, R> {
        protected Domain origin;

        protected Domain getOrigin() {
            return origin;
        }

        public AbstractGetCheckoutInfoRequestBuilder(Retrofit retrofit, Domain origin) {
            super(retrofit);
            this.origin = origin;
        }

        public AbstractGetCheckoutInfoRequestBuilder(Retrofit retrofit) {
            super(retrofit);
        }

    }

    public static abstract class AbstractCreateCustomerIdRequestBuilder<B extends AbstractCreateCustomerIdRequestBuilder, R, M extends GopayCustomerId>
            extends RetrofitRequestBuilder<M, R>{

        protected String customerId;
        protected StoreId storeId;

        public String getCustomerId() {
            return customerId;
        }

        public StoreId getStoreId() {
            return storeId;
        }

        public AbstractCreateCustomerIdRequestBuilder(Retrofit retrofit, StoreId storeId, String customerId) {
            super(retrofit);
            this.storeId = storeId;
            this.customerId = customerId;
        }
    }

}
