package jp.gopay.sdk.resources;

import jp.gopay.sdk.models.common.*;
import jp.gopay.sdk.models.common.Void;
import jp.gopay.sdk.models.request.subscription.ScheduledPaymentPatchData;
import jp.gopay.sdk.models.request.subscription.SubscriptionCreateData;
import jp.gopay.sdk.models.request.subscription.SubscriptionPlanSimulationRequest;
import jp.gopay.sdk.models.request.subscription.SubscriptionUpdateData;
import jp.gopay.sdk.models.response.PaginatedList;
import jp.gopay.sdk.models.response.PaymentsPlan;
import jp.gopay.sdk.models.response.charge.Charge;
import jp.gopay.sdk.models.response.subscription.FullSubscription;
import jp.gopay.sdk.models.response.subscription.ScheduledPayment;
import jp.gopay.sdk.models.response.subscription.Subscription;
import jp.gopay.sdk.types.CursorDirection;
import retrofit2.Call;
import retrofit2.http.*;

import javax.annotation.Nullable;

import static jp.gopay.sdk.constants.GopayConstants.idempotencyKeyHeaderName;

/**
 * Resource for managing subscriptions.
 */
public interface SubscriptionsResource {
    @GET("/subscriptions")
    Call<PaginatedList<Subscription>> listAllSubscriptions(
            @Query("limit") @Nullable Integer limit,
            @Query("cursor_direction") @Nullable CursorDirection cursorDirection,
            @Query("cursor") @Nullable SubscriptionId cursor
    );

    @GET("/stores/{storeId}/subscriptions")
    Call<PaginatedList<Subscription>> listAllSubscriptions(
            @Path("storeId") StoreId storeId,
            @Query("limit") @Nullable Integer limit,
            @Query("cursor_direction") @Nullable CursorDirection cursorDirection,
            @Query("cursor") @Nullable SubscriptionId cursor
    );

    @GET("/stores/{storeId}/subscriptions/{subscriptionId}")
    Call<FullSubscription> getSubscription(
            @Path("storeId") StoreId storeId,
            @Path("subscriptionId") SubscriptionId subscriptionId,
            @Query("polling") @Nullable Boolean polling
    );

    @POST("/subscriptions")
    Call<FullSubscription> createSubscription(
            @Body SubscriptionCreateData dataToPost,
            @Header(idempotencyKeyHeaderName) IdempotencyKey idempotencyKey
    );

    @PATCH("/stores/{storeId}/subscriptions/{subscriptionId}")
    Call<FullSubscription> updateSubscription(
            @Path("storeId") StoreId storeId,
            @Path("subscriptionId") SubscriptionId subscriptionId,
            @Body SubscriptionUpdateData dataToPost,
            @Header(idempotencyKeyHeaderName) IdempotencyKey idempotencyKey
    );

    @GET("/stores/{storeId}/subscriptions/{subscriptionId}/payments")
    Call<PaginatedList<ScheduledPayment>> listPayments(
            @Path("storeId") StoreId storeId,
            @Path("subscriptionId") SubscriptionId subscriptionId,
            @Query("limit") @Nullable Integer limit,
            @Query("cursor_direction") @Nullable CursorDirection cursorDirection,
            @Query("cursor") @Nullable ScheduledPaymentId cursor
    );

    @GET("/stores/{storeId}/subscriptions/{subscriptionId}/payments/{paymentId}")
    Call<ScheduledPayment> getPayment(
            @Path("storeId") StoreId storeId,
            @Path("subscriptionId") SubscriptionId subscriptionId,
            @Path("paymentId") ScheduledPaymentId paymentId
    );

    @GET("/stores/{storeId}/subscriptions/{subscriptionId}/payments/{paymentId}/charges")
    Call<PaginatedList<Charge>> listChargesForPayment(
            @Path("storeId") StoreId storeId,
            @Path("subscriptionId") SubscriptionId subscriptionId,
            @Path("paymentId") ScheduledPaymentId paymentId,
            @Query("limit") @Nullable Integer limit,
            @Query("cursor_direction") @Nullable CursorDirection cursorDirection,
            @Query("cursor") @Nullable ChargeId cursor
    );

    @PATCH("/stores/{storeId}/subscriptions/{subscriptionId}/payments/{paymentId}")
    Call<ScheduledPayment> updatePayment(
            @Path("storeId") StoreId storeId,
            @Path("subscriptionId") SubscriptionId subscriptionId,
            @Path("paymentId") ScheduledPaymentId paymentId,
            @Body ScheduledPaymentPatchData patchData,
            @Header(idempotencyKeyHeaderName) IdempotencyKey idempotencyKey
    );

    @DELETE("/stores/{storeId}/subscriptions/{subscriptionId}")
    Call<Void> deleteSubscription(
            @Path("storeId") StoreId storeId,
            @Path("subscriptionId") SubscriptionId subscriptionId
    );

    @GET("/stores/{storeId}/subscriptions/{subscriptionId}/charges")
    Call<PaginatedList<Charge>> listSubscriptionCharges(
            @Path("storeId") StoreId storeId,
            @Path("subscriptionId") SubscriptionId subscriptionId,
            @Query("limit") @Nullable Integer limit,
            @Query("cursor_direction") @Nullable CursorDirection cursorDirection,
            @Query("cursor") @Nullable ChargeId cursor
    );

    @POST("/subscriptions/simulate_plan")
    Call<PaymentsPlan> simulateSubscriptionPlan(
            @Body SubscriptionPlanSimulationRequest simulationRequest,
            @Header(idempotencyKeyHeaderName) IdempotencyKey idempotencyKey
    );

    @POST("/stores/{storeId}/subscriptions/simulate_plan")
    Call<PaymentsPlan> simulateSubscriptionPlan(
            @Path("storeId") StoreId storeId,
            @Body SubscriptionPlanSimulationRequest simulationRequest,
            @Header(idempotencyKeyHeaderName) IdempotencyKey idempotencyKey
    );
}
