package jp.gopay.sdk.settings;


import jp.gopay.sdk.models.common.auth.AuthStrategy;
import okhttp3.OkHttpClient;

/**
 * SDK settings, including the following:
 *
 * endpoint:       Endpoint to where requests are directed. Defaults to the value of the environment variable <code>GOPAY_ENDPOINT</code>
 *                 or to <code>"https://api.gopay.jp"</code> if not set.
 * timeoutSeconds: Set the number of seconds allowed to pass before raising a connection timeout error or read timeout error.
 *                 Defaults to 900 seconds (15 mins).
 * origin          Value of the Origin header
 */
public abstract class AbstractSDKSettings<T extends AbstractSDKSettings> {
    private static final String DEFAULT_ENDPOINT = "https://api.gopay.jp";
    private static final String GOPAY_ENDPOINT_ENVVAR = "GOPAY_ENDPOINT";
    private static final Long DEFAULT_TIMEOUT = 900L;
    private String endpointFromEnvvar = System.getenv(GOPAY_ENDPOINT_ENVVAR);

    protected String endpoint = (endpointFromEnvvar != null) ? endpointFromEnvvar : DEFAULT_ENDPOINT;
    protected long timeout = DEFAULT_TIMEOUT;
    protected String origin = "";

    /**
     * The API endpoint
     * @param endpoint
     * @return an instance of T
     */
    public T withEndpoint(String endpoint){
        this.endpoint = endpoint;
        return (T)this;
    }

    /**
     * The timeout for connecting, reading and writing
     * @param timeoutSeconds
     * @return an instance of T
     */
    public T withTimeoutSeconds(long timeoutSeconds){
        this.timeout = timeoutSeconds;
        return (T)this;
    }

    /**
     * Attach a value to the Origin header, which is added to every request sent by the SDK.
     * @param origin
     * @return an instance of T
     */
    public T attachOrigin(String origin) {
        this.origin = origin;
        return (T)this;
    }

    public String getEndpoint() {
        return endpoint;
    }

    public long getTimeout() {
        return timeout;
    }

    public String getOrigin() {
        return origin;
    }

    /**
     * Creates a copy of this instance.
     * @return an instance of T
     */
    public abstract T copy();

    /**
     *
     * @param authStrategy the authentication strategy
     * @return an OkHttpClient to be used for performing requests
     * @see OkHttpClient
     */
    public abstract OkHttpClient getClient(AuthStrategy authStrategy);
}
