@file:Suppress("DEPRECATION")

package com.paymentpage.msdk.core.googlePay

import org.json.JSONArray
import org.json.JSONObject
import java.math.BigDecimal

class GooglePayHelper(
    private val merchantId: String,
    private val merchantName: String,
    private val gatewayMerchantId: String = "ecommpay",
    private val countryCode: String? = null
) {

    fun createReadyToPayRequest(): JSONObject {
        return getBaseRequest()
            .put(
                "allowedPaymentMethods",
                JSONArray().put(getCardPaymentMethods())
            )
    }

    fun createPaymentDataRequest(price: BigDecimal, currencyCode: String): JSONObject {
        return getBaseRequest()
            .put("allowedPaymentMethods", JSONArray().put(getCardPaymentMethod()))
            .put("transactionInfo", getTransactionInfo(price, currencyCode))
            .put("merchantInfo", merchantInfo)
    }

    private fun getTransactionInfo(price: BigDecimal, currencyCode: String): JSONObject {
        val formattedPrice = price.setScale(2, BigDecimal.ROUND_HALF_EVEN).toString()
        val jsonObj = JSONObject()
            .put("totalPrice", formattedPrice)
            .put("totalPriceStatus", PRICE_STATUS)
            .put("currencyCode", currencyCode)
        if (!countryCode.isNullOrEmpty())
            jsonObj.put("countryCode", countryCode)
        return jsonObj
    }

    private fun getCardPaymentMethod(): JSONObject {
        return getCardPaymentMethods()
            .put("tokenizationSpecification", getTokenSpecification())
    }

    private fun getCardPaymentMethods(): JSONObject {
        return JSONObject()
            .put("type", "CARD")
            .put(
                "parameters", JSONObject()
                    .put("allowedAuthMethods", getAllowedCardAuthMethods())
                    .put("allowedCardNetworks", getAllowedCardNetworks())
            )
    }

    private fun getAllowedCardAuthMethods(): JSONArray {
        return JSONArray()
            .put("PAN_ONLY")
            .put("CRYPTOGRAM_3DS")
    }

    private fun getAllowedCardNetworks(): JSONArray {
        return JSONArray()
            .put("VISA")
            .put("MASTERCARD")
            .put("DISCOVER")
            .put("AMEX")
            .put("INTERAC")
            .put("JCB")
    }

    private fun getTokenSpecification(): JSONObject {
        return JSONObject()
            .put("type", GATEWAY_TYPE)
            .put(
                "parameters", JSONObject()
                    .put("gateway", GATEWAY_NAME)
                    .put("gatewayMerchantId", gatewayMerchantId)
            )

    }

    private val merchantInfo: JSONObject =
        JSONObject().put("merchantName", merchantName).put("merchantId", merchantId)

    private fun getBaseRequest(): JSONObject {
        return JSONObject()
            .put("apiVersion", API_VERSION)
            .put("apiVersionMinor", API_VERSION_MINOR)
    }

    companion object {
        private const val API_VERSION = 2
        private const val API_VERSION_MINOR = 0

        private const val GATEWAY_NAME = "ecommpay"
        private const val GATEWAY_TYPE = "PAYMENT_GATEWAY"
        private const val PRICE_STATUS = "FINAL"
    }

}